package csdk.v2.runner.filesystem;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;

/**
 * Utilitrios para manipulao de sistemas de arquivos.
 *
 * @author Tecgraf
 */
public class FileUtils {

  /**
   * Verifica se um path  absoluto para qualquer SO, ou seja, no  necessrio
   * que o path seja absoluto para o mesmo SO de quem fez esta verificao.
   *
   * @param path o caminho.
   * @return true se o path comea com '/' ou se o path comea com letra + ':\'
   *         (p.ex. "C:\")
   */
  public static boolean isAbsolutePath(String path) {
    return path.startsWith("/") || path.matches("^\\w:\\\\.*");
  }

  /**
   * Faz o agrupamento de um caminho quebrado em um array de {@code String}
   * em uma nica {@code String} contendo separadores de diretrio. O path
   * criado pode opcionalmente ser absoluto.
   *
   * @param splittedPath array de {@code String} representando um caminho
   * @return {@code String} contendo separadores de diretrio.
   */
  public static String joinPath(String... splittedPath) {
    char separator = File.separatorChar;
    StringBuilder buffer = new StringBuilder();
    if (splittedPath == null) {
      throw new IllegalArgumentException(
        "Internal error: null path not allowed");
    }
    for (int i = 0; i < splittedPath.length - 1; i++) {
      buffer.append(splittedPath[i]);
      buffer.append(separator);
    }
    if (splittedPath.length > 0) {
      buffer.append(splittedPath[splittedPath.length - 1]);
    }
    return buffer.toString();
  }

  /**
   * Quebra um caminho em forma de {@code String} com separadores de
   * diretrio do sistema operacional corrente em um array de
   * {@code String} contendo seus componentes.
   *
   * @param filePath caminho a ser quebrado.
   * @return caminho quebrado em um array de seus componentes.
   */
  public static String[] splitPath(String filePath) {
    // Teste para evitar problemas com separador windows na regex
    String separator = File.separatorChar == '\\' ? "\\\\" : File.separator;
    return filePath.split(separator);
  }

  /**
   * Fecha um {@link Closeable} (til para fechar {@link Reader readers} e
   * {@link Writer writers} em blocos {@code finally}).
   *
   * @param closeable objeto a ser fechado. Pode ser {@code null}.
   * @return {@code true} se a operao foi bem-sucedida,
   *         {@code false} em caso de erro ou se
   *         {@code closeable == null}
   */
  public static boolean close(Closeable closeable) {
    if (closeable == null) {
      return false;
    }
    try {
      closeable.close();
      return true;
    }
    catch (IOException e) {
      return false;
    }
  }

}
