package csdk.v2.runner.rest;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.ServerSocket;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import javax.ws.rs.core.Configuration;

import org.glassfish.grizzly.http.server.HttpServer;
import org.glassfish.jersey.grizzly2.httpserver.GrizzlyHttpServerFactory;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.server.model.ModelProcessor;
import org.glassfish.jersey.server.model.Resource;
import org.glassfish.jersey.server.model.ResourceMethod;
import org.glassfish.jersey.server.model.ResourceModel;
import org.glassfish.jersey.server.spi.Container;
import org.glassfish.jersey.server.spi.ContainerLifecycleListener;

import csdk.v2.runner.CSDKLogger;
import csdk.v2.runner.rest.ReloadFilter.RequestCounter;

/**
 * Controlador REST para o cliente.
 */
public class RestController {
  /**
   * Singleton do controlador.
   */
  private static RestController instance;

  /**
   * Configurao de resources oferecidos.
   */
  private ResourceConfig resourceConfig;

  /**
   * Servidor HTTP.
   */
  private HttpServer httpServer;

  /**
   * Container Jersey para atualizao dinmica de recursos.
   */
  private Container container;

  /**
   * Pacotes de recursos da API rest.
   */
  private final List<String> resourcePackages;

  /**
   * Conjunto de recursos por prefixo da aplicao (padro appId/instanceId).
   */
  private final Map<String, Set<Object>> instancesByPrefix;

  /**
   * Mapa de prefixos da aplicao por classe de recurso.
   */
  private final Map<Class<?>, String> prefixByResourceClass;

  /**
   * Flag que inidica que o container HTTP deve ser reiniciado
   */
  private boolean shouldReload;

  /**
   * Porta do servidor HTTP
   */
  private int httpPort;

  /**
   * Mtodo para recuperar a instncia do singleton do controlador REST.
   *
   * @return A instncia nica do controlador REST
   */
  public static RestController getInstance() {
    if (instance == null) {
      instance = new RestController();
    }
    return instance;
  }

  /**
   * Host padro para acesso.
   */
  private static final String DEFAULT_HOST = "127.0.0.1";

  /**
   * Porta padro do servidor REST
   */
  private static final int DEFAULT_PORT = 15000;

  /**
   * Determina uma porta TCP livre no host local
   *
   * @return Uma porta TCP que no est sendo usada
   */
  private static int getAvailablePort() {
    ServerSocket socket = null;
    int port = -1;
    try {
      socket = new ServerSocket(0);
      port = socket.getLocalPort();
    }
    catch (IOException ex) {
    } finally {
      if (socket != null) {
        try {
          socket.close();
        }
        catch (IOException e) {
          e.printStackTrace();
        }
      }
    }

    return port;
  }

  /**
   * Construtor privado do controlador REST, somente acessvel via getInstance
   */
  private RestController() {

    //pacotes de recursos REST do cliente
    this.resourcePackages = new ArrayList<>();
    resourcePackages.add("csbase.client.rest.resources");

    //objetos de recursos de aplicaes
    instancesByPrefix = new HashMap<>();

    prefixByResourceClass = new HashMap<>();

  }

  /**
   * Encerra o servidor REST.
   */
  public void stopServer() {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();
    }
  }

  /**
   * Inicia o servidor REST na porta e host padro.
   */
  public void startServer() {
    try {
      startServer(DEFAULT_HOST, DEFAULT_PORT);
    }
    catch (javax.ws.rs.ProcessingException ex) {
      startServer(DEFAULT_HOST, getAvailablePort());
    }
  }

  /**
   * Inicia o servidor REST no host padro em uma porta especfica.
   *
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(int port) {
    startServer(DEFAULT_HOST, port);
  }

  /**
   * Inicia o servidor REST em uma porta e host especficos.
   *
   * @param host Host a ser usado pelo servidor.
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(String host, int port) {
    this.httpPort = port;

    try {
      this.resourceConfig = buildResourceConfig();

      this.resourceConfig.registerInstances(new ContainerLifecycleListener() {
        @Override
        public void onStartup(Container cont) {
          RestController.this.container = cont;
        }

        @Override
        public void onReload(Container cont) {
        }

        @Override
        public void onShutdown(Container cont) {
        }
      });

      final String baseURL = "http://" + host + ":" + port + "/";

      this.httpServer = GrizzlyHttpServerFactory
        .createHttpServer(URI.create(baseURL), this.resourceConfig);

      CSDKLogger logger = CSDKLogger.getInstance();
      logger.log("REST Server started at {0}", baseURL);
    }
    catch (Exception ex) {
      CSDKLogger logger = CSDKLogger.getInstance();
      logger.logException(ex);
    }

  }

  /**
   * Metodo para registro de recursos no servidor REST.
   *
   * @param classes Lista de classes de recursos a serem registradas.
   */
  public void registerService(Class<?>[] classes) {
    resourceConfig.registerClasses(classes);
  }

  /**
   * Adiciona uma lista de pacotes a lista de recursos a serem registrados no
   * servidor REST.
   *
   * @param packages Lista de pacotes com resursos REST a serem adicionados
   */
  public void addResourcePackages(String... packages) {
    resourcePackages.addAll(Arrays.asList(packages));

    if (container != null) {
      this.resourceConfig = buildResourceConfig();
      container.reload(this.resourceConfig);
    }
  }

  /**
   * Adiciona uma lista de instncias de recursos a serem registrados no
   * servidor REST.
   *
   * @param prefix O prefixo da aplicao
   * @param config Cnfigurao de recursos REST a serem adicionados
   */
  public void addApplicationResources(String prefix, ResourceConfig config) {

    Set<Object> instances = config.getInstances();

    CSDKLogger logger = CSDKLogger.getInstance();
    for (Object i : instances) {
      prefixByResourceClass.put(i.getClass(), prefix);
      logger.log("Registering REST resource {0}", i.getClass());
    }

    instancesByPrefix.put(prefix, instances);

    shouldReload = true;
    reloadContainer();
  }

  /**
   * Recarrega o container (ou registra um timer para fazer a recarga caso no
   * seja possvel recarregar no momento).
   */
  private void reloadContainer() {
    RequestCounter counter = ReloadFilter.getCounter();

    synchronized (counter.getLock()) {
      if (shouldReload) {
        // S recarrega efetivamente se no existirem requisies sendo
        // atendidas no momento.
        if (counter.isZero()) {
          if (container != null) {
            this.resourceConfig = buildResourceConfig();
            container.reload(this.resourceConfig);
          }
          shouldReload = false;
        }
        else {
          // Caso no seja possvel recarregar, registrar um timer para
          // tentar novamente.
          TimerTask timerTask = new TimerTask() {
            @Override
            public void run() {
              reloadContainer();
            }
          };
          Timer timer = new Timer("RestController reload timer");
          timer.schedule(timerTask, 300);
        }
      }
    }
  }

  /**
   * Remove uma lista de recursos registrados no servidor REST.
   *
   * @param prefix O prefixo da aplicao
   */
  public void removeApplicationResources(String prefix) {
    instancesByPrefix.remove(prefix);

    prefixByResourceClass.entrySet()
      .removeIf(entry -> entry.getValue().equals(prefix));

    shouldReload = true;
    reloadContainer();
  }

  /**
   * Consulta a porta REST usada pelo cliente.
   *
   * @return porta.
   */
  public int getPort() {
    if (httpServer == null || !httpServer.isStarted()) {
      return -999;
    }
    return httpPort;
  }

  /**
   * Metodo para registro de pacotes com recursos no servidor REST.
   *
   * @param packages Lista de packages de recursos a serem registradas.
   */
  public void registerPackages(String... packages) {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();
    }
    resourcePackages.addAll(Arrays.asList(packages));
    this.startServer();
  }

  /**
   * Cria a configurao de recursos de acordo com os recursos atuais definidos
   *
   * @return Uma nova configurao de recursos
   */
  private ResourceConfig buildResourceConfig() {
    final ResourceConfig rc = new ResourceConfig();

    //Processador de recursos das instncias de aplicao
    rc.register(ApplicationApiModelProcessor.class);

    //Registro dos recursos REST do cliente
    rc.packages(true,
      resourcePackages.toArray(new String[resourcePackages.size()]));

    //Registra os recursos das aplicaes
    instancesByPrefix.entrySet()
      .forEach(e -> rc.registerInstances(e.getValue().toArray()));

    //Registro de recurso para serializao JSON
    rc.register(JacksonFeature.class);

    //Registro de recursos para tratamento de requisies e respostas
    rc.register(DebugMapper.class);
    rc.register(ReloadFilter.class);

    return rc;
  }

  /**
   * Processador do modelo de recursos, usado para modificar os recursos
   * providos por instncias de aplicao. Os recursos do prprio cliente no
   * so modificados.
   */
  public static class ApplicationApiModelProcessor implements ModelProcessor {

    /**
     * Cria um novo modelo de recursos, separados os recursos comuns a mais de
     * uma instncia de aplicao em recursos prprios, prefixados pelo
     * identificador da instncia.
     *
     * {@inheritDoc}
     */
    @Override
    public ResourceModel processResourceModel(ResourceModel resourceModel,
      Configuration configuration) {
      ResourceModel.Builder newModelBuilder = new ResourceModel.Builder(false);

      for (Resource res : resourceModel.getResources()) {
        boolean isAppResource = false;
        for (Map.Entry<Class<?>, String> prefixByClass : RestController
          .getInstance().prefixByResourceClass.entrySet()) {
          if (res.getHandlerClasses().contains(prefixByClass.getKey())) {
            isAppResource = true;
            Resource newRes = createResource(res, prefixByClass.getValue(),
              prefixByClass.getKey(), true);
            newModelBuilder.addResource(newRes);
          }
        }
        if (!isAppResource) {
          newModelBuilder.addResource(res);
        }
      }

      return newModelBuilder.build();
    }

    /**
     * Cria um novo recurso, adicionando somente os mtodos que pertencem 
     * instncia especificada.
     *
     * @param resource recurso original.
     * @param prefix prefixo da instncia de aplicao.
     * @param clazz a classe que implementa o recurso na instncia.
     * @param isParentResource verdadeiro se o recurso em questo  o recurso
     * pai.
     * @return o novo recurso.
     */
    private Resource createResource(Resource resource, String prefix,
      Class<?> clazz, boolean isParentResource) {
      final Resource.Builder resourceBuilder =
        Resource.builder().name(resource.getName());
      if (isParentResource) {
        resourceBuilder.path("/" + prefix + resource.getPath());
      }
      else {
        resourceBuilder.path(resource.getPath());
      }
      resource.getChildResources().forEach((r) -> resourceBuilder
        .addChildResource(createResource(r, prefix, clazz, false)));

      for (final ResourceMethod resourceMethod : resource
        .getResourceMethods()) {
        Method classMethod = getClassMethod(resourceMethod);
        if (classMethod.getDeclaringClass().equals(clazz)) {
          resourceBuilder.addMethod(resourceMethod);
        }
      }

      return resourceBuilder.build();
    }

    /**
     * Obtm o mtodo da classe que implementa o mtodo do recurso.
     *
     * @param resourceMethod o mtodo do recurso.
     * @return o mtodo da classe.
     */
    private Method getClassMethod(final ResourceMethod resourceMethod) {
      return resourceMethod.getInvocable().getDefinitionMethod();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ResourceModel processSubResource(ResourceModel resourceModel,
      Configuration configuration) {
      return resourceModel;
    }
  }

}
