package csdk.v1_0.runner.application;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;

/**
 * Carregador de classes especial para as aplicaes CSDK.
 */
public class ApplicationClassLoader extends URLClassLoader {

  /**
   * Construtor.
   *
   * @param urls as urls (diretrios ou jars) de onde sero carregadas as
   *        classes.
   * @param parent o {@link ClassLoader} pai.
   */
  public ApplicationClassLoader(URL[] urls, ClassLoader parent) {
    super(urls, parent);
  }

  /**
   * Construtor.
   *
   * @param urls as urls (diretrios ou jars) de onde sero carregadas as
   *        classes.
   */
  public ApplicationClassLoader(URL[] urls) {
    this(urls, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected synchronized Class<?> loadClass(String name, boolean resolve)
    throws ClassNotFoundException {
    //Verifica se a classe j foi carregada anteriormente.
    Class<?> c = findLoadedClass(name);
    if (c == null) {
      try {
        //Busca a classe nas URLs indicadas no construtor.
        c = findClass(name);
      }
      catch (ClassNotFoundException ignored) {
        //A classe no foi encontrada nas URLs indicadas no construtor.
      }
      if (c == null) {
        //Procura a classe no classloader-pai.
        c = getParent().loadClass(name);
        if (c == null) {
          /*
           * Se o classloader-pai tambm no conseguir fazer a carga, a classe
           * realmente no est disponvel.
           */
          throw new ClassNotFoundException(name);
        }
      }
    }
    if (resolve) {
      resolveClass(c);
    }
    return c;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URL getResource(String name) {
    URL url = findResource(name);
    if (url == null && getParent() != null) {
      url = getParent().getResource(name);
    }
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getResourceAsStream(String name) {
    URL url = getResource(name);
    try {
      if (url != null) {
        return url.openStream();
      }
      else {
        return null;
      }
    }
    catch (IOException e) {
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Enumeration<URL> getResources(String name) throws IOException {
    Enumeration<URL> res = findResources(name);
    if (res == null && getParent() != null) {
      res = getParent().getResources(name);
    }
    return res;
  }

}