package csdk.v1_0.runner.filesystem;

import java.util.ArrayList;

import csdk.v1_0.runner.Runner;

/**
 * Lista dos tipos de arquivo conhecidos pelo {@link Runner}
 * 
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class FileTypes {

  /**
   * Tipo do arquivo que indica "Arquivo Desconhecido"
   */
  public static final String UNKNOWN = "UNKNOWN";

  /**
   * Tipo do arquivo que indica diretrio.
   */
  public static final String DIRECTORY_TYPE = "DIRECTORY_TYPE";

  /**
   * Lista de associaes de extenses e tipos de arquivos.
   */
  static private ArrayList<FileType> e2tList = new ArrayList<FileType>();

  /**
   * Cria um tipo de arquivo para o {@link Runner}.
   * 
   * @param typeName o nome do novo tipo.
   * @return o tipo criado.
   */
  public static FileType createFileType(String typeName) {
    if (hasFileType(typeName)) {
      String err = "Type " + typeName + " already exists!";
      throw new IllegalArgumentException(err);
    }
    FileType typ = new FileType(typeName);
    e2tList.add(typ);
    return typ;
  }

  /**
   * Indica se um tipo j existe.
   * 
   * @param fileTypeName o tipo
   * @return indicativo
   */
  static public FileType getFileType(String fileTypeName) {
    if (fileTypeName == null) {
      return null;
    }
    String typeName = fileTypeName.trim();
    if (typeName.equals(UNKNOWN)) {
      return null;
    }

    for (FileType value : e2tList) {
      String name = value.getName();
      if (name.equalsIgnoreCase(typeName)) {
        return value;
      }
    }
    return null;
  }

  /**
   * Indica se um tipo j existe.
   * 
   * @param typeName o tipo
   * @return indicativo
   */
  static public boolean hasFileType(String typeName) {
    return getFileType(typeName) != null;
  }

  /**
   * Busca um tipo de arquivo aplicvel a extenso sugerida.
   * 
   * @param extension extenso
   * @return o tipo
   */
  static public FileType getFileTypeFromExtension(String extension) {
    if (extension == null) {
      throw new RuntimeException("Internal error: null extension not allowed");
    }
    for (FileType value : e2tList) {
      ArrayList<String> extensions = value.getExtensions();
      for (String ext : extensions) {
        if (extension.trim().equalsIgnoreCase((ext.trim()))) {
          return value;
        }
      }
    }
    return null;
  }

  /**
   * Busca a extenso sugerida com base em um tipo de arquivo aplicvel.
   * 
   * @param typeName o tipo
   * @return as extenses associadas ao tipo
   */
  static public String[] getExtensionsFromFileType(String typeName) {
    if (typeName == null) {
      throw new RuntimeException("Internal error: null type name not allowed");
    }
    if (typeName.equals(UNKNOWN)) {
      return null;
    }

    for (FileType value : e2tList) {
      String name = value.getName();
      if (name.trim().equalsIgnoreCase(typeName.trim())) {
        ArrayList<String> extensions = value.getExtensions();
        int sz = extensions.size();
        return extensions.toArray(new String[sz]);
      }
    }
    return null;
  }
}
