package testeproj;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Collection;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import csdk.v1_0.api.application.ApplicationException;
import csdk.v1_0.api.application.IApplication;
import csdk.v1_0.api.application.IApplicationContext;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.filesystem.FileSelectionType;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.IFileLock;
import csdk.v1_0.api.filesystem.IFileLock.LockStatus;
import csdk.v1_0.api.filesystem.project.IProjectContext;
import csdk.v1_0.api.filesystem.project.ProjectException;

/**
 * Aplicao de teste simples.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class TesteProj implements IApplication {

  /**
   * Interface padro para o ambiente CSDK.
   */
  private final ICSDKEnvironment csdkInterface;

  /**
   * Contexto de projeto da aplicao.
   */
  private final IProjectContext projectContext;

  /**
   * Contexto da aplicao.
   */
  private final IApplicationContext appContext;

  /**
   * Locks obtidos para os arquivos.
   */
  private final Collection<IFileLock> locks;

  /**
   * Janela principal da aplicao.
   */
  private final JFrame mainFrame;

  /**
   * Construtor padro.
   * 
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public TesteProj(final ICSDKEnvironment csdkInterface) {
    this.csdkInterface = csdkInterface;
    this.appContext = csdkInterface.getContext(IApplicationContext.class);
    this.projectContext = csdkInterface.getContext(IProjectContext.class);
    this.locks = new ArrayList<IFileLock>();
    this.mainFrame = new JFrame();
    this.mainFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    this.mainFrame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        csdkInterface.finishApplication();
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    int result =
      JOptionPane.showConfirmDialog(mainFrame, appContext
        .getString("confirm.message"), appContext.getApplicationName(),
        JOptionPane.YES_NO_OPTION);
    return result == JOptionPane.YES_OPTION;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onMessageReceived(IMessage message, IMessageSender sender) {
    // No faz nada
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationEnd() throws ApplicationException {
    mainFrame.dispose();
    for (IFileLock lock : locks) {
      lock.releaseLock(mainFrame);
    }
  }

  /**
   * Inicializa a interface da aplicao.
   */
  private void initUI() {
    String openLabelName = appContext.getString("open.label");
    final JLabel openLabel = new JLabel(openLabelName);
    String openButtonName = appContext.getString("open.button");
    JButton openButton = new JButton(openButtonName);
    openButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        IFile file = openFile(FileSelectionType.FILES_ONLY);
        if (file != null) {
          try {
            IFileLock sharedLock = file.acquireSharedLock(mainFrame);
            locks.add(sharedLock);
            if (sharedLock.getLockStatus() == LockStatus.LOCK_SHARED
              || sharedLock.getLockStatus() == LockStatus.LOCK_EXCLUSIVE) {
              openLabel.setText(file.getStringPath());
            }
            else {
              String lockMessage = appContext.getString("lock.message");
              JOptionPane.showMessageDialog(mainFrame, lockMessage);
            }
          }
          catch (Exception e) {
            csdkInterface.handleException(e, mainFrame);
          }
        }
      }
    });
    openButton.setEnabled(projectContext != null);
    String openMultipleLabelName = appContext.getString("open.multiple.label");
    final JLabel openMultipleLabel = new JLabel(openMultipleLabelName);
    String openMultipleButtonName =
      appContext.getString("open.multiple.button");
    JButton openMultipleButton = new JButton(openMultipleButtonName);
    openMultipleButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        IFile[] files = openMultipleFiles(FileSelectionType.FILES_ONLY);
        if (files != null) {
          StringBuilder builder = new StringBuilder();
          for (IFile file : files) {
            try {
              IFileLock sharedLock = file.acquireSharedLock(mainFrame);
              locks.add(sharedLock);
              if (sharedLock.getLockStatus() == LockStatus.LOCK_SHARED
                || sharedLock.getLockStatus() == LockStatus.LOCK_EXCLUSIVE) {
                builder.append(file.getStringPath());
                builder.append(" ");
              }
              else {
                String lockMessage = appContext.getString("lock.message");
                JOptionPane.showMessageDialog(mainFrame, lockMessage);
              }
            }
            catch (Exception e) {
              csdkInterface.handleException(e, mainFrame);
            }
            openMultipleLabel.setText(builder.toString());
          }
        }
      }
    });
    openMultipleButton.setEnabled(projectContext != null);
    String openDirLabelName = appContext.getString("open.dir.label");
    final JLabel openDirLabel = new JLabel(openDirLabelName);
    String openDirButtonName = appContext.getString("open.dir.button");
    JButton openDirButton = new JButton(openDirButtonName);
    openDirButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        IFile file = openFile(FileSelectionType.DIRECTORIES_ONLY);
        if (file != null) {
          openDirLabel.setText(file.getStringPath());
        }
      }
    });
    openDirButton.setEnabled(projectContext != null);
    String openMultipleDirLabelName =
      appContext.getString("open.multiple.dir.label");
    final JLabel openMultipleDirLabel = new JLabel(openMultipleDirLabelName);
    String openMultipleDirButtonName =
      appContext.getString("open.multiple.dir.button");
    JButton openMultipleDirButton = new JButton(openMultipleDirButtonName);
    openMultipleDirButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        IFile[] files = openMultipleFiles(FileSelectionType.DIRECTORIES_ONLY);
        if (files != null) {
          StringBuilder builder = new StringBuilder();
          for (IFile file : files) {
            builder.append(file.getStringPath());
            builder.append(" ");
          }
          openMultipleDirLabel.setText(builder.toString());
        }
      }
    });
    openMultipleDirButton.setEnabled(projectContext != null);
    String saveLabelName = appContext.getString("save.label");
    final JLabel saveLabel = new JLabel(saveLabelName);
    String saveButtonName = appContext.getString("save.button");
    JButton saveButton = new JButton(saveButtonName);
    saveButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        IFile file = saveFile();
        if (file != null) {
          try {
            IFileLock sharedLock = file.acquireExclusiveLock(mainFrame);
            locks.add(sharedLock);
            if (sharedLock.getLockStatus() == LockStatus.LOCK_SHARED
              || sharedLock.getLockStatus() == LockStatus.LOCK_EXCLUSIVE) {
              saveLabel.setText(file.getStringPath());
            }
            else {
              String lockMessage = appContext.getString("lock.message");
              JOptionPane.showMessageDialog(mainFrame, lockMessage);
            }
          }
          catch (Exception e) {
            csdkInterface.handleException(e, mainFrame);
          }
        }
      }
    });
    saveButton.setEnabled(projectContext != null);
    JPanel panel = new JPanel();
    panel.add(openLabel);
    panel.add(openButton);
    panel.add(openMultipleLabel);
    panel.add(openMultipleButton);
    panel.add(openDirLabel);
    panel.add(openDirButton);
    panel.add(openMultipleDirLabel);
    panel.add(openMultipleDirButton);
    panel.add(saveLabel);
    panel.add(saveButton);
    mainFrame.add(panel);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationStart() throws ApplicationException {
    initUI();
    mainFrame.pack();
    mainFrame.setVisible(true);
  }

  /**
   * Apresenta um navegador de arquivos em modo de abertura.
   * 
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   *        somente diretrios).
   * 
   * @return o arquivo selecionado pelo usurio ou nulo, caso a operao tenha
   *         sido cancelada pelo usurio.
   */
  protected IFile openFile(FileSelectionType selectionType) {
    try {
      String[] fileTypes = appContext.getApplicationFileTypes();
      return projectContext.browseProjectFileInOpenMode(fileTypes, true,
        selectionType, mainFrame);
    }
    catch (ProjectException e) {
      csdkInterface.handleException(e, mainFrame);
      return null;
    }
  }

  /**
   * Apresenta um navegador de mltiplos arquivos em modo de abertura.
   * 
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   *        somente diretrios).
   * 
   * @return o arquivo selecionado pelo usurio ou nulo, caso a operao tenha
   *         sido cancelada pelo usurio.
   */
  private IFile[] openMultipleFiles(FileSelectionType selectionType) {
    try {
      String[] fileTypes = appContext.getApplicationFileTypes();
      return projectContext.browseMultipleProjectFilesInOpenMode(fileTypes,
        false, selectionType, mainFrame);
    }
    catch (ProjectException e) {
      this.csdkInterface.handleException(e, mainFrame);
      return null;
    }
  }

  /**
   * Apresenta um navegador de arquivos em modo de gravao.
   * 
   * @return o arquivo selecionado pelo usurio ou nulo, caso a operao tenha
   *         sido cancelada pelo usurio.
   */
  protected IFile saveFile() {
    try {
      String[] fileTypes = appContext.getApplicationFileTypes();
      return projectContext.browseProjectFileInSaveMode(fileTypes, true, null,
        mainFrame);
    }
    catch (ProjectException e) {
      csdkInterface.handleException(e, mainFrame);
      return null;
    }
  }

}
