/*
 * $Id$
 */
package tecgraf.javautils.ant.taskdefs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

import org.apache.tools.ant.BuildException;

/**
 * Representa uma verso contida no arquivo de propriedades
 * 
 * @author Tecgraf/PUC-Rio
 * 
 */
public class Version {
  private static final String VERSIONDATE_KEY = "version-date";
  private static final String RELEASEDATE_KEY = "release-date";
  private static final String MAJORVERSION_KEY = "major-version";
  private static final String MINORVERSION_KEY = "minor-version";
  private static final String RELEASEVERSION_KEY = "release-version";
  private static final String FORMAT_KEY = "format";
  private static final String DATEFORMAT_KEY = "date-format";
  private static final String SEPARATOR_KEY = "separator";
 
  /** data da verso principal */ 
  private Date versionDate;
  
  /** data da verso de release */
  private Date releaseDate;
  
  /** a verso major */
  private String majorVersion;
  
  /** a verso minor */
  private String minorVersion;
  
  /** a verso release */
  private String releaseVersion;
  
  /** o formato usado na verso */  
  private String format;
  
  /** formato de data */
  private String dateFormat;
  
  /** separador usado na verso */
  private String separator;

  /** caminho do arquivo de propriedades */
  private String filePath;
  
  /** propriedades carregadas do arquivo */
  private Properties properties = new Properties();


  /**
   * Inicializa o controle de verso do arquivo passado como parmetro
   * 
   * @param versionPropertiesFile
   * @throws BuildException
   */
  public Version(String versionPropertiesFile) throws BuildException {
    try {
      this.loadVersionProperties(versionPropertiesFile);      
      this.dateFormat = properties.getProperty(DATEFORMAT_KEY).trim();
      SimpleDateFormat sdf = new SimpleDateFormat(this.dateFormat);
      this.versionDate =
        sdf.parse(properties.getProperty(VERSIONDATE_KEY).trim());
      this.releaseDate =
        sdf.parse(properties.getProperty(RELEASEDATE_KEY).trim());
      this.majorVersion = properties.getProperty(MAJORVERSION_KEY).trim();
      this.minorVersion = properties.getProperty(MINORVERSION_KEY).trim();
      this.releaseVersion = properties.getProperty(RELEASEVERSION_KEY).trim();
      this.format = properties.getProperty(FORMAT_KEY).trim();      
      this.separator = properties.getProperty(SEPARATOR_KEY).trim();
    }
    catch (Exception e) {
      throw new BuildException(e);
    }
    if (!this.validate())
      throw new BuildException("Formato invlido");
  }

  /**
   * Carrega o arquivo de verso
   * 
   * @param filePath
   * @throws IOException
   */
  private void loadVersionProperties(String filePath) throws IOException {
    this.filePath = filePath;
    InputStream is = new FileInputStream(new File(filePath));
    properties.load(is);
    is.close();
  }

  /**
   * @return A verso atual no formatada sem o release.
   */
  public String getCurrentVersion() {
    return this.majorVersion + "." + this.minorVersion;
  }

  /**
   * @return A verso atual no formatada incluindo o release.
   */
  public String getCurrentRelease() {    
    return this.majorVersion + "." + this.minorVersion + "." + this.releaseVersion;
  }

  /**
   * Atualiza o arquivo de verso de acordo com o estado
   * atual
   * 
   * @throws BuildException
   */
  public void updateProperties() throws BuildException {
    SimpleDateFormat sdf = new SimpleDateFormat(this.dateFormat);
    this.properties.put(VERSIONDATE_KEY, sdf.format(this.versionDate));
    this.properties.put(RELEASEDATE_KEY, sdf.format(this.releaseDate));
    this.properties.put(MAJORVERSION_KEY, this.majorVersion);
    this.properties.put(MINORVERSION_KEY, this.minorVersion);
    this.properties.put(RELEASEVERSION_KEY, this.releaseVersion);
    try {
      OutputStream os = new FileOutputStream(new File(filePath));
      this.properties.store(os, "");
      os.close();
    }
    catch (IOException e) {
      throw new BuildException(e);
    }
  }

  /**
   * @return Retorna a verso no formatada com o major version
   * incrementado. 
   */
  public String incMajorVersion() {
    this.versionDate = new Date();
    this.releaseDate = new Date();
    this.majorVersion =
      Integer.toString(Integer.parseInt(this.majorVersion) + 1);
    this.minorVersion = "0";
    this.releaseVersion = "0";
    return this.getCurrentVersion();
  }

  /**
   * @return Retorna a verso no formatada com o minor version
   * incrementado. 
   */
  public String incMinorVersion() {
    this.versionDate = new Date();
    this.releaseDate = new Date();
    this.minorVersion =
      Integer.toString(Integer.parseInt(this.minorVersion) + 1);
    this.releaseVersion = "0";
    return this.getCurrentVersion();
  }

  /**
   * @return Retorna a verso no formatada com o release version
   * incrementado. 
   */
  public String incReleaseVersion() {
    this.releaseDate = new Date();
    this.releaseVersion =
      Integer.toString(Integer.parseInt(this.releaseVersion) + 1);
    return this.getCurrentRelease();
  }
  
  /**
   * Retorna a verso formatada de acordo
   * com o formato escolhido no arquivo de propriedades.
   * 
   * @return
   */
  private String format(boolean showReleaseVersion) {
    SimpleDateFormat sdf = new SimpleDateFormat(this.dateFormat);
    String v = this.format;
    v = v.replace("<VERSIONDATE>", sdf.format(this.versionDate));
    v = v.replace("<MAJORVERSION>", this.majorVersion);
    v = v.replace("<MINORVERSION>", this.minorVersion);
    if (!this.releaseVersion.equals("-1") && showReleaseVersion)
      v = v.replace("<RELEASEVERSION>", this.releaseVersion);
    else {//argh
      v = v.replace(this.separator + "<RELEASEVERSION>", "");
      v = v.replace("<RELEASEVERSION>" + this.separator, "");
      v = v.replace(this.separator + "<RELEASEVERSION>" + this.separator, "");
    }
    return v;
  }
  
  /** 
   * @return Retorna a verso formatada de acordo
   * com o formato escolhido no arquivo de propriedades.
   * O retorno est de acordo com o estado atual da verso.
   *  
   */
  public String getFormattedVersion(boolean showReleaseVersion) {
    return this.format(showReleaseVersion);
  }

  /**
   * Atualiza a verso para a passada por parmetro
   * 
   * @param newVersion A nova verso a ser carregada, ex 2.1
   * @return True se a nova verso foi carregada com sucesso.
   */
  public boolean parse(String newVersion) {    
    boolean r;
    String[] a = newVersion.split("\\.");
    try {
      this.majorVersion = a[0];
      this.minorVersion = a[1];
      if (a.length == 3) 
        this.releaseVersion = a[2];              
      else 
        this.releaseVersion = "-1";        
      r = this.validate();
    }
    catch(Exception e) {
      e.printStackTrace();
      r = false;
    }
    return r;
  }

  /** 
   * @return True se as verses so inteiros vlidos.
   */
  private boolean validate() {
    boolean valid = true;
    try {
      Integer.parseInt(this.majorVersion);
      Integer.parseInt(this.minorVersion);
      Integer.parseInt(this.releaseVersion);
    }
    catch (NumberFormatException e) {
      valid = false;
    }
    return valid;
  }

}
