package tecgraf.javautils.concurrent.locks;

import java.lang.ref.WeakReference;
import java.util.List;
import java.util.Vector;

/**
 * Classe de lock
 * 
 * 
 */
public class Lock implements Comparable<Lock> {

  /**
   * Identificador do lock.  usado para identificar que o desbloqueio  feito
   * por quem pediu o lock.
   */
  private LockId id;

  /**
   * Poltica do lock
   */
  private LockPolicy policy;

  /**
   * Indica se o lock foi solicitado por um servio, a partir de uma
   * <code>Thread</code> do servidor. Quando o lock  obtido por uma
   * <code>Thread</code> que no vem diretamente do cliente, no h sesso
   * associada.
   */
  private boolean serverRequested = true;

  /**
   * Identificador do lock que originou este lock
   */
  private LockId originatorLockId;

  /**
   * Identificadores de locks dependentes deste lock
   */
  private List<LockId> dependencies;

  /**
   * <code>WeakReference</code> para o objeto de identificao do usurio que
   * obteve o lock.
   */
  private WeakReference<Object> ownerKeyRef = null;

  /**
   * Constri um <code>Lock</code>. Usado quando o pedido de lock veio de uma
   * chamada do servidor.
   * 
   * @param lockPolicy poltica do lock
   * @param originatorLockId identificador do lock originador deste
   */
  Lock(LockPolicy lockPolicy, LockId originatorLockId) {
    this.id = new LockId();
    this.policy = lockPolicy;
    this.originatorLockId = originatorLockId;
    if (this.originatorLockId == null) {
      this.originatorLockId = this.id;
    }
    //Cria vector de identificadores dependentes
    this.dependencies = new Vector<>();
  }

  /**
   * Constri um <code>Lock</code> com uma <code>WeakReference</code> para o
   * objeto de identificao do usurio que obteve o lock.
   * 
   * @param lockPolicy poltica do lock
   * @param ownerKey identificador do usurio detentor do lock
   * @param originatorLockId identificador do lock originador deste
   */
  Lock(LockPolicy lockPolicy, Object ownerKey, LockId originatorLockId) {
    this(lockPolicy, originatorLockId);
    this.ownerKeyRef = new WeakReference<>(ownerKey);
    this.serverRequested = false;
  }

  /**
   * Retorna o identificador do lock
   * 
   * @return o objeto identificador do lock
   */
  LockId getId() {
    return id;
  }

  /**
   * Retorna o objeto identificador do usurio dono do lock
   * 
   * @return flag indicando se o lock  compartilhado
   */
  Object getOwnerKey() {
    if (ownerKeyRef == null) {
      return null;
    }
    else {
      return ownerKeyRef.get();
    }
  }

  /**
   * Retorna a poltica de lock
   * 
   * @return a poltica de lock
   */
  public LockPolicy getPolicy() {
    return policy;
  }

  /**
   * Retorna se o lock  de primeira ordem
   * 
   * @return flag indicando se lock  de primeira ordem
   */
  public boolean isFirstOrder() {
    return this.id.equals(originatorLockId);
  }

  /**
   * Retorna o identificador do lock que originou este
   * 
   * @return o identificador do lock que originou este
   */
  public LockId getOriginatorLockId() {
    return this.originatorLockId;
  }

  /**
   * Retorna as dependncias do lock obtido
   * 
   * @return dependncias do lock obtido
   */
  public List<LockId> getIdDependencies() {
    return dependencies;
  }

  /**
   * Adiciona identificador de um lock dependente a este
   * 
   * @param lockId identificador de um lock dependente a este
   */
  public void addIdDependency(LockId lockId) {
    this.dependencies.add(lockId);
  }

  /**
   * Indica se todas a referncias ao lock expirou.
   * 
   * @return flag indicando se a referncia ao lock expirou
   */
  public boolean isInvalid() {
    if (serverRequested) {
      return false;
    }
    if (ownerKeyRef == null) {
      return true;
    }
    return ownerKeyRef.get() == null;
  }

  /**
   * Mtodo de comparao entre dois locks
   * 
   * {@inheritDoc}
   */
  @Override
  public int compareTo(Lock o) {
    if (this.id.equals(o.id)) {
      return 0;
    }
    return -1;
  }

}
