package tecgraf.javautils.concurrent.locks;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.Vector;

/**
 * Classe gerenciadora dos observadores dos estados dos locks
 * 
 * 
 * @author Tecgraf
 */
public class LockListenerManager {

  /**
   * Mapa de locklisteners e seus tratamentos
   */
  private HashMap<LockListener, Vector<ListenerVerifier>> listenerMap;

  /**
   * Construtor do gerenciador de listeners
   */
  LockListenerManager() {
    this.listenerMap = new HashMap<>(); //Ou arraylist?
  }

  /**
   * Listener padro - Est interessado em todos os objetos
   * 
   * @author Tecgraf
   */
  class ListenerVerifier {

    /** Listener de lock associado */
    LockListener lockListener;

    /**
     * Retorna se est interessado ou no
     * 
     * @param object objeto a ser investigado
     * @return se o listener est interessado sobre o objeto
     */
    boolean hasInterest(Object object) {
      return true;
    }
  }

  /**
   * Listener de tipo - Est interessado em todos os objetos de um certo tipo
   * 
   * @author Tecgraf
   */
  class ClassListenerVerifier extends ListenerVerifier {

    /** Tipo do objeto interessado */
    Class<?> theClass;

    /**
     * 
     * {@inheritDoc}
     */
    @Override
    boolean hasInterest(Object object) {
      return theClass.isInstance(object);
    }
  }

  /**
   * Listener de objeto - Est interessado em um objeto
   * 
   * 
   * @author Tecgraf
   */
  class ObjectListenerVerifier extends ListenerVerifier {

    /** Objeto pelo qual este listener est interessado */
    Object theObject;

    /**
     * 
     * {@inheritDoc}
     */
    @Override
    boolean hasInterest(Object object) {
      return theObject.equals(object);
    }
  }

  /**
   * Listener para observar o estado de todos os locks obtidos e liberados
   * 
   * @param lockListener observador utilizado para receber eventos
   */
  public void addLockListener(LockListener lockListener) {
    //Cria um verificador para observadores de todos os objetos
    ListenerVerifier listener = new ListenerVerifier();
    //Adiciona o lock listener
    listener.lockListener = lockListener;
    //Inser o novo listener
    this.newListener(listener);
  }

  /**
   * Listener para observar o estado de todos os locks de um determinado tipo de
   * objeto
   * 
   * @param theClass o tipo do objeto a ser observado
   * @param lockListener observador utilizado para receber eventos
   */
  void addLockListener(Class<?> theClass, LockListener lockListener) {
    //Cria um verificador para observadores de certo tipo
    ClassListenerVerifier listener = new ClassListenerVerifier();
    //Adicona o tipo
    listener.theClass = theClass;
    //Adiciona o lock listener
    listener.lockListener = lockListener;
    //Inser o novo listener
    this.newListener(listener);
  }

  /**
   * Listener para observar o estado de todos os locks de um determinado objeto
   * 
   * @param object o objeto a ser observado
   * @param lockListener observador utilizado para receber eventos
   */
  void addLockListener(Object object, LockListener lockListener) {
    //Cria um verificador para observadores de um certo objeto
    ObjectListenerVerifier listener = new ObjectListenerVerifier();
    //Adicona o tipo
    listener.theObject = object;
    //Adiciona o lock listener
    listener.lockListener = lockListener;
    //Inser o novo listener
    this.newListener(listener);
  }

  /**
   * Remove o listener de observao do estado de locks
   * 
   * @param lockListener observador utilizado para receber eventos
   */
  void removeLockListener(LockListener lockListener) {
    this.removeListener(lockListener);
  }

  /**
   * Mtodo utilizado para retornar todos os listeners interessados sobre o
   * estado do objeto passado como parmetro
   * 
   * @param object
   * @return o conjunto de listeners interessados sobre o objeto
   */
  public Set<LockListener> getListeners(Object object) {
    //Cria conjunto de observadores
    Set<LockListener> lockListeners = new HashSet<>();
    //Iterando sobre a lista de verificadores
    for (Vector<ListenerVerifier> listenerVerifierList : this.listenerMap
      .values()) {
      for (ListenerVerifier verifier : listenerVerifierList) {
        //Verifica se o listener est interessado no objeto
        if (verifier.hasInterest(object)) {
          //Adiciona o listener ao conjunto
          lockListeners.add(verifier.lockListener);
        }
      }
    }
    //Retorna o conjunto de listeners
    return lockListeners;
  }

  /**
   * Insero de um novo listener
   * 
   * @param listener o verificador do interesse
   */
  private void newListener(ListenerVerifier listener) {
    //Recupera a lista de observadores 
    Vector<ListenerVerifier> listeners =
      this.listenerMap.get(listener.lockListener);
    //Se no existem verificadores para o observador
    if (listeners == null) {
      //Cria a lista de verificadores para observador 
      listeners = new Vector<>();
      //Insere no mapa de observadores e verificadores a nova lista
      this.listenerMap.put(listener.lockListener, listeners);
    }
    //Adiciona o verificador criado na lista de locks
    listeners.add(listener);
  }

  /**
   * Remoo do observador e verificadores
   * 
   * @param lockListener o observador a ser removido
   */
  private void removeListener(LockListener lockListener) {
    //Remove do mapa os verificadores
    this.listenerMap.remove(lockListener);
  }

}
