package tecgraf.javautils.concurrent.locks;

/**
 * Poltica do Lock
 * 
 * @author Tecgraf
 */
public enum LockPolicy {
  /**
   * Exclusivo
   * 
   * - No pode haver nenhum outro lock. Porm, se houver reentrncia, o usurio
   * s obter este tipo de lock se j tiver obtido um lock do tipo exclusivo.
   */
  EXCLUSIVE(3) {
    @Override
    boolean allowsLock(LockInfo lockInfo, Object ownerKey,
      LockId originatorLockId) {
      return (lockInfo.getLocks(originatorLockId,
        this.getIncompatibleLockPolicies()).size() <= 0)
        || (lockInfo.allowsReentrance()
          && (lockInfo.getLocks(originatorLockId, LockPolicy.PROTECTED_WRITE,
            LockPolicy.PROTECTED_READ, LockPolicy.CONCURRENT_READ).size() <= 0) && (lockInfo
          .getLocks(ownerKey, LockPolicy.EXCLUSIVE).size() == lockInfo
          .getLocks(LockPolicy.EXCLUSIVE).size()));
    }

    @Override
    public LockPolicy[] getIncompatibleLockPolicies() {
      return new LockPolicy[] { LockPolicy.EXCLUSIVE,
          LockPolicy.PROTECTED_WRITE, LockPolicy.PROTECTED_READ,
          LockPolicy.CONCURRENT_READ };
    }
  },

  /**
   * Escritor protegido
   * 
   * - No pode haver nenhum outro lock de escrita (exclusiva ou protegida) e/ou
   * de leitura protegida. Porm, se houver reentrncia, todos os locks de
   * escrita devem pertencer ao usurio solicitante deste tipo de lock. Contudo,
   * o usurio no poder obter este tipo de lock se j tiver obtido um lock do
   * tipo leitor protegido ou concorrente.
   */
  PROTECTED_WRITE(2) {
    @Override
    boolean allowsLock(LockInfo lockInfo, Object ownerKey,
      LockId originatorLockId) {
      return (lockInfo.getLocks(originatorLockId,
        this.getIncompatibleLockPolicies()).size() <= 0)
        || (lockInfo.allowsReentrance()
          && (lockInfo.getLocks(originatorLockId, LockPolicy.PROTECTED_READ)
            .size() <= 0) && (lockInfo.getLocks(ownerKey,
          LockPolicy.PROTECTED_WRITE, LockPolicy.EXCLUSIVE).size() == lockInfo
          .getLocks(LockPolicy.PROTECTED_WRITE, LockPolicy.EXCLUSIVE).size()));
    }

    @Override
    public LockPolicy[] getIncompatibleLockPolicies() {
      return new LockPolicy[] { LockPolicy.EXCLUSIVE,
          LockPolicy.PROTECTED_WRITE, LockPolicy.PROTECTED_READ };
    }
  },
  /**
   * Leitor protegido (Protected read - Shared) / Compartilhado
   * 
   * - No pode haver nenhum lock de escrita. Porm, se houver reentrncia,
   * todos os locks de escrita devem pertencer ao usurio solicitante deste tipo
   * de lock. Contudo, o usurio no poder obter este tipo de lock se j tiver
   * obtido um lock do tipo leitor concorrente
   * 
   */
  PROTECTED_READ(1) {
    @Override
    boolean allowsLock(LockInfo lockInfo, Object ownerKey,
      LockId originatorLockId) {
      return (lockInfo.getLocks(originatorLockId,
        this.getIncompatibleLockPolicies()).size() <= 0)
        || (lockInfo.allowsReentrance() && (lockInfo.getLocks(ownerKey,
          LockPolicy.PROTECTED_WRITE, LockPolicy.EXCLUSIVE).size() == lockInfo
          .getLocks(LockPolicy.PROTECTED_WRITE, LockPolicy.EXCLUSIVE).size()));
    }

    @Override
    public LockPolicy[] getIncompatibleLockPolicies() {
      return new LockPolicy[] { LockPolicy.EXCLUSIVE,
          LockPolicy.PROTECTED_WRITE };
    }
  },

  /**
   * Leitor concorrente
   * 
   * - No pode haver locks exclusivos. Porm, se houver reentrncia, todos os
   * locks exclusivos devem pertencer ao usurio solicitante deste tipo de lock
   * 
   */
  CONCURRENT_READ(0) {
    @Override
    boolean allowsLock(LockInfo lockInfo, Object ownerKey,
      LockId originatorLockId) {
      return (lockInfo.getLocks(originatorLockId,
        this.getIncompatibleLockPolicies()).size() <= 0)
        || (lockInfo.allowsReentrance() && (lockInfo.getLocks(ownerKey,
          EXCLUSIVE).size() == lockInfo.getLocks(EXCLUSIVE).size()));
    }

    @Override
    public LockPolicy[] getIncompatibleLockPolicies() {
      return new LockPolicy[] { LockPolicy.EXCLUSIVE };
    }

  };

  /**
   * Valor da poltica
   */
  private int value;

  /**
   * Construtor do enumerador
   * 
   * @param value
   */
  LockPolicy(int value) {
    this.value = value;
  }

  /**
   * Retorna o valor da poltica
   * 
   * @return o valor da poltica
   */
  int getValue() {
    return this.value;
  }

  /**
   * Verifica se a poltica permite a aquisio de um lock
   * 
   * @param lockInfo o info de lock utilizado nesta verificao
   * @param ownerKey o identificador do usurio requisitor do lock
   * @param originatorLockId identificador do lock originador do lock solicitado
   * @return se o lock pode ser adquirido
   */
  abstract boolean allowsLock(LockInfo lockInfo, Object ownerKey,
    LockId originatorLockId);

  /**
   * Retorna as polticas de lock incompatveis com a poltica verificada
   * 
   * @return as polticas de lock incompatveis com a poltica verificada
   */
  public abstract LockPolicy[] getIncompatibleLockPolicies();

}
