package tecgraf.javautils.excel.v1.poi;

import java.util.HashMap;
import java.util.Map;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFPalette;
import org.apache.poi.hssf.util.HSSFColor;

import tecgraf.javautils.excel.v1.ExcelColor;
import tecgraf.javautils.excel.v1.ExcelStroke;
import tecgraf.javautils.excel.v1.ExcelStyle;
import tecgraf.javautils.excel.v1.ExcelStyleTool;

/**
 * Classe responsvel por realizar operaes de formatao.
 * 
 * 
 * @author bbreder
 */
public class PoiExcelStyleTool extends PoiExcelTool implements ExcelStyleTool {

  /** Cache dos estilos utilizados */
  private Map<String, ExcelStyle> styleNames =
    new HashMap<String, ExcelStyle>();
  /** Cache dos estilos utilizados para fazer um rebuild quando necessrio */
  private Map<HSSFCell, PoiExcelStyle> styles =
    new HashMap<HSSFCell, PoiExcelStyle>();
  /**
   * Cache de Estilos que dado um novo estilo, o cache informa se j existe um
   * criado anteriormente e o retorna. Isso serve para diminuir a quantidade de
   * estilos extras.
   */
  private Map<PoiExcelStyle, PoiExcelStyle> cacheOfBoxStyle =
    new HashMap<PoiExcelStyle, PoiExcelStyle>();

  /**
   * Construtor padro
   * 
   * @param helper
   */
  public PoiExcelStyleTool(PoiExcelSheet helper) {
    super(helper.getExcel().getWorkbook(), helper);
    this.confPalette();
  }

  /**
   * Configura a paleta de cores
   */
  private void confPalette() {
    HSSFPalette palette = this.getWorkbook().getCustomPalette();
    palette.setColorAtIndex(HSSFColor.ROSE.index, (byte) 255, (byte) 225,
      (byte) 225);
    palette.setColorAtIndex(HSSFColor.GREEN.index, (byte) 225, (byte) 255,
      (byte) 225);
    palette.setColorAtIndex(HSSFColor.BLUE.index, (byte) 225, (byte) 225,
      (byte) 255);
    palette.setColorAtIndex(HSSFColor.YELLOW.index, (byte) 255, (byte) 255,
      (byte) 225);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle createStyle() {
    PoiExcelSheet sheet = this.getSheet();
    PoiExcel excel = sheet.getExcel();
    return excel.getFactory().buildExcelStyle(excel);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool setStyle(int column, int row, ExcelStyle style) {
    return this.setStyle(column, column, row, row, style);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool setStyleRow(int column, int rowBegin, int rowEnd,
    ExcelStyle style) {
    return this.setStyle(column, column, rowBegin, rowEnd, style);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool setStyleColumn(int columnBegin, int columnEnd, int row,
    ExcelStyle style) {
    return this.setStyle(columnBegin, columnEnd, row, row, style);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool setStyle(int columnBegin, int columnEnd, int rowBegin,
    int rowEnd, ExcelStyle jstyle) {
    for (int column = columnBegin; column <= columnEnd; column++) {
      for (int row = rowBegin; row <= rowEnd; row++) {
        HSSFCell cell = this.getSheet().getCellMerged(column, row);
        HSSFCellStyle style = (HSSFCellStyle) jstyle.getAdapter();
        this.styles.put(cell, (PoiExcelStyle) jstyle);
        cell.setCellStyle(style);
      }
    }
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool addBox(int column, int row, ExcelStroke stroke,
    ExcelColor color) {
    return this.addBox(column, column, row, row, stroke, color);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool addBoxRow(int column, int rowBegin, int rowEnd,
    ExcelStroke stroke, ExcelColor color) {
    return this.addBox(column, column, rowBegin, rowEnd, stroke, color);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool addBoxColumn(int columnBegin, int columnEnd, int row,
    ExcelStroke stroke, ExcelColor color) {
    return this.addBox(columnBegin, columnEnd, row, row, stroke, color);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyleTool addBox(int columnBegin, int columnEnd, int rowBegin,
    int rowEnd, ExcelStroke stroke, ExcelColor color) {
    for (int row = rowBegin; row <= rowEnd; row++) {
      for (int column = columnBegin; column <= columnEnd; column++) {
        if (row == rowBegin || row == rowEnd || column == columnBegin
          || column == columnEnd) {
          PoiExcelStyle jstyle = this.getBoxStyle(column, row);
          if (jstyle == null) {
            jstyle = new PoiExcelStyle(this.getWorkbook());
            applyBorder(columnBegin, columnEnd, rowBegin, rowEnd, column, row,
              jstyle, stroke, color);
          }
          else {
            jstyle = jstyle.clone();
            applyBorder(columnBegin, columnEnd, rowBegin, rowEnd, column, row,
              jstyle, stroke, color);
          }
          PoiExcelStyle styleCache = this.cacheOfBoxStyle.get(jstyle);
          if (styleCache == null) {
            this.cacheOfBoxStyle.put(jstyle, jstyle);
            jstyle.buildAdapter();
          }
          else {
            jstyle = styleCache;
          }
          HSSFCell cell = this.getSheet().getCell(column, row);
          cell.setCellStyle((HSSFCellStyle) jstyle.getAdapter());
          this.setStyle(column, row, jstyle);
        }
      }
    }
    return this;
  }

  /**
   * Aplica a borda
   * 
   * @param columnBegin
   * @param columnEnd
   * @param rowBegin
   * @param rowEnd
   * @param column
   * @param row
   * @param style
   * @param stroke
   * @param color
   */
  private void applyBorder(int columnBegin, int columnEnd, int rowBegin,
    int rowEnd, int column, int row, PoiExcelStyle style, ExcelStroke stroke,
    ExcelColor color) {
    if (color == null) {
      color = ExcelColor.BLACK;
    }
    if (row == rowBegin) {
      style.borderTop(stroke, color);
    }
    if (row == rowEnd) {
      style.borderBottom(stroke, color);
    }
    if (column == columnBegin) {
      style.borderLeft(stroke, color);
    }
    if (column == columnEnd) {
      style.borderRight(stroke, color);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle getStyle(String key) {
    return this.styleNames.get(key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setStyle(String key, ExcelStyle style) {
    this.styleNames.put(key, style);
  }

  /**
   * Retorna o estilo de uma linha e coluna
   * 
   * @param columnIndex
   * @param rowIndex
   * @return estilo
   */
  public PoiExcelStyle getBoxStyle(int columnIndex, int rowIndex) {
    HSSFCell cell = this.getSheet().getCell(columnIndex, rowIndex);
    PoiExcelStyle style = this.styles.get(cell);
    return style;
  }

  /**
   * Retorna o estilo de uma linha e coluna
   * 
   * @param column
   * @param row
   * @param style
   */
  private void setStyle(int column, int row, PoiExcelStyle style) {
    HSSFCell cell = this.getSheet().getCell(column, row);
    this.styles.put(cell, style);
  }

}
