package tecgraf.javautils.excel.v1.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Estrutura simples de Xml para fazer parse. O parse pode ser feito pelo
 * construtor da classe.
 * 
 * @author bernardobreder
 * 
 */
public class XmlNode implements Comparable<XmlNode> {

  /** Nome da tag */
  private String name;
  /** Attributos */
  private final TreeMap<String, String> attributes =
    new TreeMap<String, String>();
  /** Filhos */
  private final SortedSet<XmlNode> nodes = new TreeSet<XmlNode>();

  /**
   * Construtor
   * 
   * @param name
   */
  public XmlNode(String name) {
    this.name = name;
  }

  /**
   * Construtor
   * 
   * @param input
   * @param onlyHeader
   * @throws ParseException
   * @throws IOException
   */
  public XmlNode(InputStream input, boolean onlyHeader) throws ParseException,
    IOException {
    try {
      DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
      DocumentBuilder db = dbf.newDocumentBuilder();
      InputSource source = new InputSource(input);
      source.setEncoding("utf-8");
      Document document = db.parse(source);
      if (document.getFirstChild() instanceof Element) {
        Element element = (Element) document.getFirstChild();
        this.read(element, onlyHeader);
      }
    }
    catch (ParserConfigurationException e) {
      throw new ParseException(e.getMessage(), 0);
    }
    catch (SAXException e) {
      throw new ParseException(e.getMessage(), 0);
    }
  }

  /**
   * Construtor
   * 
   * @param element
   */
  private XmlNode(Element element) {
    this.read(element, false);
  }

  /**
   * Realiza a leitura da tag
   * 
   * @param root
   * @param onlyHeader
   */
  private void read(Element root, boolean onlyHeader) {
    this.name = root.getNodeName();
    NamedNodeMap atts = root.getAttributes();
    if (atts != null) {
      int size = atts.getLength();
      for (int n = 0; n < size; n++) {
        Node node = atts.item(n);
        if (node instanceof Attr) {
          Attr attr = (Attr) node;
          this.attributes.put(attr.getName(), attr.getValue());
        }
      }
    }
    if (onlyHeader) {
      NodeList nodes = root.getChildNodes();
      if (nodes != null) {
        int size = nodes.getLength();
        for (int n = 0; n < size; n++) {
          Node node = nodes.item(n);
          if (node instanceof Element) {
            Element element = (Element) node;
            if (element.getTagName().equals("head")) {
              XmlNode child = new XmlNode(element);
              this.nodes.add(child);
              break;
            }
          }
        }
      }
    }
    else {
      NodeList nodes = root.getChildNodes();
      if (nodes != null) {
        int size = nodes.getLength();
        for (int n = 0; n < size; n++) {
          Node node = nodes.item(n);
          if (node instanceof Element) {
            Element element = (Element) node;
            XmlNode child = new XmlNode(element);
            this.nodes.add(child);
          }
        }
      }
    }
  }

  /**
   * @return the name
   */
  public String getName() {
    return name;
  }

  /**
   * @return the attributes
   */
  public Map<String, String> getAttributes() {
    return attributes;
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @return tem o atributo
   */
  public boolean hasAttribute(String attribute) {
    if (this.attributes == null) {
      return false;
    }
    return this.attributes.get(attribute) != null;
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @return tem o atributo
   */
  public String getAttribute(String attribute) {
    if (this.attributes == null) {
      return null;
    }
    return this.attributes.get(attribute);
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @param defaultValue
   * @return tem o atributo
   */
  public Integer getAttribute(String attribute, Integer defaultValue) {
    if (this.attributes == null) {
      return defaultValue;
    }
    try {
      return Integer.valueOf(this.attributes.get(attribute));
    }
    catch (NumberFormatException e) {
      return defaultValue;
    }
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @param defaultValue
   * @return tem o atributo
   */
  public String getAttribute(String attribute, String defaultValue) {
    if (this.attributes == null) {
      return defaultValue;
    }
    String value = this.attributes.get(attribute);
    if (value == null) {
      return defaultValue;
    }
    return value;
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @param defaultValue
   * @return tem o atributo
   */
  public Double getAttribute(String attribute, Double defaultValue) {
    if (this.attributes == null) {
      return defaultValue;
    }
    try {
      return Double.valueOf(this.attributes.get(attribute));
    }
    catch (NumberFormatException e) {
      return defaultValue;
    }
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @param defaultValue
   * @return tem o atributo
   */
  public Long getAttribute(String attribute, Long defaultValue) {
    if (this.attributes == null) {
      return defaultValue;
    }
    try {
      return Long.valueOf(this.attributes.get(attribute));
    }
    catch (NumberFormatException e) {
      return defaultValue;
    }
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @param defaultValue
   * @return tem o atributo
   */
  public Float getAttribute(String attribute, Float defaultValue) {
    if (this.attributes == null) {
      return defaultValue;
    }
    try {
      return Float.valueOf(this.attributes.get(attribute));
    }
    catch (NumberFormatException e) {
      return defaultValue;
    }
  }

  /**
   * Indica se tem o atributo
   * 
   * @param attribute
   * @param defaultValue
   * @return tem o atributo
   */
  public Boolean getAttribute(String attribute, Boolean defaultValue) {
    if (this.attributes == null) {
      return defaultValue;
    }
    return Boolean.valueOf(this.attributes.get(attribute));
  }

  /**
   * Adiciona ou modifica um atributo
   * 
   * @param key
   * @param value
   * @return owner
   */
  public XmlNode setAttribute(String key, String value) {
    this.attributes.put(key, value);
    return this;
  }

  /**
   * @param name the name to set
   * @return this
   */
  public XmlNode setName(String name) {
    this.name = name;
    return this;
  }

  /**
   * @return the nodes
   */
  public SortedSet<XmlNode> getNodes() {
    return nodes;
  }

  /**
   * @return the nodes
   */
  public int getNodeCount() {
    return nodes == null ? 0 : nodes.size();
  }

  /**
   * Adiciona uma tag
   * 
   * @param node
   * @return owner
   */
  public XmlNode addNode(XmlNode node) {
    this.nodes.add(node);
    return this;
  }

  /**
   * Adiciona varios nodes
   * 
   * @param list
   * @return this
   */
  public XmlNode addNodes(Collection<XmlNode> list) {
    if (list != null) {
      this.nodes.addAll(list);
    }
    return this;
  }

  /**
   * @param name
   * @return the nodes
   */
  public List<XmlNode> getNodesByTagName(String name) {
    List<XmlNode> list = new ArrayList<XmlNode>();
    for (XmlNode node : this.nodes) {
      if (node.name.equals(name)) {
        list.add(node);
      }
    }
    return list;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public List<XmlNode> getNodesByAttributeValue(String attribute, String value) {
    List<XmlNode> list = new ArrayList<XmlNode>();
    for (XmlNode node : this.nodes) {
      if (node.attributes != null) {
        String attValue = node.attributes.get(attribute);
        if (attValue != null && attValue.equals(value)) {
          list.add(node);
        }
      }
    }
    return list;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public List<XmlNode> getNodesByAttributeKey(String attribute, String value) {
    List<XmlNode> list = new ArrayList<XmlNode>();
    for (XmlNode node : this.nodes) {
      if (node.attributes != null) {
        String attValue = node.attributes.get(attribute);
        if (attValue != null) {
          list.add(node);
        }
      }
    }
    return list;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public List<XmlNode> getNodesByAttributeContainValue(String attribute,
    String value) {
    List<XmlNode> list = new ArrayList<XmlNode>();
    for (XmlNode node : this.nodes) {
      if (node.attributes != null) {
        String attValue = node.attributes.get(attribute);
        if (attValue != null && attValue.contains(value)) {
          list.add(node);
        }
      }
    }
    return list;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public List<XmlNode> getNodesByAttributeMatchValue(String attribute,
    String value) {
    List<XmlNode> list = new ArrayList<XmlNode>();
    for (XmlNode node : this.nodes) {
      if (node.attributes != null) {
        String attValue = node.attributes.get(attribute);
        if (attValue != null && attValue.matches(value)) {
          list.add(node);
        }
      }
    }
    return list;
  }

  /**
   * @param name
   * @return the nodes
   */
  public XmlNode getNodeByTagName(String name) {
    for (XmlNode node : this.nodes) {
      if (node.name.equals(name)) {
        return node;
      }
    }
    return null;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public XmlNode getNodeByAttributeValue(String attribute, String value) {
    for (XmlNode node : this.nodes) {
      String attValue = node.attributes.get(attribute);
      if (attValue != null && attValue.equals(value)) {
        return node;
      }
    }
    return null;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public XmlNode getNodeByAttributeKey(String attribute, String value) {
    for (XmlNode node : this.nodes) {
      String attValue = node.attributes.get(attribute);
      if (attValue != null) {
        return node;
      }
    }
    return null;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public XmlNode getNodeByAttributeContainValue(String attribute, String value) {
    for (XmlNode node : this.nodes) {
      String attValue = node.attributes.get(attribute);
      if (attValue != null && attValue.contains(value)) {
        return node;
      }
    }
    return null;
  }

  /**
   * @param attribute
   * @param value
   * @return the nodes
   */
  public XmlNode getNodeByAttributeMatchValue(String attribute, String value) {
    if (this.nodes != null) {
      for (XmlNode node : this.nodes) {
        if (node.attributes != null) {
          String attValue = node.attributes.get(attribute);
          if (attValue != null && attValue.matches(value)) {
            return node;
          }
        }
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.toString(0);
  }

  /**
   * @param tab
   * @return String
   */
  private String toString(int tab) {
    StringBuilder sb = new StringBuilder();
    for (int n = 0; n < tab; n++) {
      sb.append('\t');
    }
    sb.append('<');
    sb.append(name);
    for (String key : this.attributes.keySet()) {
      sb.append(' ');
      sb.append(key);
      sb.append('=');
      sb.append('\"');
      sb.append(this.attributes.get(key));
      sb.append('\"');
    }
    if (this.nodes.isEmpty()) {
      sb.append('/');
      sb.append('>');
    }
    else {
      sb.append('>');
      sb.append('\n');
      for (XmlNode node : this.nodes) {
        sb.append(node.toString(tab + 1));
        sb.append('\n');
      }
      for (int n = 0; n < tab; n++) {
        sb.append('\t');
      }
      sb.append('<');
      sb.append('/');
      sb.append(name);
      sb.append('>');
    }
    return sb.toString();
  }

  /**
   * Recupera os bytes do xml
   * 
   * @return bytes
   * @throws IOException
   */
  public byte[] getBytes() throws IOException {
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    StringOutputStream string = new StringOutputStream(output);
    getBytes(string, this);
    return output.toByteArray();
  }

  /**
   * Recupera os bytes do xml
   * 
   * @param output
   * @param root
   * @throws IOException
   */
  private static void getBytes(StringOutputStream output, XmlNode root)
    throws IOException {
    output.write('<');
    output.append(root.name);
    if (root.attributes != null) {
      for (String key : root.attributes.keySet()) {
        output.write(' ');
        output.append(key);
        output.write('=');
        output.write('\"');
        output.append(root.attributes.get(key));
        output.write('\"');
      }
    }
    if (root.nodes.isEmpty()) {
      output.write('/');
      output.write('>');
    }
    else {
      output.write('>');
      for (XmlNode node : root.nodes) {
        getBytes(output, node);
      }
      output.write('<');
      output.write('/');
      output.append(root.name);
      output.write('>');
    }
  }

  /**
   * Escreve no output
   * 
   * @param output
   * @throws IOException
   */
  public void write(OutputStream output) throws IOException {
    StringOutputStream out = new StringOutputStream(output);
    getBytes(out, this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(XmlNode o) {
    if (o == null || o.getClass() != XmlNode.class) {
      return this.getClass().getName().compareTo(o.getClass().getName());
    }
    int compareTo = this.name.compareTo(o.name);
    if (compareTo != 0) {
      return compareTo;
    }
    if (this.attributes.size() != o.attributes.size()) {
      return this.attributes.size() - o.attributes.size();
    }
    if (this.nodes.size() != o.nodes.size()) {
      return this.nodes.size() - o.nodes.size();
    }
    String tOrder = this.attributes.get("order");
    String oOrder = o.attributes.get("order");
    if (tOrder != null && oOrder != null) {
      compareTo = tOrder.compareTo(oOrder);
      if (compareTo != 0) {
        return compareTo;
      }
    }
    else if (tOrder != null && oOrder == null) {
      return -1;
    }
    else if (tOrder == null && oOrder != null) {
      return 1;
    }
    for (String key : this.attributes.keySet()) {
      String oValue = o.attributes.get(key);
      if (oValue != null) {
        String tValue = this.attributes.get(key);
        compareTo = tValue.compareTo(oValue);
        if (compareTo != 0) {
          return compareTo;
        }
      }
    }
    for (XmlNode node : this.nodes) {
      if (!o.nodes.contains(node)) {
        return 1;
      }
    }
    return 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return 31 * (31 * 1 + name.hashCode()) + attributes.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    XmlNode other = (XmlNode) obj;
    if (name == null) {
      if (other.name != null) {
        return false;
      }
    }
    else if (!name.equals(other.name)) {
      return false;
    }
    if (attributes == null) {
      if (other.attributes != null) {
        return false;
      }
    }
    else if (!attributes.equals(other.attributes)) {
      return false;
    }
    if (nodes == null) {
      if (other.nodes != null) {
        return false;
      }
    }
    else if (!nodes.equals(other.nodes)) {
      return false;
    }
    return true;
  }

  /**
   * String para output stream
   * 
   * @author bernardobreder
   * 
   */
  private static class StringOutputStream extends OutputStream {

    /** Saída */
    private final OutputStream output;

    /**
     * Construtor
     * 
     * @param output
     */
    public StringOutputStream(OutputStream output) {
      this.output = output;
    }

    /**
     * Acrescenta uma string
     * 
     * @param text
     * @throws IOException
     */
    public void append(String text) throws IOException {
      int size = text.length();
      for (int n = 0; n < size; n++) {
        char c = text.charAt(n);
        if (c <= 0x7F) {
          output.write(c);
        }
        else if (c <= 0x7FF) {
          output.write(((c >> 6) & 0x1F) + 0xC0);
          output.write((c & 0x3F) + 0x80);
        }
        else {
          output.write(((c >> 12) & 0xF) + 0xE0);
          output.write(((c >> 6) & 0x3F) + 0x80);
          output.write((c & 0x3F) + 0x80);
        }
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void write(int n) throws IOException {
      output.write(n);
    }

  }

  /**
   * Leitor de String em UTF8
   * 
   * 
   * @author Tecgraf
   */
  public static class StringInputStream extends InputStream {

    /** Stream */
    private final InputStream input;

    /**
     * @param input
     */
    public StringInputStream(InputStream input) {
      this.input = input;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int read() throws IOException {
      int c = this.input.read();
      if (c <= 0x7F) {
        return c;
      }
      else if ((c >> 5) == 0x6) {
        int i2 = this.input.read();
        return ((c & 0x1F) << 6) + (i2 & 0x3F);
      }
      else {
        int i2 = this.input.read();
        int i3 = this.input.read();
        return ((c & 0xF) << 12) + ((i2 & 0x3F) << 6) + (i3 & 0x3F);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int available() throws IOException {
      return this.input.available();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void close() throws IOException {
      this.input.close();
    }

  }

}
