package tecgraf.javautils.excel.v1.poi;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.Region;

import tecgraf.javautils.excel.v1.ExcelModel;
import tecgraf.javautils.excel.v1.ExcelSheet;

/**
 * Classe responsvel por facilitar a construo de uma Sheet. Nela, ser
 * trabalhado todas as etapas para construir e popular uma sheet.
 * 
 * 
 * @author bbreder
 */
public class PoiExcelSheet implements ExcelSheet {

  /** Ajudante da Planilha Excel */
  private final PoiExcel helper;
  /** Sheet Excel */
  private HSSFSheet sheet;
  /** Modelo utilizado para popular o sheet */
  private ExcelModel model;

  /**
   * Construtor padro
   * 
   * @param helper
   */
  public PoiExcelSheet(PoiExcel helper) {
    this.helper = helper;
  }

  /**
   * Funo que ir construir uma planilha excel.
   * 
   * @param model
   * @param workbook
   * @param sheet
   */
  public void build(ExcelModel model, HSSFWorkbook workbook, HSSFSheet sheet) {
    this.model = model;
    this.sheet = sheet;
    {
      PoiExcelStructureTool head = new PoiExcelStructureTool(this);
      head.setSheet(sheet);
      PoiExcelDataTool data = new PoiExcelDataTool(this);
      data.setSheet(sheet);
      PoiExcelStyleTool style = new PoiExcelStyleTool(this);
      head.setSheet(sheet);
      this.model.build(head, data, style);
    }
  }

  /**
   * Metodo que retorna um celula, dado uma coordenada. Essa funo considera
   * celulas com merge.
   * 
   * @param column
   * @param row
   * @return celula do sheet
   */
  protected HSSFCell getCellMerged(int column, int row) {
    int rowIndex = row - 1;
    short colIndex = (short) (column - 1);
    for (int n = 0; n < this.sheet.getNumMergedRegions(); n++) {
      Region region = this.sheet.getMergedRegionAt(n);
      if (region.contains(rowIndex, colIndex)) {
        rowIndex = region.getRowFrom();
        colIndex = region.getColumnFrom();
        break;
      }
    }
    HSSFRow poiRow = this.sheet.getRow(rowIndex);
    if (poiRow == null) {
      poiRow = this.sheet.createRow(rowIndex);
    }
    HSSFCell poiCell = poiRow.getCell(colIndex);
    if (poiCell == null) {
      poiCell = poiRow.createCell(colIndex);
    }
    return poiCell;
  }

  /**
   * Metodo que retorna um celula, dado uma coordenada.
   * 
   * @param column
   * @param row
   * @return celula do sheet
   */
  protected HSSFCell getCell(int column, int row) {
    int rowIndex = row - 1;
    short colIndex = (short) (column - 1);
    HSSFRow poiRow = this.sheet.getRow(rowIndex);
    if (poiRow == null) {
      poiRow = this.sheet.createRow(rowIndex);
    }
    HSSFCell poiCell = poiRow.getCell(colIndex);
    if (poiCell == null) {
      poiCell = poiRow.createCell(colIndex);
    }
    return poiCell;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcel getExcel() {
    return helper;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelModel getModel() {
    return model;
  }

  /**
   * Retorna a planilha poi
   * 
   * @return estrutrua poi
   */
  public HSSFSheet getPoiSheet() {
    return sheet;
  }

}
