package tecgraf.javautils.excel.v1.poi;

import org.apache.poi.ss.util.CellRangeAddress;

import tecgraf.javautils.excel.v1.ExcelMerge;
import tecgraf.javautils.excel.v1.ExcelStructureTool;
import tecgraf.javautils.excel.v1.util.DefaultExcelMerge;

/**
 * Classe responsvel por realizar operaes de merges.
 * 
 * 
 * @author bbreder
 */
public class PoiExcelStructureTool extends PoiExcelTool implements
  ExcelStructureTool {

  /** Fator multiplicador para converter unidade Excel pra Pixel */
  private static final int EXCEL_UNIT_TO_PIXEL = 10000 / 235;
  /** Referencia para o ajudante do sheet */
  private final PoiExcelSheet helper;
  /** Inicio de coluna corrente */
  private int columnBegin;
  /** Fim de coluna corrente */
  private int columnEnd;
  /** Inicio de linha corrente */
  private int rowBegin;
  /** Fim de linha corrente */
  private int rowEnd;

  /**
   * Construtor padro
   * 
   * @param helper
   */
  public PoiExcelStructureTool(PoiExcelSheet helper) {
    super(helper.getExcel().getWorkbook(), helper);
    this.helper = helper;
    this.columnBegin = 1;
    this.columnEnd = 1;
    this.rowBegin = 1;
    this.rowEnd = 1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStructureTool merge(int columnBegin, int columnEnd,
    int rowBegin, int rowEnd) {
    this.columnBegin = columnBegin;
    this.columnEnd = columnEnd;
    this.rowBegin = rowBegin;
    this.rowEnd = rowEnd;
    this.getPoiSheet().addMergedRegion(
      new CellRangeAddress(rowBegin - 1, rowEnd - 1, columnBegin - 1,
        columnEnd - 1));
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelMerge hasMerge(int columnIndex, int rowIndex) {
    short colIndex = (short) (columnIndex - 1);
    for (int n = 0; n < this.getPoiSheet().getNumMergedRegions(); n++) {
      CellRangeAddress region = this.getPoiSheet().getMergedRegion(n);
      if (region.isInRange(rowIndex - 1, colIndex)) {
        return new DefaultExcelMerge(region.getFirstColumn() + 1, region
          .getLastColumn() + 1, region.getFirstRow() + 1,
          region.getLastRow() + 1);
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStructureTool mergeRow(int column, int rowBegin, int rowEnd) {
    return this.merge(column, column, rowBegin, rowEnd);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStructureTool mergeColumn(int columnBegin, int columnEnd,
    int row) {
    return this.merge(columnBegin, columnEnd, row, row);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStructureTool repeatLineMerge(int... columns) {
    if (rowBegin < 1 || rowEnd < 1 || rowBegin > Short.MAX_VALUE
      || rowEnd > Short.MAX_VALUE) {
      throw new IllegalStateException();
    }
    for (int column : columns) {
      this.mergeRow(column, rowBegin, rowEnd);
    }
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStructureTool repeatColumnMerge(int... rows) {
    if (columnBegin < 1 || columnEnd < 1 || columnBegin > Short.MAX_VALUE
      || columnEnd > Short.MAX_VALUE) {
      throw new IllegalStateException();
    }
    for (int row : rows) {
      this.mergeColumn(columnBegin, columnEnd, row);
    }
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStructureTool setColumnWidth(int column, int width) {
    this.columnBegin = column;
    this.columnEnd = column;
    width = Math.min(width * EXCEL_UNIT_TO_PIXEL, Short.MAX_VALUE);
    this.getPoiSheet().setColumnWidth(column - 1, width);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnWidth(int column) {
    return this.getPoiSheet().getColumnWidth(column - 1) / EXCEL_UNIT_TO_PIXEL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStructureTool setRowHeight(int row, int height) {
    this.rowBegin = row;
    this.rowEnd = row;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelSheet getHelper() {
    return helper;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumn(int columnBegin, int columnEnd) {
    this.setColumnBegin(columnBegin);
    this.setColumnEnd(columnEnd);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setRow(int rowBegin, int rowEnd) {
    this.setRowBegin(rowBegin);
    this.setRowEnd(rowEnd);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnBegin() {
    return columnBegin;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnBegin(int columnBegin) {
    if (columnBegin < 1 || columnBegin > Short.MAX_VALUE) {
      throw new IllegalArgumentException("" + columnBegin);
    }
    this.columnBegin = columnBegin;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnEnd() {
    return columnEnd;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnEnd(int columnEnd) {
    if (columnEnd < 1 || columnEnd > Short.MAX_VALUE) {
      throw new IllegalArgumentException("" + columnEnd);
    }
    this.columnEnd = columnEnd;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowBegin() {
    return rowBegin;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setRowBegin(int rowBegin) {
    this.rowBegin = rowBegin;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowEnd() {
    return rowEnd;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setRowEnd(int rowEnd) {
    this.rowEnd = rowEnd;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return "PoiExcelStructureTool [columnBegin=" + columnBegin + ", columnEnd="
      + columnEnd + ", rowBegin=" + rowBegin + ", rowEnd=" + rowEnd + "]";
  }

}
