package tecgraf.javautils.excel.v1.style;

import tecgraf.javautils.excel.v1.ExcelColor;
import tecgraf.javautils.excel.v1.ExcelStyle;
import tecgraf.javautils.excel.v1.ExcelStyleTool;

/**
 * Classe que armazena todos os estilos utilizado pelo sistema BandeiraBr. Todos
 * os relatrios do BandeiraBr deve utilizar destes estilos para manter um
 * padro de relatrio.
 * 
 * 
 * @author bbreder
 */
public abstract class DefaultExcelStyleSet {

  /**
   * Constroi e armazena em cache um estilo no meio
   * 
   * @param styleTool ferramenta de estilo
   * @return estilo negrito
   */
  public static ExcelStyle buildCenter(ExcelStyleTool styleTool) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle = styleTool.createStyle().center();
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Constroi e armazena em cache um estilo centralizado e pintado.
   * 
   * @param styleTool referncia para a ferramenta de criao de estilos.
   * @param excelColor cor para pintar o fundo.
   * @return estilo para clulas Excel.
   */
  public static ExcelStyle buildCenteredAndPainted(ExcelStyleTool styleTool,
    ExcelColor excelColor) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle = styleTool.createStyle().center().backcolor(excelColor);
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Constroi e armazena em cache um estilo negrito
   * 
   * @param styleTool ferramenta de estilo
   * @return estilo negrito
   */
  public static ExcelStyle buildBold(ExcelStyleTool styleTool) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle = styleTool.createStyle().bold();
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Constroi e armazena em cache um estilo italico
   * 
   * @param styleTool ferramenta de estilo
   * @return estilo italico
   */
  public static ExcelStyle buildItalic(ExcelStyleTool styleTool) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle = styleTool.createStyle().italic();
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Constroi um estilo para a palavra 'reservada' e armazena em cache um estilo
   * para a palavra "Reservado"
   * 
   * @param styleTool ferramenta de estilo
   * @return estilo para o reservado
   */
  public static ExcelStyle buildReserved(ExcelStyleTool styleTool) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle =
        styleTool.createStyle().right().bold().forecolor(ExcelColor.RED);
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Constroi um estilo de cabealho da planilha e armazena em cache um estilo
   * para a palavra "Reservado"
   * 
   * @param styleTool ferramenta de estilo
   * @return estilo para o reservado
   */
  public static ExcelStyle buildTopHeader(ExcelStyleTool styleTool) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle = styleTool.createStyle().bold().backcolor(ExcelColor.BLUE);
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Constroi um estilo de cabealho de linha e armazena em cache um estilo para
   * a palavra "Reservado"
   * 
   * @param styleTool ferramenta de estilo
   * @return estilo para o reservado
   */
  public static ExcelStyle buildRowHeader(ExcelStyleTool styleTool) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle =
        styleTool.createStyle().bold().left().backcolor(ExcelColor.YELLOW);
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Constroi um estilo de cabealho de coluna e armazena em cache um estilo
   * para a palavra "Reservado"
   * 
   * @param styleTool ferramenta de estilo
   * @return estilo para o reservado
   */
  public static ExcelStyle buildColumnHeader(ExcelStyleTool styleTool) {
    String key = getKey();
    ExcelStyle excelStyle = styleTool.getStyle(key);
    if (excelStyle == null) {
      excelStyle =
        styleTool.createStyle().bold().center().backcolor(ExcelColor.GREEN);
      styleTool.setStyle(key, excelStyle);
    }
    return excelStyle;
  }

  /**
   * Retorna a chave utilizado para efetuar o cache do estilo. Retorna o nome do
   * mtodo que chamou essa operao "getKey()".
   * 
   * @return chave para o cache
   */
  protected static String getKey() {
    return Thread.currentThread().getStackTrace()[2].toString();
  }

}
