package tecgraf.javautils.excel.v1.util;

import java.awt.Component;
import java.awt.Desktop;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileFilter;

import tecgraf.javautils.excel.v1.ExcelModel;
import tecgraf.javautils.excel.v1.poi.PoiExcel;

/**
 * Ao que serve para exportar os dados de um componente para o formato XLS.
 * 
 * @author mjulia
 */
public abstract class ExcelAction extends AbstractAction {

  /** A janela de onde essa ao  chamada */
  public Component owner;

  /**
   * Construtor.
   * 
   * @param owner a janela de onde essa ao  chamada
   * @param title
   */
  public ExcelAction(Component owner, String title) {
    super(ExcelAction.NAME);
    putValue(Action.NAME, title);
    this.owner = owner;
  }

  /**
   * Retorna os modelos
   * 
   * @return modelos
   */
  public abstract ExcelModel[] getModels();

  /**
   * @param e
   */
  protected void handler(Throwable e) {
    e.printStackTrace();
  }

  /**
   * @param file
   * @throws IOException
   */
  protected void openInEDT(final File file) throws IOException {
    if (Desktop.isDesktopSupported()) {
      Desktop.getDesktop().open(file);
    }
  }

  /**
   * @param excel
   * @param file
   * @throws FileNotFoundException
   * @throws IOException
   */
  protected void export(PoiExcel excel, File file) throws IOException {
    try (FileOutputStream output = new FileOutputStream(file)) {
      excel.save(output);
    }
    catch (IOException e) {
      file.delete();
      throw e;
    }
    catch (Throwable e) {
      file.delete();
      throw new RuntimeException(e);
    }
  }

  /**
   * Cria um arquivo xls para escrita. Se o nome do arquivo passado como
   * parmetro j existir no diretrio, pergunta ao usurio se quer criar um
   * outro com o nome ps-fixado com um nmero que o diferencia do original. Se
   * o usurio confirmar, o arquivo com o novo nome  criado. Seno, o arquivo
   * no  criado.
   * 
   * @param owner a janela pai
   * @param filePath o caminho para o arquivo que se deseja criar
   * @return o arquivo criado que pode ter o nome solicitado ou um nome
   *         diferente no caso do solicitado j existir
   */
  protected File createFileForWrite(Component owner, String filePath) {
    if (filePath.endsWith(".xls")) {
      filePath = filePath.substring(0, filePath.length() - 4);
    }
    File originalFile = new File(filePath);
    String name =
      StringUtils.removeAccents(originalFile.getName()).replaceAll("[^-\\w. ]",
        "_")
        + ".xls";
    originalFile = new File(originalFile.getParent(), name);
    filePath = originalFile.getAbsolutePath();
    if (!originalFile.exists()) {
      return originalFile;
    }
    File file = originalFile;
    int n = 1;
    while (file.exists()) {
      filePath = filePath.replaceAll("[\\$[0-9]]*.xls", "");
      filePath = filePath + "$" + n++ + ".xls";
      file = new File(filePath);
    }
    Object[] options = { getYesTitle(), getNoTitle() };
    int ans =
      JOptionPane.showOptionDialog(owner, getFileExistsMessage(originalFile,
        file), getFileExistsTitle(), JOptionPane.YES_NO_OPTION,
        JOptionPane.QUESTION_MESSAGE, null, options, options[0]);
    if (ans == 1) {
      return null;
    }
    return file;
  }

  /**
   * Obtm o nome do arquivo a ser utilizado.
   * 
   * @return nome do arquivo a ser utilizado.
   */
  protected String getExportFileName() {
    return "noname";
  }

  /**
   * @return ttulo
   */
  protected String getFileExistsTitle() {
    return "Arquivo j existe";
  }

  /**
   * @param originalFile
   * @param file
   * @return menssagem
   */
  protected String getFileExistsMessage(File originalFile, File file) {
    return MessageFormat.format(
      "Esta pasta j contm um arquivo com o nome {0}.\n"
        + "Ser criado um arquivo com o nome {1}.\n" + "Confirma?",
      new Object[] { originalFile.getName(), file.getName() });
  }

  /**
   * @return ttulo
   */
  protected String getYesTitle() {
    return "Sim";
  }

  /**
   * @return ttulo
   */
  protected String getNoTitle() {
    return "No";
  }

  /**
   * @param r
   */
  protected void runLaterInEDT(Runnable r) {
    SwingUtilities.invokeLater(r);
  }

  /**
   * Classe utilitria
   * 
   * @author Tecgraf
   */
  public static class StringUtils {

    /**
     * Mapa para remoo de acentos.
     */
    private static Map<Character, Character> ACCENT_REMOVAL_MAP =
      new HashMap<>();

    /*
     * inicializao do mapa
     */
    static {
      ACCENT_REMOVAL_MAP.put('', 'a');
      ACCENT_REMOVAL_MAP.put('', 'a');
      ACCENT_REMOVAL_MAP.put('', 'a');
      ACCENT_REMOVAL_MAP.put('', 'a');
      ACCENT_REMOVAL_MAP.put('', 'A');
      ACCENT_REMOVAL_MAP.put('', 'A');
      ACCENT_REMOVAL_MAP.put('', 'A');
      ACCENT_REMOVAL_MAP.put('', 'A');
      ACCENT_REMOVAL_MAP.put('', 'e');
      ACCENT_REMOVAL_MAP.put('', 'e');
      ACCENT_REMOVAL_MAP.put('', 'e');
      ACCENT_REMOVAL_MAP.put('', 'E');
      ACCENT_REMOVAL_MAP.put('', 'E');
      ACCENT_REMOVAL_MAP.put('', 'E');
      ACCENT_REMOVAL_MAP.put('', 'i');
      ACCENT_REMOVAL_MAP.put('', 'i');
      ACCENT_REMOVAL_MAP.put('', 'i');
      ACCENT_REMOVAL_MAP.put('', 'i');
      ACCENT_REMOVAL_MAP.put('', 'o');
      ACCENT_REMOVAL_MAP.put('', 'o');
      ACCENT_REMOVAL_MAP.put('', 'o');
      ACCENT_REMOVAL_MAP.put('', 'o');
      ACCENT_REMOVAL_MAP.put('', 'O');
      ACCENT_REMOVAL_MAP.put('', 'O');
      ACCENT_REMOVAL_MAP.put('', 'O');
      ACCENT_REMOVAL_MAP.put('', 'O');
      ACCENT_REMOVAL_MAP.put('', 'u');
      ACCENT_REMOVAL_MAP.put('', 'u');
      ACCENT_REMOVAL_MAP.put('', 'u');
      ACCENT_REMOVAL_MAP.put('', 'u');
      ACCENT_REMOVAL_MAP.put('', 'c');
      ACCENT_REMOVAL_MAP.put('', 'C');
    }

/**
     * Altera a string, substituindo os caracteres 
     * {@code '\', '[', ']', '>', '<', '*', * '?', ':', '/'}. 
     * Estes caracteres no so aceitos, em nomes de arquivos.
     * Alm disso a string  truncado para 30 caracteres.
     * 
     * @param str a string a ser alterada
     * @return a string alterada
     */
    public static String fixName(String str) {
      String string = str;
      string = string.replace('\\', ' ');
      string = string.replace('[', ' ');
      string = string.replace(']', ' ');
      string = string.replace('>', ' ');
      string = string.replace('<', ' ');
      string = string.replace('*', ' ');
      string = string.replace('?', ' ');
      string = string.replace(':', ' ');
      string = string.replace('/', ' ');
      while (string.indexOf("  ") >= 0) {
        string = string.replace("  ", " ");
      }
      return string.substring(0, Math.min(string.length(), 29));
    }

    /**
     * "Remove" acentos: substitui caracteres acentuados pelos caracteres
     * correspondentes sem acento (i.e. '' por 'A', '' por 'c' etc.)
     * 
     * @param str string com acentos
     * @return nova string com os acentos "removidos"
     */
    public static String removeAccents(String str) {
      char[] cStr = str.toCharArray();
      for (int i = 0; i < cStr.length; i++) {
        Character c = ACCENT_REMOVAL_MAP.get(Character.valueOf(cStr[i]));
        if (c != null) {
          cStr[i] = c.charValue();
        }
      }
      return new String(cStr);
    }

  }

  /**
   * Filtro que aceita arquivos cuja extenso seja xls.
   */
  public static class XLSFilter extends FileFilter {

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean accept(File file) {
      String filename = file.getName();
      if (file.isDirectory()) {
        return true;
      }
      return filename.endsWith(".xls");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getDescription() {
      return "*.xls";
    }
  }

}
