/*
 * Detalhes da ltima alterao:
 * 
 * $Author: leone $ $Date: 2013-05-08 17:54:13 -0300 (Wed, 08 May 2013) $
 * $Revision: 140702 $
 */
package tecgraf.javautils.gui.wizard;

import java.awt.Container;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Representa um passo de um wizard.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class Step {
  /** A operao a ser realizada ao se confirmar um passo. */
  private ConfirmOperation confirmOperation;

  /** Conjunto de ouvintes interessados em eventos do passo. */
  private Set<StepListener> listeners;

  /** O passo anterior. */
  private Step previousStep;

  /**
   * Cria um passo.
   */
  protected Step() {
    this(ConfirmOperation.NOTHING);
  }

  /**
   * Cria um passo.
   * 
   * @param confirmOperation A operao que ser realizada aps a confirmao do
   *        passo.
   */
  protected Step(ConfirmOperation confirmOperation) {
    this.listeners = new HashSet<StepListener>();
    this.setConfirmOperation(confirmOperation);
  }

  /**
   * Adiciona um listener interessado em eventos do passo.
   * 
   * @param listener O listener.
   */
  public final void addListener(StepListener listener) {
    this.listeners.add(listener);
  }

  /**
   * Remove um listener que estava interessado em eventos do passo.
   * 
   * @param listener O listener.
   */
  public final void removeListener(StepListener listener) {
    this.listeners.remove(listener);
  }

  /**
   * Define se o passo pode ser fechado.
   * 
   * @return true, caso seja possvel fechar o passo, ou false, caso contrrio.
   */
  protected abstract boolean canClose();

  /**
   * Define se o passo pode ser cancelado.
   * 
   * @return true, caso seja possvel cancelar o passo, ou false, caso
   *         contrrio.
   */
  protected abstract boolean canCancel();

  /**
   * Define se o passo pode ser confirmado.
   * 
   * @return true, caso seja possvel confirmar o passo, ou false, caso
   *         contrrio.
   */
  protected abstract boolean canConfirm();

  /**
   *  executado sempre que se entra num passo.
   * 
   * @param history O histrico.
   * 
   * @throws WizardException Caso haja algum problema com o passo.
   */
  protected abstract void enter(History history) throws WizardException;

  /**
   *  executado sempre ao passar para o prximo passo ou para o passo anterior.
   * 
   * @param history O histrico.
   */
  protected abstract void exit(History history);

  /**
   * Obtm o nome da ao de confirmao.
   * 
   * @return O nome da ao de confirmao. Pode ser nulo, caso o passo no seja
   *         de confirmao.
   */
  protected abstract String getConfirmActionName();

  /**
   * Obtm a dica da ao de confirmao.
   * 
   * @return A dica da ao de confirmao. Pode ser nula, caso o passo no seja
   *         de confirmao.
   */
  protected abstract String getConfirmActionTooltip();

  /**
   * Obtm o container do passo.
   * 
   * @return O container.
   */
  protected abstract Container getContainer();

  /**
   * Obtm as instrues para utilizao do passo. So exibidas acima do passo.
   * 
   * @return As instrues para utilizao do passo.
   */
  protected abstract String getInstruction();

  /**
   * Obtm a classe do prximo passo.
   * 
   * @param history O histrico.
   * 
   * @return A classe do prximo passo, ou null, caso no exista prximo passo.
   */
  protected abstract Class<?> getNext(History history);

  /**
   * Obtm o ttulo do passo.  exibido acima das instrues.
   * 
   * @return O ttulo.
   */
  protected abstract String getTitle();

  /**
   * Verifica se o passo est pronto. Esse mtodo  usado para verificar se 
   * possvel ir para o prximo passo.
   * 
   * @return true, caso o passo esteja pronto, ou false, caso contrrio.
   */
  protected abstract boolean isReady();

  /**
   * Obtm os dados presentes em <code>data</code> que pertencem ao passo e os
   * insere no histrico.
   * 
   * @param history O histrico.
   * @param data Os dados.
   */
  protected abstract void translate(History history, Object data);

  /**
   * Verifica se o passo conseguiu obter todos os dados necessrios no processo
   * de traduo.
   * 
   * @param history O histrico.
   * 
   * @return true, caso a traduo esteja completa, ou false, caso contrrio.
   */
  protected abstract boolean isTranslationCompleted(History history);

  /**
   * Realiza a ao de fechamento de um passo e sada do wizard.
   * 
   * @throws WizardException Caso haja algum problema com o passo.
   */
  protected void close() throws WizardException {
  }

  /**
   * Realiza a ao de cancelamento de um passo e consequentemente do wizard.
   * 
   * @throws WizardException Caso haja algum problema com o passo.
   */
  protected void cancel() throws WizardException {
  }

  /**
   * Realiza a ao de confirmao de um passo e sada do wizard.
   * 
   * @param history O histrico.
   * 
   * @return O resultado da confirmao.
   * 
   * @throws WizardException Caso haja algum problema com o passo.
   */
  protected Object confirm(History history) throws WizardException {
    return null;
  }

  /**
   * Verifica se  possvel ir para o passo anterior.
   * 
   * @return true, caso seja possvel ir para o passo anterior, ou false, caso
   *         contrrio.
   */
  protected boolean canGoPrevious() {
    return hasPrevious();
  }

  /**
   * Verifica se  possvel ir para o prximo passo.
   * <p>
   * Este mtodo  chamado para habilitar/desabilitar o boto <i>prximo</i>
   * quando o wizard entra em um passo ou quando existe alguma atualizao nos
   * listeners e <b>no</b> deve ser usado como validao de um passo. Para
   * validar se o wizard pode ou no seguir para o prximo passo, deve-se usar
   * os listeners listados no pacote <code>tecgraf.javautils.gui.wizard</code>
   * 
   * @param history O histrico.
   * @return true, caso seja possvel ir para o prximo passo, ou false, caso
   *         contrrio.
   */
  protected boolean canGoNext(History history) {
    if (!isReady()) {
      return false;
    }
    if (!hasNext(history)) {
      return false;
    }
    return true;
  }

  /**
   * Obtm a operao de confirmao.
   * 
   * @return A operao de confirmao.
   */
  protected final ConfirmOperation getConfirmOperation() {
    return this.confirmOperation;
  }

  /**
   * Obtm o passo anterior.
   * 
   * @return O passo anterior.
   */
  protected final Step getPrevious() {
    return this.previousStep;
  }

  /**
   * Verifica se o passo possui prximo passo.
   * 
   * @param history .
   * 
   * @return true, caso possua prximo passo, ou false, caso contrrio.
   */
  protected final boolean hasNext(History history) {
    return this.getNext(history) != null;
  }

  /**
   * Verifica se o passo possui passo anterior.
   * 
   * @return true, caso possua passo anterior, ou false, caso contrrio.
   */
  protected final boolean hasPrevious() {
    return this.getPrevious() != null;
  }

  /**
   * Notifica aos eventos que o passo foi alterado.
   */
  protected final void notifyChanged() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      StepListener listener = (StepListener) listenerIterator.next();
      listener.wasChanged(this);
    }
  }

  /**
   * Define a ao a ser realizada ao se confirmar o passo.
   * 
   * @param operation A operao a ser realizada.
   * 
   * @throws IllegalArgumentException Caso a operao recebida esteja nula.
   */
  protected final void setConfirmOperation(ConfirmOperation operation) {
    if (operation == null) {
      throw new IllegalArgumentException(
        "O parmetro operation no pode ser nulo.");
    }
    this.confirmOperation = operation;
  }

  /**
   * Define o passo anterior.
   * 
   * @param previousStep O passo anterior.
   */
  final void setPrevious(Step previousStep) {
    this.previousStep = previousStep;
  }

  /**
   * Representa a operao que ser realizada aps o usurio confirmar um passo.
   * 
   * @author Tecgraf/PUC-Rio
   */
  public static final class ConfirmOperation {
    /** No executa nenhuma operaao. */
    public static final ConfirmOperation NOTHING = new ConfirmOperation(0);

    /** Vai para o prximo passo. */
    public static final ConfirmOperation GO_NEXT = new ConfirmOperation(1);

    /** O identificador da operao de confirmao. */
    private int id;

    /**
     * Cria uma operao de confirmao.
     * 
     * @param id O identificador da operao.
     */
    private ConfirmOperation(int id) {
      this.id = id;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj) {
      if (obj == null) {
        return false;
      }
      if (!getClass().equals(obj.getClass())) {
        return false;
      }
      ConfirmOperation operation = (ConfirmOperation) obj;
      return this.id == operation.id;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
      return this.id;
    }
  }
}
