/*
 * $Id: Wizard.java 150399 2014-02-26 19:08:39Z oikawa $
 */
package tecgraf.javautils.gui.wizard;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.SwingConstants;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNGKeys;
import tecgraf.javautils.gui.GUIResources;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * Representa um wizard (assistente) para a obteno de dados do usurio e,
 * posteriormente, execuo de uma ao.
 * 
 * @see WizardSample
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class Wizard extends JPanel {
  /** Prefixo para o nome das chaves de traduo. */
  private static final String LNG_KEY_PREFFIX = Wizard.class.getName() + ".";

  /**
   * Define os tipos de finalizao suportados.
   * 
   * @author Tecgraf/PUC-Rio
   */
  public static class FinishingType {
    /**
     * Significa que o wizard pode ser cancelado. Um boto Cancelar poder ser
     * exibido.
     */
    public static final FinishingType CANCELABLE = new FinishingType(0);

    /**
     * Significa que o wizard pode ser fechado. Um boto Fechar poder ser
     * exibido.
     */
    public static final FinishingType CLOSEABLE = new FinishingType(1);

    /**
     * Significa que o wizard pode ser cancelado e fechado. Um boto de Cancelar
     * e/ou um botao de fechar podero ser exibidos.
     */
    public static final FinishingType CANCELABLE_AND_CLOSEABLE =
      new FinishingType(2);
    private int type;

    private FinishingType(int type) {
      this.type = type;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj) {
      if (obj == null) {
        return false;
      }
      if (!this.getClass().equals(obj.getClass())) {
        return false;
      }
      FinishingType wizardType = (FinishingType) obj;
      return this.type == wizardType.type;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
      return this.type;
    }
  }

  /**
   * Representa a ao de cancelar.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class CancelAction extends AbstractAction {
    private static final String CANCEL_SHORT_DESCRIPTION =
      "cancelShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      cancel();
    }

    CancelAction() {
      putValue(Action.NAME, LNG.get(LNGKeys.CANCEL));
      putValue(Action.SHORT_DESCRIPTION,
        LNG.get(LNG_KEY_PREFFIX + CANCEL_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa a ao de fechar.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class CloseAction extends AbstractAction {
    private static final String CLOSE_SHORT_DESCRIPTION =
      "closeShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      close();
    }

    private CloseAction() {
      putValue(Action.NAME, LNG.get(LNGKeys.CLOSE));
      putValue(Action.SHORT_DESCRIPTION,
        LNG.get(LNG_KEY_PREFFIX + CLOSE_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa a ao de confirmao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class ConfirmAction extends AbstractAction {
    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      confirm();
    }

    ConfirmAction(Step step) {
      putValue(Action.NAME, step.getConfirmActionName());
      putValue(Action.SHORT_DESCRIPTION, step.getConfirmActionTooltip());
    }
  }

  /**
   * Representa a ao de ir para o prximo passo.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class GoNextAction extends AbstractAction {
    private static final String NEXT = "next";
    private static final String NEXT_SHORT_DESCRIPTION = "nextShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      exitCurrentStep();
      goNext();
      enterCurrentStep();
      notifyGoneNext();
    }

    GoNextAction() {
      putValue(Action.SMALL_ICON, GUIResources.FORWARD_ICON);
      putValue(Action.NAME, LNG.get(LNG_KEY_PREFFIX + NEXT));
      putValue(Action.SHORT_DESCRIPTION,
        LNG.get(LNG_KEY_PREFFIX + NEXT_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa a ao de ir para o passo anterior.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class GoPreviousAction extends AbstractAction {
    private static final String PREVIOUS = "previous";
    private static final String NEXT_SHORT_DESCRIPTION =
      "previousShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      exitCurrentStep();
      goPrevious();
      enterCurrentStep();
      notifyGonePrevious();
    }

    GoPreviousAction() {
      putValue(Action.SMALL_ICON, GUIResources.BACK_ICON);
      putValue(Action.NAME, LNG.get(LNG_KEY_PREFFIX + PREVIOUS));
      putValue(Action.SHORT_DESCRIPTION,
        LNG.get(LNG_KEY_PREFFIX + NEXT_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa um ouvinte de passos.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class WizardStepListener implements StepListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void wasChanged(Step step) {
      updateProgressButtons();
    }
  }

  private static final Insets INSETS = new Insets(2, 2, 2, 2);

  /** A largura mnima do wizard. */
  private int minimum_width = FollowStepPanel.PREFERRED_WIDTH + 500;

  /** A altura mnima do wizard. */
  private int minimum_height = 250;

  /** O boto para cancelamento do wizard.  opcional. */
  private JButton cancelButton;

  /** O boto para fechamento do wizard.  opcional. */
  private JButton closeButton;

  /**
   * O boto de confirmao do wizard. S  exibido em passos onde  possvel
   * confirmar o wizard.
   */
  private JButton confirmButton;

  /** O boto que leva para o prximo passo. */
  private JButton nextButton;

  /** O boto que leva para o passo anterior. */
  private JButton previousButton;

  /** Serve para exibir o ttulo do passo corrente. */
  private JLabel stepTitleLabel;

  /** Server para exibir as instrues do passo corrente. */
  private JTextArea stepInstructionTextArea;

  /**
   * O histrico utilizado para armazenar os dados globais e os dados obtidos em
   * cada passo.
   */
  private History history;

  /** O conjunto de listeners interessados em eventos do wizard. */
  private Set listeners;

  /** O ouvinte de passos.  cadastrado no passo corrente. */
  private WizardStepListener wasChangedStepListener;

  /** Representa o painel onde ser inserido o painel de cada passo. */
  private JPanel stepPanel;

  /** Lista com os grupos de passos que compem o wizard. */
  private List<?> stepGroupList;

  /** Tipo de finalizao suportado pelo wizard. */
  private FinishingType finishingType;

  /** O passo corrente. */
  private Step currentStep;

  /** O grupo de passos corrente. */
  private StepGroup currentGroup;

  /** Indica se o tamanho dos botes deve ser equalizado */
  private boolean equalizeButtons = false;

  /**
   * Indica se os botes de finilizao, Fechar, Cancelar e Confirmar
   * permanecero visivis quando desabilitados Default: false
   */
  private boolean hideFinishButtonsWhenDisabled = false;

  /**
   * Indica se o histrico ser limpo em caso de retrocesso do wizard e mudana
   * dos dados
   */
  private boolean resetHistoryOnChange = false;

  /** Indica se  possvel saltar um conjunto de passos */
  private boolean canSkipSteps = false;

  /** Painel com as instrues dos passos do wizard */
  private FollowStepPanel followStepPanel;

  /** SplitPane usado no wizard */
  private JSplitPane splitPane;

  /**
   * Adiciona um ouvinte  lista de ouvintes do wizard.
   * 
   * @param listener O ouvinte a ser adicionado.
   */
  public void addListener(WizardListener listener) {
    this.listeners.add(listener);
  }

  /**
   * Remove um ouvinte da lista de ouvintes do wizard.
   * 
   * @param listener O ouvinte a ser removido.
   */
  public void removeListener(WizardListener listener) {
    this.listeners.remove(listener);
  }

  /**
   * Cancela o wizrd.
   */
  private void cancel() {
    try {
      this.currentStep.cancel();
      notifyCancelled();
    }
    catch (WizardException exception) {
      manageException(exception);
    }
  }

  /**
   * Fecha o wizard.
   */
  private void close() {
    notifyClosed();
  }

  /**
   * Confirma o wizard.
   */
  private void confirm() {
    try {
      this.history.setResult(this.currentStep.confirm(this.history));
      notifyConfirmed();
      if (this.currentStep.getConfirmOperation().equals(
        Step.ConfirmOperation.GO_NEXT)) {
        exitCurrentStep();
        goNext();
        enterCurrentStep();
        notifyGoneNext();
      }
    }
    catch (WizardException exception) {
      manageException(exception);
    }
  }

  /**
   * Preenche a interface grfica de usurio. So criados o painel de sequncia
   * de passos e o painel de controle.
   */
  private void fillGui() {
    followStepPanel = new FollowStepPanel(this.stepGroupList);
    addListener(followStepPanel);
    Dimension wizardDimension = this.getPreferredDimension();
    splitPane = new JSplitPane();
    JPanel controlStepPanel = createControlStepPane();
    Dimension controlStepDimension =
      new Dimension(((int) wizardDimension.getWidth())
        - FollowStepPanel.PREFERRED_WIDTH - splitPane.getDividerSize(),
        (int) wizardDimension.getHeight());
    controlStepPanel.setPreferredSize(controlStepDimension);
    splitPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
    splitPane.setLeftComponent(followStepPanel);
    splitPane.setRightComponent(controlStepPanel);
    splitPane.setOneTouchExpandable(true);
    splitPane.setContinuousLayout(true);
    splitPane.getLeftComponent().setMinimumSize(new Dimension(0, 0));
    splitPane.getRightComponent().setMinimumSize(new Dimension(0, 0));
    splitPane.setDividerLocation(followStepPanel.getPreferredSize().width);
    splitPane.setLastDividerLocation(0);
    this.setLayout(new BorderLayout());
    this.add(splitPane, BorderLayout.CENTER);
    this.setPreferredSize(wizardDimension);
  }

  /**
   * Retorna o tamanho inicial do wizard. O tamanho do wizard  calculado a
   * partir do tamanho do maior passo.
   * 
   * @return O tamanho do wizard.
   */
  private Dimension getPreferredDimension() {
    Iterator<?> stepGroupIterator = this.stepGroupList.iterator();
    int maximumStepHeight = 0;
    int maximumStepWidth = 0;
    while (stepGroupIterator.hasNext()) {
      StepGroup stepGroup = (StepGroup) stepGroupIterator.next();
      Set<?> stepSet = stepGroup.getAll();
      Iterator<?> stepSetIterator = stepSet.iterator();
      while (stepSetIterator.hasNext()) {
        Step step = (Step) stepSetIterator.next();
        int currentStepPreferredHeight =
          (int) step.getContainer().getPreferredSize().getHeight();
        if (currentStepPreferredHeight > maximumStepHeight) {
          maximumStepHeight = currentStepPreferredHeight;
        }
        int currentStepPreferredWidth =
          (int) step.getContainer().getPreferredSize().getWidth();
        if (currentStepPreferredWidth > maximumStepWidth) {
          maximumStepWidth = currentStepPreferredWidth;
        }
      }
    }
    if (maximumStepHeight < minimum_height) {
      maximumStepHeight = minimum_height;
    }
    if (maximumStepWidth < minimum_width) {
      maximumStepWidth = minimum_width;
    }

    return new Dimension(FollowStepPanel.PREFERRED_WIDTH + maximumStepWidth,
      maximumStepHeight);
  }

  /**
   * Atualiza o estado dos botes desabilitando-os ou tornando-os invisveis,
   * quando necessrio.
   */
  private void updateButtons() {
    this.previousButton.setEnabled(this.currentStep.hasPrevious()
      && this.currentStep.canGoPrevious());
    this.confirmButton
      .setVisible(this.currentStep.getConfirmActionName() != null);
    this.confirmButton.setAction(new ConfirmAction(this.currentStep));
    updateProgressButtons();
    updateFinishButtons();
  }

  /**
   * Atualiza os botes de finalizao (cancelar e fechar).
   */
  private void updateFinishButtons() {
    if (this.cancelButton != null) {
      this.cancelButton.setEnabled(this.currentStep.canCancel());
      if (hideFinishButtonsWhenDisabled) {
        this.cancelButton.setVisible(this.currentStep.canCancel());
      }
    }
    if (this.closeButton != null) {
      this.closeButton.setEnabled(this.currentStep.canClose());
      if (hideFinishButtonsWhenDisabled) {
        this.closeButton.setVisible(this.currentStep.canClose());
      }
    }
  }

  /**
   * Entra no passo corrente.
   */
  private void enterCurrentStep() {
    try {
      this.currentStep.addListener(this.wasChangedStepListener);
      this.currentStep.addListener(this.history);
      this.currentStep.enter(this.history);
      this.stepTitleLabel.setText(this.currentStep.getTitle());
      this.stepInstructionTextArea.setText(this.currentStep.getInstruction());
      this.stepInstructionTextArea.setLineWrap(true);
      this.stepInstructionTextArea.setWrapStyleWord(true);
      this.stepPanel.removeAll();
      this.stepPanel.setLayout(new BorderLayout());
      this.stepPanel.add(this.currentStep.getContainer(), BorderLayout.CENTER);
      this.updateButtons();
      this.stepPanel.revalidate();
      this.repaint();
    }
    catch (WizardException exception) {
      manageException(exception);
    }
  }

  /**
   * Sai do passo corrente.
   */
  private void exitCurrentStep() {
    this.currentStep.exit(this.history);
    this.currentStep.removeListener(this.wasChangedStepListener);
    this.currentStep.removeListener(this.history);
  }

  /**
   * Cria o painel de botes.
   * 
   * @return O painel de botes.
   */
  private JPanel createButtonPanel() {
    JPanel buttonPanel = new JPanel();
    Vector<JButton> buttons = new Vector<JButton>();
    buttonPanel.setLayout(new BorderLayout());
    JPanel browserButtonPanel = new JPanel();
    this.previousButton = new JButton(new GoPreviousAction());
    buttons.add(this.previousButton);
    browserButtonPanel.add(this.previousButton);
    this.nextButton = new JButton(new GoNextAction());
    buttons.add(this.nextButton);
    this.nextButton.setHorizontalTextPosition(SwingConstants.LEFT);
    browserButtonPanel.add(this.nextButton);
    JPanel actionButtonPanel = new JPanel();
    this.confirmButton = new JButton();
    buttons.add(this.confirmButton);
    actionButtonPanel.add(this.confirmButton);
    if (this.finishingType.equals(FinishingType.CANCELABLE)
      || this.finishingType.equals(FinishingType.CANCELABLE_AND_CLOSEABLE)) {
      this.cancelButton = new JButton(new CancelAction());
      buttons.add(this.cancelButton);
      actionButtonPanel.add(this.cancelButton);
    }
    if (this.finishingType.equals(FinishingType.CLOSEABLE)
      || this.finishingType.equals(FinishingType.CANCELABLE_AND_CLOSEABLE)) {
      this.closeButton = new JButton(new CloseAction());
      buttons.add(this.closeButton);
      actionButtonPanel.add(this.closeButton);
    }
    if (equalizeButtons) {
      equalizeButtonSizes(buttons.toArray(new JButton[buttons.size()]));
    }
    buttonPanel.add(browserButtonPanel, BorderLayout.WEST);
    buttonPanel.add(actionButtonPanel, BorderLayout.EAST);
    return buttonPanel;
  }

  /**
   * Iguala o tamanho dos botes contidos no array passado como parmetro.
   * 
   * @param buttons o array com os botes
   */
  public static void equalizeButtonSizes(JButton[] buttons) {
    Dimension maxSize = new Dimension(0, 0);
    int i;

    // Encontra maior dimenso
    for (i = 0; i < buttons.length; ++i) {
      maxSize.width =
        Math.max(maxSize.width, buttons[i].getPreferredSize().width);
      maxSize.height =
        Math.max(maxSize.height, buttons[i].getPreferredSize().height);
    }

    // Atribui novos valores para "preferred" e "maximum size", uma vez que
    // BoxLayout leva ambos os valores em considerao. */
    for (i = 0; i < buttons.length; ++i) {
      buttons[i].setPreferredSize((Dimension) maxSize.clone());
      buttons[i].setMaximumSize((Dimension) maxSize.clone());
    }
  }

  /**
   * Cria o painel de controle dos passos. Este painel consiste no painel de
   * cabealho, no painel do passo eno painel de botes.
   * 
   * @return O painel de controle dos passos.
   */
  private JPanel createControlStepPane() {
    JPanel controlStepPanel = new JPanel();
    controlStepPanel.setLayout(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = INSETS;
    JPanel headerPanel = createHeaderPanel();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 1.0;
    constraints.weighty = 0.0;
    controlStepPanel.add(headerPanel, constraints);
    this.stepPanel = new JPanel();
    constraints.fill = GridBagConstraints.BOTH;
    constraints.gridx = 0;
    constraints.gridy = 1;
    constraints.weightx = 1.0;
    constraints.weighty = 1.0;
    controlStepPanel.add(this.stepPanel, constraints);
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridx = 0;
    constraints.gridy = 2;
    constraints.weightx = 1.0;
    constraints.weighty = 0.0;
    controlStepPanel
      .add(new JSeparator(SwingConstants.HORIZONTAL), constraints);
    JPanel buttonPanel = createButtonPanel();
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridx = 0;
    constraints.gridy = 3;
    constraints.weightx = 1.0;
    constraints.weighty = 0.0;
    controlStepPanel.add(buttonPanel, constraints);
    return controlStepPanel;
  }

  /**
   * Cria o painel com o cabealho, onde so exibidos o ttulo e as instrues
   * do passo corrente.
   * 
   * @return O painel com o cabealho.
   */
  private JPanel createHeaderPanel() {
    JPanel headerPanel = new JPanel();
    headerPanel.setLayout(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = INSETS;
    constraints.anchor = GridBagConstraints.NORTHWEST;
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.gridx = 0;
    constraints.weightx = 1.0;
    constraints.weighty = 0.0;
    this.stepTitleLabel = new JLabel();
    constraints.gridy = 0;
    headerPanel.add(this.stepTitleLabel, constraints);
    JSeparator separator = new JSeparator(SwingConstants.HORIZONTAL);
    constraints.gridy = 1;
    headerPanel.add(separator, constraints);
    this.stepInstructionTextArea = new JTextArea();
    this.stepInstructionTextArea.setBackground(this.getBackground());
    this.stepInstructionTextArea.setEditable(false);
    constraints.gridy = 2;
    headerPanel.add(this.stepInstructionTextArea, constraints);
    return headerPanel;
  }

  /**
   * Configura a fonte utilizada na caixa de texto da instruo do passo
   * 
   * @param font - nova fonte a ser utilizada na caixa de texto de instrues
   */
  public void setStepInstructionTextAreaFont(Font font) {
    this.stepInstructionTextArea.setFont(font);
  }

  /**
   * Vai para o prximo passo.
   * 
   * @throws IllegalStateException Caso o passo no seja encontrado no grupo
   *         corrente e no prximo grupo.
   */
  private void goNext() {
    /*
     * obtemos a classe do prximo passo
     */
    Class<?> stepClass = this.currentStep.getNext(this.history);
    if (stepClass.equals(this.currentStep.getClass())) {
      /*
       * Caso o prximo passo seja igual ao passo corrente, no  necessrio
       * verificar o stepGroup ou atualizar valores.
       * 
       * OBS.: o fluxo de processamento prosseguir como se tivesse havido
       * realmente uma mudana de passo, ou seja, o mtodo enter() do passo
       * corrente ser novamente executado.
       */
      return;
    }
    Step nextStep;
    StepGroup nextStepGroup;
    if (!this.canSkipSteps) {
      /*
       * se no  possvel pular passos, o prximo passo s pode ser do mesmo
       * grupo que o passo corrente
       */
      nextStepGroup = this.currentGroup;
      nextStep = nextStepGroup.get(stepClass);
      if (nextStep == null) {
        int groupIndex = this.stepGroupList.indexOf(nextStepGroup);
        if (groupIndex == this.stepGroupList.size() - 1) {
          throw new IllegalStateException(
            MessageFormat
              .format(
                "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
                new Object[] { this.currentStep.getClass(), stepClass }));
        }
        nextStepGroup = (StepGroup) this.stepGroupList.get(groupIndex + 1);
        nextStep = nextStepGroup.get(stepClass);
        if (nextStep == null) {
          throw new IllegalStateException(
            MessageFormat
              .format(
                "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
                new Object[] { this.currentStep.getClass(), stepClass }));
        }
      }
    }
    else {
      /*
       * se podemos pular passos,  necessrio primeiro descobrir a qual grupo
       * pertence o prximo passo
       */
      nextStep = null;
      nextStepGroup = null;

      for (Object o : stepGroupList) {
        nextStepGroup = (StepGroup) o;
        nextStep = nextStepGroup.get(stepClass);
        if (nextStep != null) {
          break;
        }
      }

      if (nextStep == null) {
        throw new IllegalStateException(
          MessageFormat
            .format(
              "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
              new Object[] { this.currentStep.getClass(), stepClass }));
      }
    }
    nextStep.setPrevious(currentStep);
    currentStep = nextStep;
    currentGroup = nextStepGroup;
  }

  /**
   * Recupera um passo
   * 
   * @param stepClass A classe do passo
   * @return A instncia do passo, null se o passo no existir
   */
  protected Step getStep(Class<?> stepClass) {
    for (Object o : stepGroupList) {
      StepGroup stepGroup = (StepGroup) o;
      Step step = stepGroup.get(stepClass);
      if (step != null) {
        return step;
      }
    }
    return null;
  }

  /**
   * Vai para o passo anterior.
   * 
   * @throws IllegalStateException Caso o passo no seja encontrado no grupo
   *         corrente e no grupo anterior.
   */
  private void goPrevious() {
    if (!this.canSkipSteps) {
      Step previousStep = this.currentStep.getPrevious();
      StepGroup stepGroup = this.currentGroup;
      if (!stepGroup.contains(previousStep)) {
        int groupIndex = this.stepGroupList.indexOf(stepGroup);
        if (groupIndex == 0) {
          throw new IllegalStateException(
            MessageFormat
              .format(
                "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
                new Object[] { this.currentStep.getClass(),
                    previousStep.getClass() }));
        }
        stepGroup = (StepGroup) this.stepGroupList.get(groupIndex - 1);
        if (!stepGroup.contains(previousStep)) {
          throw new IllegalStateException(
            MessageFormat
              .format(
                "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
                new Object[] { this.currentStep.getClass(),
                    previousStep.getClass() }));
        }
      }
      this.currentStep = previousStep;
      this.currentGroup = stepGroup;
    }
    else {
      Step nextStep = null;
      StepGroup nextStepGroup = null;
      Class<?> stepClass = this.currentStep.getPrevious().getClass();

      for (Object o : stepGroupList) {
        nextStepGroup = (StepGroup) o;
        nextStep = nextStepGroup.get(stepClass);
        if (nextStep != null) {
          break;
        }
      }

      if (nextStep == null) {
        throw new IllegalStateException(
          MessageFormat
            .format(
              "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
              new Object[] { this.currentStep.getClass(), stepClass }));
      }

      this.currentStep = nextStep;
      this.currentGroup = nextStepGroup;
    }
  }

  /**
   * Faz o tratamento de uma exceo ocorrida. O wizard  fechado aps
   * 
   * @param exception A exceo ocorrida.
   */
  private void manageException(WizardException exception) {
    notifyException(exception);
    close();
  }

  /**
   * Gera uma notificao de que o wizard foi cancelado.
   */
  private void notifyCancelled() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      final WizardListener listener = (WizardListener) listenerIterator.next();
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          listener.wasCancelled(currentStep);
        }
      });
    }
  }

  /**
   * Gera uma notificao de que o wizard foi fechado.
   */
  private void notifyClosed() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      final WizardListener listener = (WizardListener) listenerIterator.next();
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          listener.wasClosed(currentStep);
        }
      });
    }
  }

  /**
   * Gera uma notificao de que o wizard foi confirmado.
   */
  private void notifyConfirmed() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      WizardListener listener = (WizardListener) listenerIterator.next();
      listener.wasConfirmed(this.currentStep, this.history.getResult());
    }
  }

  /**
   * Gera uma notificao de que uma exceo foi levantada.
   * 
   * @param exception A exceo que foi levantada.
   */
  private void notifyException(WizardException exception) {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      WizardListener listener = (WizardListener) listenerIterator.next();
      listener.wasHappenedException(this.currentStep, exception);
    }
  }

  /**
   * Gera uma notificao de que o wizard foi para o prximo passo.
   */
  private void notifyGoneNext() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      WizardListener listener = (WizardListener) listenerIterator.next();
      listener.wasGoneNext(this.currentStep, this.history);
    }
  }

  /**
   * Gera uma notificao de que o wizard voltou ao passo anterior.
   */
  private void notifyGonePrevious() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      WizardListener listener = (WizardListener) listenerIterator.next();
      listener.wasGonePrevious(this.currentStep, this.history);
    }
  }

  /**
   * Atualiza os botes de progresso. Os botes de progresso so o prximo e o
   * de confirmao.
   */
  private void updateProgressButtons() {
    this.confirmButton.setEnabled(this.currentStep.canConfirm());
    this.nextButton.setEnabled(this.currentStep.hasNext(this.history)
      && this.currentStep.canGoNext(this.history));
  }

  /**
   * <p>
   * Faz uma iterao por todos os passos do wizard traduzindo-os.
   * </p>
   * 
   * <p>
   * O processo de traduo  efetuado pelo passo a partir dos dados existentes
   * em data.
   * </p>
   * 
   * <p>
   * Ao terminar o mtodo, o passo corrente ser o primeiro passo cuja traduo
   * no tenha sido concluda.
   * </p>
   * 
   * @param data Os dados que sero utilizados pelo passos para se traduzirem.
   */
  private void translateSteps(Object data) {
    Iterator<?> stepGroupIterator = this.stepGroupList.iterator();
    while (stepGroupIterator.hasNext()) {
      StepGroup stepGroup = (StepGroup) stepGroupIterator.next();
      Set<?> stepSet = stepGroup.getAll();
      Iterator<?> stepIterator = stepSet.iterator();
      while (stepIterator.hasNext()) {
        Step step = (Step) stepIterator.next();
        step.translate(this.history, data);
      }
    }
    while (this.currentStep.isTranslationCompleted(this.history)) {
      this.goNext();
      this.notifyGoneNext();
    }
  }

  /**
   * Cria o wizard.
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList Uma lista com os grupos de passos que sero exibidos
   *        pelo wizard.
   * @param globalData Dados globais do wizard que podero ser acessados por
   *        qualquer passo.
   * @param finishingType O tipo de fechamento do wizard.
   */
  private void createWizard(Class<?> firstStepClass, List<?> stepGroupList,
    Map<?, ?> globalData, FinishingType finishingType) {
    this.wasChangedStepListener = new WizardStepListener();
    this.listeners = new HashSet();
    this.history = new History(this, this.resetHistoryOnChange);
    this.history.addGlobal(globalData);
    this.addListener(this.history);
    this.stepGroupList = Collections.unmodifiableList(stepGroupList);
    this.currentGroup = (StepGroup) this.stepGroupList.get(0);
    this.currentStep = this.currentGroup.get(firstStepClass);
    this.finishingType = finishingType;
    this.fillGui();
  }

  /**
   * Cria um wizard que pode ser fechado e cancelado.
   * 
   * @param firstStepClass A classe do primeiro passo do wizard.
   * @param stepGroupList A lista de grupos de passos.
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList) {
    this(firstStepClass, stepGroupList, Collections.EMPTY_MAP,
      FinishingType.CANCELABLE_AND_CLOSEABLE, false);
  }

  /**
   * Cria um wizard.
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    FinishingType wizardType) {
    this(firstStepClass, stepGroupList, Collections.EMPTY_MAP, wizardType,
      false);
  }

  /**
   * Cria um wizard.
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   * @param equalizeButtons Indica se o tamanho dos botes do wizard deve ser
   *        equalizado
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    FinishingType wizardType, boolean equalizeButtons) {
    this(firstStepClass, stepGroupList, Collections.EMPTY_MAP, wizardType,
      equalizeButtons);
  }

  /**
   * Cria um wizard.
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   * @param equalizeButtons Indica se o tamanho dos botes do wizard deve ser
   *        equalizado
   * @param hideFinishButtonsWhenDisabled Indica se os botes Fechar, Cancelar e
   *        Confirmar permanecero visiveis mesmo quando desabilitados
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    FinishingType wizardType, boolean equalizeButtons,
    boolean hideFinishButtonsWhenDisabled) {
    this(firstStepClass, stepGroupList, Collections.EMPTY_MAP, wizardType,
      equalizeButtons, hideFinishButtonsWhenDisabled);
  }

  /**
   * Cria um wizard que pode ser fechado e cancelado.
   * 
   * @param firstStepClass A classe do primeiro passo do wizard.
   * @param stepGroupList A lista de grupos de passos.
   * @param globalData Dados globais a serem inseridos no histrico do wizard.
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    Map<?, ?> globalData) {
    this(firstStepClass, stepGroupList, globalData,
      FinishingType.CANCELABLE_AND_CLOSEABLE, false);
  }

  /**
   * <p>
   * Cria um wizard.
   * </p>
   * 
   * <p>
   * Com os dados recebidos,  possvel iniciar o wizard num passo posterior ao
   * primeiro passo.
   * </p>
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param stepData Os dados a serem traduzidos pelos passos.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   * @param resetHistoryOnchange true para limpar o histrico caso o passo seja
   *        alterado e tenha havido um retrocesso
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    Object stepData, FinishingType wizardType, boolean resetHistoryOnchange) {
    this(firstStepClass, stepGroupList, Collections.EMPTY_MAP, stepData,
      wizardType, false, resetHistoryOnchange);
  }

  /**
   * <p>
   * Cria um wizard.
   * </p>
   * 
   * <p>
   * Com os dados recebidos,  possvel iniciar o wizard num passo posterior ao
   * primeiro passo.
   * </p>
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param stepData Os dados a serem traduzidos pelos passos.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    Object stepData, FinishingType wizardType) {
    this(firstStepClass, stepGroupList, Collections.EMPTY_MAP, stepData,
      wizardType, false, false);
  }

  /**
   * <p>
   * Cria um wizard que pode ser fechado e cancelado.
   * </p>
   * 
   * <p>
   * Com os dados recebidos,  possvel iniciar o wizard num passo posterior ao
   * primeiro passo.
   * </p>
   * 
   * @param firstStepClass A classe do primeiro passo do wizard.
   * @param stepGroupList A lista de grupos de passos.
   * @param requisites Dados globais a serem inseridos no histrico do wizard.
   * @param stepData Os dados a serem traduzidos pelos passos.
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    Map<?, ?> requisites, Object stepData) {
    this(firstStepClass, stepGroupList, requisites, stepData,
      FinishingType.CANCELABLE_AND_CLOSEABLE, false, false);
  }

  /**
   * <p>
   * Cria um wizard que pode ser fechado e cancelado.
   * </p>
   * 
   * <p>
   * Com os dados recebidos,  possvel iniciar o wizard num passo posterior ao
   * primeiro passo.
   * </p>
   * 
   * @param firstStepClass A classe do primeiro passo do wizard.
   * @param stepGroupList A lista de grupos de passos.
   * @param stepData Os dados a serem traduzidos pelos passos.
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList, Object stepData) {
    this(firstStepClass, stepGroupList, Collections.EMPTY_MAP, stepData,
      FinishingType.CANCELABLE_AND_CLOSEABLE, false, false);
  }

  /**
   * <p>
   * Cria um wizard.
   * </p>
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param globalData Dados globais a serem inseridos no histrico do wizard.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   * @param equalizeButtons Indica se o tamanho dos botes do wizard deve ser
   *        equalizado
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    Map<?, ?> globalData, FinishingType wizardType, boolean equalizeButtons) {
    this(firstStepClass, stepGroupList, globalData, wizardType,
      equalizeButtons, false);
  }

  /**
   * <p>
   * Cria um wizard.
   * </p>
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param globalData Dados globais a serem inseridos no histrico do wizard.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   * @param equalizeButtons Indica se o tamanho dos botes do wizard deve ser
   *        equalizado
   * @param hideFinishButtonsWhenDisabled Indica se os botes Fechar, Cancelar e
   *        Confirmar permanecero visiveis mesmo quando desabilitados
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    Map<?, ?> globalData, FinishingType wizardType, boolean equalizeButtons,
    boolean hideFinishButtonsWhenDisabled) {
    this.equalizeButtons = equalizeButtons;
    this.hideFinishButtonsWhenDisabled = hideFinishButtonsWhenDisabled;
    this.createWizard(firstStepClass, stepGroupList, globalData, wizardType);
    this.enterCurrentStep();
  }

  /**
   * <p>
   * Cria um wizard.
   * </p>
   * 
   * <p>
   * Com os dados recebidos,  possvel iniciar o wizard num passo posterior ao
   * primeiro passo.
   * </p>
   * 
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   * @param globalData Dados globais a serem inseridos no histrico do wizard.
   * @param stepData Os dados a serem traduzidos pelos passos.
   * @param wizardType O tipo de finalizao suportada pelo wizard.
   * @param equalizeButtons Indica se o tamanho dos botes do wizard deve ser
   *        equalizado
   */
  public Wizard(Class<?> firstStepClass, List<?> stepGroupList,
    Map<?, ?> globalData, Object stepData, FinishingType wizardType,
    boolean equalizeButtons, boolean resetHistoryOnChange) {
    this.resetHistoryOnChange = resetHistoryOnChange;
    this.equalizeButtons = equalizeButtons;
    this.createWizard(firstStepClass, stepGroupList, globalData, wizardType);
    this.translateSteps(stepData);
    this.enterCurrentStep();
  }

  /**
   * Habilita/desabilita o salto de passos.
   * 
   * @param canSkipSteps
   */
  public void setCanSkipSteps(boolean canSkipSteps) {
    this.canSkipSteps = canSkipSteps;
  }

  /**
   * Atribui uma nova altura mnima para o wizard
   * 
   * @param height a nova altura
   */
  public void setMinHeight(int height) {
    minimum_height = height;
    this.setPreferredSize(this.getPreferredDimension());
  }

  /**
   * Atribui uma nova largura mnima para o wizard
   * 
   * @param width a nova largura
   */
  public void setMinWidth(int width) {
    minimum_width = width;
    this.setPreferredSize(this.getPreferredDimension());
  }

  /**
   * @return true se o painel de instrues estiver recolhido, false caso
   *         contrrio
   */
  public boolean isFollowStepPanelHidden() {
    return splitPane.getLeftComponent().getSize().width <= 0;
  }

  /**
   * Recolhe o painel de instrues
   */
  public void hideFollowStepPanel() {
    this.splitPane.setDividerLocation(0);
    this.splitPane.setLastDividerLocation(this.splitPane.getLeftComponent()
      .getPreferredSize().width);
  }

  /**
   * Exibe o painel de instrues
   */
  public void showFollowStepPanel() {
    this.splitPane.setDividerLocation(this.splitPane.getLeftComponent()
      .getPreferredSize().width);
  }
}
