package tecgraf.javautils.gui;

import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.Locale;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

/**
 * Modelo para campos de texto.<BR>
 * Permite apenas caracteres numricos e o separador do decimal de acordo com o
 * locale.
 * 
 */
public class NumberDocument extends PlainDocument {

  private String separator;
  private String validateRegex;
  private String insertRegex;

  /**
   * Construtor<BR>
   * Aceita valores decimais
   */
  public NumberDocument() {
    this(Locale.getDefault(), true);
  }

  /**
   * Construtor
   * 
   * @param acceptDecimal indica se aceita valores decimais
   */
  public NumberDocument(boolean acceptDecimal) {
    this(Locale.getDefault(), acceptDecimal);
  }

  /**
   * Construtor
   * 
   * @param locale {@link Locale} para definir o separador do decimal
   */
  public NumberDocument(Locale locale) {
    this(locale, true);
  }

  /**
   * Construtor
   * 
   * @param locale {@link Locale} para definir o separador do decimal
   * @param acceptDecimal indica se aceita valores decimais
   */
  public NumberDocument(Locale locale, boolean acceptDecimal) {
    if (acceptDecimal) {
      DecimalFormatSymbols dfs = DecimalFormatSymbols.getInstance(locale);
      separator = Character.toString(dfs.getDecimalSeparator());
      validateRegex = "([-]{1,1})?[0-9]{1,}([" + separator + "][0-9]{1,})?";
      insertRegex = "([-]{1,1})?([0-9]{1,})?([" + separator + "])?([0-9]{1,})?";
    }
    else {
      validateRegex = "([-]{1,1})?[0-9]{1,}";
      insertRegex = "([-]{1,1})?([0-9]{1,})?";
    }
  }

  /**
   * Obtm o valor do modelo.
   * 
   * @return {@link Double} ou <CODE>NULL</CODE> caso o campo esteja em branco
   *         ou preenchido errado
   */
  public Double getValue() {
    try {
      String text = getText();
      if (text.matches(validateRegex)) {
        NumberFormat n = NumberFormat.getInstance(Locale.getDefault());
        try {
          return n.parse(text).doubleValue();
        }
        catch (ParseException e) {
          e.printStackTrace();
        }
      }
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Seta um valor
   * 
   * @param value valor
   */
  public void setValue(double value) {
    NumberFormat n = NumberFormat.getInstance(Locale.getDefault());
    String text = n.format(value);
    try {
      remove(0, getLength());
      insertString(0, text, null);
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
  }

  /**
   * Obtm o texto
   * 
   * @return {@link String}
   * @throws BadLocationException
   */
  private String getText() throws BadLocationException {
    return getText(0, getLength());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void insertString(int offs, String str, AttributeSet a)
    throws BadLocationException {
    String text = getText();
    String newText =
      text.substring(0, offs) + str + text.substring(offs, text.length());
    if (newText.matches(insertRegex))
      super.insertString(offs, str, a);
  }

}
