package tecgraf.javautils.gui.field;

import javax.swing.text.BadLocationException;

/**
 * Campo para Latitude ou Longitude. Escolhe Latitude ou Longitude, exibir
 * efeito visual em caso de valor errado, representar hemisfrio via sinal de
 * negativo e casas decimais para os segundos
 */
public class LatLongDocument extends RegexDocument {
  public static enum Orientation {
    LATITUDE,
    LONGITUDE;
  };

  /** Orientao: logintude ou latitude */
  private Orientation orientation;
  private String maxSizeDegree;
  private int decimal;
  private boolean hemisphereEnabled;

  /**
   * Default: exibir erro em validao desligado ; quantidade de casas decimais
   * para segundos  2 ; representando hemisfrio via sinal de negativo
   * 
   * @param orientation
   */
  public LatLongDocument(Orientation orientation) {
    this(orientation, 2, true);
  }

  /**
   * Default: exibir erro em validao  false ; representando hemisfrio via
   * sinal de negativo
   * 
   * @param orientation
   * @param decimal quantidade de casas decimais para segundos
   */
  public LatLongDocument(Orientation orientation, int decimal) {
    this(orientation, decimal, true);
  }

  /**
   * Default: quantidade de casas decimais para segundos  2
   * 
   * @param orientation
   * @param showValidation exibir erro em validao
   * @param hemisphereEnabled representar hemisfrio via sinal de negativo
   */
  public LatLongDocument(Orientation orientation, boolean hemisphereEnabled) {
    this(orientation, 2, hemisphereEnabled);
  }

  /**
   * Construtor
   * 
   * @param orientation {@link Orientation}
   * @param decimal quantidade de casas decimais para segundos
   * @param hemisphereEnabled representar hemisfrio via sinal de negativo
   */
  public LatLongDocument(Orientation orientation, int decimal,
    boolean hemisphereEnabled) {
    super(getRegex(hemisphereEnabled, orientation, decimal), buildAutoComplete(
      hemisphereEnabled, orientation, decimal, true));
    this.hemisphereEnabled = hemisphereEnabled;
    this.orientation = orientation;
    if (orientation == Orientation.LONGITUDE) {
      maxSizeDegree = "3";
    }
    else {
      maxSizeDegree = "2";
    }
    setDecimal(decimal);
  }
  
  protected void rebuildCompleText() {
    setCompleteText(buildAutoComplete(hemisphereEnabled, orientation, decimal, true));
  }

  private static CompleteText buildAutoComplete(
    final boolean hemisphereEnabled, final Orientation orientation,
    int decimal, boolean enabled) {
    if (enabled) {
      final int newDecimal = normalizeDecimal(decimal);
      return new CompleteText() {
        public String buildCompletedText(String text, String newStr) {
          String hem = "";
          if (hemisphereEnabled) {
            hem = "\\-?";
          }
          String maxSizeDegree = "";
          if (orientation == Orientation.LONGITUDE) {
            maxSizeDegree = "3";
          }
          else {
            maxSizeDegree = "2";
          }
          String regex1 = hem + "[0-9]{" + maxSizeDegree + "}";
          String regex2 = hem + "[0-9]{1," + maxSizeDegree + "}:[0-9]{2}";
          String regex3 =
            hem + "[0-9]{1," + maxSizeDegree + "}:[0-9]{1,2}:[0-9]{2}";
          String complt1 =
            "[0-9]{1,2}(:([0-9]{1,2}(\\.([0-9]{1," + newDecimal + "})?)?)?)?";
          String complt2 = "[0-9]{1,2}(\\.([0-9]{1," + newDecimal + "})?)?";
          String complt3 = "([0-9]{1," + newDecimal + "})?";
          String regex1Longitude = hem + "[0-9]{2}";

          /*
           * caso especial para longitude. no existe degree 222 por exemplo.
           * Caso j tenha sido digitado 22, o prximo 2 ser minutos
           */
          if (orientation == Orientation.LONGITUDE
            && text.matches(regex1Longitude) && newStr.matches(complt1)) {
            String s = newStr.split(":")[0];
            int degree = Math.abs(Integer.parseInt(text + s));
            if (degree > 180) {
              return ":" + newStr;
            }
          }
          if (text.matches(regex1) && newStr.matches(complt1)) {
            return ":" + newStr;
          }
          else if (text.matches(regex2) && newStr.matches(complt2)) {
            return ":" + newStr;
          }
          else if (text.matches(regex3) && newStr.matches(complt3)) {
            return "." + newStr;
          }
          return newStr;
        }
      };
    }
    else {
      return null;
    }
  }

  /**
   * Ativa o auto complemento de texto. Complementa com : aps o grau e minutos
   * e com . para os decimais dos segundos
   * 
   * @param complete
   */
  public void setAutoComplete(boolean complete) {
    setCompleteText(buildAutoComplete(hemisphereEnabled, orientation, decimal,
      true));
  }

  public boolean isValid() {
    String hem = "";
    if (hemisphereEnabled) {
      hem = "\\-?";
    }
    String validRegex =
      hem + "[0-9]{1," + maxSizeDegree + "}:[0-9]{1,2}:[0-9]{1,2}(\\.[0-9]{1,"
        + decimal + "})?";
    String text = getText();
    if (text.matches(validRegex)) {
      String[] str = text.split(":");
      int degree = Math.abs(Integer.parseInt(str[0]));
      int minute = Math.abs(Integer.parseInt(str[1]));
      float second = Math.abs(Float.parseFloat(str[2]));
      if (orientation == Orientation.LONGITUDE && degree >= 0 && degree <= 180
        && minute <= 60 && second <= 60F) {
        return true;
      }
      else if (orientation == Orientation.LATITUDE && degree >= 0
        && degree <= 90 && minute <= 60 && second <= 60F) {
        return true;
      }
    }
    return false;
  }

  private static String getRegex(boolean hemisphereEnabled,
    Orientation orientation, int decimal) {
    decimal = normalizeDecimal(decimal);
    String hem = "";
    if (hemisphereEnabled) {
      hem = "(\\-)?";
    }
    String maxSizeDegree = "";
    if (orientation == Orientation.LONGITUDE) {
      maxSizeDegree = "3";
    }
    else {
      maxSizeDegree = "2";
    }
    return hem + "([0-9]{1," + maxSizeDegree
      + "})?(:)?([0-9]{1,2})?(:)?([0-9]{1,2})?(\\.)?([0-9]{1," + decimal
      + "})?";
  }

  /**
   * Seta a quantidade de casas decimais para os segundos, o mnimo  1 casa
   * decimal
   * 
   * @param decimal
   */
  public void setDecimal(int decimal) {
    this.decimal = normalizeDecimal(decimal);
  }

  private static int normalizeDecimal(int decimal) {
    if (decimal >= 1) {
      return decimal;
    }
    else {
      return 1;
    }
  }

  /**
   * Retorna o grau
   * 
   * @return null caso ainda no tenha sido digitado ou no seja vlido
   */
  public Integer getDegree() {
    if (isValid()) {
      String[] str = getText().split(":");
      Integer degree = Integer.parseInt(str[0]);
      return degree;
    }
    return null;
  }

  /**
   * Retorna os minutos
   * 
   * @return null caso ainda no tenha sido digitado ou no seja vlido
   */
  public Integer getMinute() {
    if (isValid()) {
      String[] str = getText().split(":");
      Integer minute = Integer.parseInt(str[1]);
      return minute;
    }
    return null;
  }

  /**
   * Retorna os segundos.
   * 
   * @return null caso ainda no tenha sido digitado ou no seja vlido
   */
  public Float getSecond() {
    if (isValid()) {
      String[] str = getText().split(":");
      Float second = Float.parseFloat(str[2]);
      return second;
    }
    return null;
  }

  private String getText() {
    try {
      return getText(0, getLength());
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
    return null;
  }

  public boolean isHemisphereEnabled() {
    return hemisphereEnabled;
  }

  @Override
  protected Object getValue() {
    return getText();
  }
}
