package tecgraf.javautils.gui.field;

import javax.swing.text.Document;

import tecgraf.javautils.gui.field.LatLongDocument.Orientation;

/**
 * Campo para Latitude ou Longitude. Escolhe Latitude ou Longitude, exibir
 * efeito visual em caso de valor errado, representar hemisfrio via sinal de
 * negativo e casas decimais para os segundos
 */
public class LatLongField extends AbstractRegexField {

  public static final Orientation LATITUDE = Orientation.LATITUDE;
  public static final Orientation LONGITUDE = Orientation.LONGITUDE;

  /**
   * Default: exibir erro em validao desligado ; quantidade de casas decimais
   * para segundos  2 ; representando hemisfrio via sinal de negativo
   * 
   * @param orientation
   */
  public LatLongField(Orientation orientation) {
    this(orientation, 2, false, true);
  }

  /**
   * Default: exibir erro em validao  false ; representando hemisfrio via
   * sinal de negativo
   * 
   * @param orientation
   * @param decimal quantidade de casas decimais para segundos
   */
  public LatLongField(Orientation orientation, int decimal) {
    this(orientation, decimal, false);
  }

  /**
   * Default: quantidade de casas decimais para segundos  2 ; representando
   * hemisfrio via sinal de negativo
   * 
   * @param orientation
   * @param showValidation exibir erro em validao
   */
  public LatLongField(Orientation orientation, boolean showValidation) {
    this(orientation, 2, showValidation);
  }

  /**
   * Default: quantidade de casas decimais para segundos  2
   * 
   * @param orientation
   * @param showValidation exibir erro em validao
   * @param hemisphereEnabled representar hemisfrio via sinal de negativo
   */
  public LatLongField(Orientation orientation, boolean showValidation,
    boolean hemisphereEnabled) {
    this(orientation, 2, showValidation, hemisphereEnabled);
  }

  /**
   * Default: representando hemisfrio via sinal de negativo
   * 
   * @param orientation
   * @param decimal quantidade de casas decimais para segundos
   * @param showValidation exibir erro em validao
   */
  public LatLongField(Orientation orientation, int decimal,
    boolean showValidation) {
    this(orientation, decimal, showValidation, true);
  }

  /**
   * Construtor
   * 
   * @param orientation
   * @param decimal quantidade de casas decimais para segundos
   * @param showValidation exibir erro em validao
   */
  public LatLongField(Orientation orientation, int decimal,
    boolean showValidation, boolean hemisphereEnabled) {
    super(showValidation);
    setDocument(new LatLongDocument(orientation, decimal, hemisphereEnabled));
    setDecimal(decimal);
  }

  /**
   * Ativa o auto complemento de texto. Complementa com : aps o grau e minutos
   * e com . para os decimais dos segundos
   * 
   * @param complete
   */
  public void setAutoComplete(boolean complete) {
    if (complete) {
      getLatLongDocument().rebuildCompleText();
    }
    else {
      getLatLongDocument().setCompleteText(null);
    }
  }

  @Override
  public boolean isValidValue() {
    return getLatLongDocument().isValid();
  }

  /**
   * Seta a quantidade de casas decimais para os segundos, o mnimo  1 casa
   * decimal
   * 
   * @param decimal
   */
  public void setDecimal(int decimal) {
    getLatLongDocument().setDecimal(decimal);
  }

  /**
   * Retorna o grau
   * 
   * @return null caso ainda no tenha sido digitado ou no seja vlido
   */
  public Integer getDegree() {
    return getLatLongDocument().getDegree();
  }

  /**
   * Retorna os minutos
   * 
   * @return null caso ainda no tenha sido digitado ou no seja vlido
   */
  public Integer getMinute() {
    return getLatLongDocument().getMinute();
  }

  /**
   * Retorna os segundos.
   * 
   * @return null caso ainda no tenha sido digitado ou no seja vlido
   */
  public Float getSecond() {
    return getLatLongDocument().getSecond();
  }

  @Override
  public void setDocument(Document doc) {
    if (doc != null && doc instanceof LatLongDocument) {
      super.setDocument(doc);
    }
  }

  /**
   * Obtm o modelo
   * 
   * @return {@link LatLongDocument}
   */
  private LatLongDocument getLatLongDocument() {
    return (LatLongDocument) getDocument();
  }

  public boolean isHemisphereEnabled() {
    return getLatLongDocument().isHemisphereEnabled();
  }
}
