/*
 * $Id: FooterComparatorWrapper.java 85678 2008-11-14 18:58:47Z letnog $
 */
package tecgraf.javautils.gui.table;

import java.util.Comparator;
import java.util.List;

import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.table.TableModel;

/**
 * Classe que "decora" (<i>Wrapper</i>) um comparador de colunas de uma
 * tabela, garantindo que determinadas linhas, identificadas por classes
 * "marcadoras", mantenham uma posio fixa no fim da tabela. Esta
 * funcionalidade  til para tabelas com totalizao dos valores de
 * determinadas colunas.
 * 
 * @param <T> Tipo de objetos que podem ser comparados por este comparador.
 * 
 * @author Tecgraf
 */
public class FooterComparatorWrapper<T> implements Comparator<T>,
  SortableTableListener {
  /** Comparador original, decorado por este objeto. */
  private Comparator<T> actual;
  /** Tabela ordenvel que utiliza este comparador. */
  private SortableTable table;
  /** Sentido atual de ordenao da coluna sendo comparada. */
  private SortOrder sortOrder;

  /**
   * Cria o decorador a partir de um comparador comum.
   * 
   * @param table tabela ordenvel que utiliza este comparador.
   * @param actual comparador sendo decorado.
   */
  public FooterComparatorWrapper(SortableTable table, Comparator<T> actual) {
    this.actual = actual;
    this.table = table;
  }

  /**
   * Retorna o sentido atual de ordenao da coluna sendo comparada.
   * 
   * @return sentido atual de ordenao.
   */
  private SortOrder getSortOrder() {
    if (sortOrder != null) {
      return sortOrder;
    }
    RowSorter<? extends TableModel> sorter = table.getRowSorter();
    if (sorter != null) {
      List<? extends RowSorter.SortKey> sortKeys = sorter.getSortKeys();
      if (sortKeys != null && sortKeys.size() > 0) {
        return sortKeys.get(0).getSortOrder();
      }
    }
    return null;
  }

  /**
   * Compara dois objetos para determinar sua precedncia relativa. Se um dos
   * objetos estiver "marcado" como {@link FooterCell}, ser sempre o ltimo da
   * lista, ou seja, dependendo do sentido atual da ordenao, a no ser que o
   * outro objeto tambm esteja "marcado" como {@link FooterCell} e tenha um
   * peso maior que este.
   * 
   * @param o1 primeiro objeto a ser comparado.
   * @param o2 segundo objeto a ser comparado.
   * 
   * @return -1 se o primeiro objeto for menor do que o segundo, 1 se o primeiro
   *         objeto for maior que o segundo, 0 se forem considerados iguais.
   */
  @Override
  public int compare(T o1, T o2) {
    sortOrder = getSortOrder();
    if (FooterCell.class.isInstance(o1)) {
      if (FooterCell.class.isInstance(o2)) {
        int o1Weight = FooterCell.class.cast(o1).getWeight();
        int o2Weight = FooterCell.class.cast(o2).getWeight();
        if (o1Weight > o2Weight) {
          return (sortOrder == SortOrder.ASCENDING) ? 1 : -1;
        }
        return (sortOrder == SortOrder.ASCENDING) ? -1 : 1;
      }
      return (sortOrder == SortOrder.ASCENDING) ? 1 : -1;
    }
    if (FooterCell.class.isInstance(o2)) {
      return (sortOrder == SortOrder.ASCENDING) ? -1 : 1;
    }
    if (o1 == null) {
      return (o2 == null) ? 0 : -1;
    }
    if (o2 == null) {
      return 1;
    }
    return actual.compare(o1, o2);
  }

  /**
   * Callback para reordenaes da tabela. Indica que a ordenao mudou,
   * portanto ser necessrio rever o seu valor atual (mantido em um atributo do
   * objeto para efeitos de desempenho) na prxima ordenao.
   * 
   * @param source objeto (tabela) que disparou o evento (no usado).
   */
  @Override
  public void tableSorted(Object source) {
    this.sortOrder = null;
  }

}
