package tecgraf.javautils.gui.panel;

import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.JButton;

import tecgraf.javautils.gui.GUIResources;

/**
 * Agregador de {@link ExpandablePanel painis expansveis} . Gerencia operaes
 * "expandir todos" e "fechar todos", e possibilita a definio de um listener
 * para o grupo.
 * 
 * @author Tecgraf
 * 
 * @see ExpandablePanel
 */
public class ExpandablePanelGroup {
  /**
   * Id do evento enviado quando todos os painis so abertos.
   */
  public static final String EXPAND_ALL_ACTION_ID = "groupExpandAll";
  /**
   * Id do evento enviado quando todos os painis so fechados.
   */
  public static final String COLLAPSE_ALL_ACTION_ID = "groupCollapseAll";
  /**
   * Conjunto com os painis gerenciados pelo agregador.
   */
  private final Set<ExpandablePanel> panels =
    new LinkedHashSet<ExpandablePanel>();
  /**
   * Listener do agregador.
   */
  private ActionListener listener;

  /**
   * Acrescenta novos painis ao agregador.
   * 
   * @param panels painis
   * @return o prprio agregador, para encadeamento
   */
  public ExpandablePanelGroup add(ExpandablePanel... panels) {
    for (ExpandablePanel section : panels) {
      this.panels.add(section);
    }
    return this;
  }

  /**
   * Remove um painel do agregador.
   * 
   * @param panel painel a ser removido
   * @return <code>true</code> se o painel era gerenciado pelo agregador e foi
   *         removido
   */
  public boolean remove(ExpandablePanel panel) {
    return panels.remove(panel);
  }

  /**
   * Expande todos os painis.
   */
  public void expandAll() {
    expandAll(true);
  }

  /**
   * Fecha todos os painis.
   */
  public void collapseAll() {
    expandAll(false);
  }

  /**
   * Expande ou fecha todos os painis.
   * 
   * @param expand <code>true</code> para expandir todos os painis,
   *        <code>false</code> para fech-los
   */
  private void expandAll(boolean expand) {
    Set<Container> parents = new HashSet<Container>();
    boolean anyChange = false;
    for (ExpandablePanel section : panels) {
      if (section.setExpanded(expand, false, true)) {
        section.revalidate();
        parents.add(section.getParent());
        anyChange = true;
      }
    }
    /*
     * se ao menos um dos painis foi alterado, devemos avisar ao listener
     */
    if (anyChange) {
      if (listener != null) {
        listener.actionPerformed(new ActionEvent(this, (int) System
          .currentTimeMillis(), expand ? EXPAND_ALL_ACTION_ID
          : COLLAPSE_ALL_ACTION_ID));
      }
      /*
       * todos os pais de painis que foram alterados devem ser redesenhados
       */
      for (Container parent : parents) {
        parent.repaint();
      }
    }
  }

  /**
   * Define um listener para o agregador.
   * 
   * @param newListener listener. Se for <code>null</code>, remove o listener
   *        corrente
   * @return listener cadastrado anteriormente (ou <code>null</code> se no
   *         havia um listener)
   */
  public ActionListener setListener(ActionListener newListener) {
    ActionListener oldListener = listener;
    listener = newListener;
    return oldListener;
  }

  /**
   * Remove o listener. Equivale a {@link #setListener(ActionListener)} com
   * parmetro <code>null</code>.
   * 
   * @return listener cadastrado anteriormente (ou <code>null</code> se no
   *         havia um listener)
   */
  public ActionListener removeListener() {
    return setListener(null);
  }

  /**
   * Cria um boto para expandir todos os painis associados ao grupo.
   * 
   * @return boto para expandir todos os painis associados ao grupo
   */
  public JButton createExpandAllButton() {
    JButton button =
      new JButton(
        new AbstractAction(null, GUIResources.BUTTON_EXPAND_ALL_ICON) {
          @Override
          public void actionPerformed(ActionEvent e) {
            expandAll();
          }
        });
    ExpandablePanel.makeButtonTransparent(button);
    return button;
  }

  /**
   * Cria um boto para fechar todos os painis associados ao grupo.
   * 
   * @return boto para fechar todos os painis associados ao grupo
   */
  public JButton createCollapseAllButton() {
    JButton button =
      new JButton(new AbstractAction(null,
        GUIResources.BUTTON_COLLAPSE_ALL_ICON) {
        @Override
        public void actionPerformed(ActionEvent e) {
          collapseAll();
        }
      });
    ExpandablePanel.makeButtonTransparent(button);
    return button;
  }
}
