package tecgraf.javautils.gui;

import java.awt.Color;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dialog.ModalityType;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JProgressBar;
import javax.swing.JTextPane;
import javax.swing.Timer;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.Document;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNGKeys;

/**
 * A classe <code>Task</code> modela uma tarefa possivelmente demorada executada
 * no cliente. Uma thread  criada para a execuo da tarefa, para que eventos
 * de interface (como redesenho) possam ser atendidos. As interaes de teclado
 * e mouse, contudo, so bloqueadas. Caso a durao da tarefa ultrapasse um
 * valor limite configurvel, um dilogo modal contendo uma barra de progresso
 * ser exibido. Este dilogo ser fechado automaticamente ao final da tarefa.
 * Aps um intervalo de valor configurvel, um boto para fechamento do dilogo
 *  oferecido ao usurio. O texto do boto (Fechar ou Cancelar)  tambm
 * configurvel, pois depende do tipo de tarefa, e do efeito de seu fechamento.
 * 
 * Essa classe permite aninhamento. Isto , se dentro do mtodo performTask uma
 * outra tarefa for criada e executada, o retorno visual para seu acompanhamento
 *  dado na mesma janela da tarefa anterior, evitando que mltiplas janelas de
 * acompanhamento sejam abertas.
 * 
 *  importante notar que tarefas aninhadas devem, obrigatoriamente, usar os
 * mtodos beforeTaskUI e afterTaskUI para qualquer cdigo que atue sobre a
 * interface grfica.
 * 
 * @param <R> A classe do resultado da tarefa.
 * @author Tecgraf/PUC-Rio
 */
public abstract class Task<R> implements Runnable {

  /** Prefixo para as chaves de i18n. */
  private static final String LNG_KEY_PREFIX = Task.class.getName() + ".";

  /**
   * Armazena informao, na thread corrente, de que uma tarefa est executando.
   *  atravs dessa informao que uma tarefa detecta quando  chamada de forma
   * aninhada, pelo mtodo performTask de uma outra tarefa. A informao
   * armazenada  a lista de tarefas aninhadas em execuo.
   */
  private static ThreadLocal<List<Task<?>>> threadLocal =
    new ThreadLocal<List<Task<?>>>();

  /**
   * Lista de tarefas aninhadas em execuo.  a mesma informao que 
   * armazenada locamente na thread corrente. Essa informao  duplicada em
   * cada objeto tarefa para permitir seu acesso pelos mtodos executados pela
   * EDT.
   */
  private List<Task<?>> taskList = null;

  /**
   * Indice da tarefa na lista de tarefas aninhadas.
   */
  private int taskIndex = 0;

  /**
   * Indica se essa tarefa est aninhada em outra.
   */
  private boolean isNested = false;

  /**
   * Thread que executa a tarefa.
   */
  private Thread worker = null;

  /**
   * Tempo, conforme fornecido por System.currentTimeMillis, de incio da
   * execuo da tarefa.
   */
  private long startTime = 0;

  /**
   * Indica se os componentes de interface foram criados e exibidos. Esse campo
   * deve ser declarado volatile porque  acessado por diferentes threads sem
   * que seja feita sincronizao.
   */
  private volatile boolean uiShowing = false;

  /**
   * Indica se a execuo da tarefa terminou. Esse campo deve ser declarado
   * volatile porque  acessado por diferentes threads sem que seja feita
   * sincronizao.
   */
  private volatile boolean finished = false;

  /**
   * Indica se a execuo da tarefa foi cancelada. Esse campo deve ser declarado
   * volatile porque  acessado por diferentes threads sem que seja feita
   * sincronizao.
   */
  private volatile boolean cancelled = false;

  /**
   * Status de execuo da tarefa. Esse campo deve ser declarado volatile porque
   *  acessado por diferentes threads sem que seja feita sincronizao.
   */
  private volatile boolean status = false;

  /**
   * Resultado gerado pela tarefa.
   */
  private R result = null;

  /**
   * Erro detetado durante a execuo da tarefa.
   */
  private Exception error = null;

  /**
   * Tempo para abrir o dilogo de progresso.
   */
  private int progressDialogDelay = 2;

  /**
   * Tempo padro para cancelamento.
   */
  public static final int DEFAULT_CANCEL_DELAY = 30;

  /**
   * Tipo de texto do cancelamento.
   */
  public static final int CANCEL_BUTTON = 0;

  /**
   * Tipo de texto de fechamento.
   */
  public static final int CLOSE_BUTTON = 1;

  /**
   * Largura mnima para o dilogo de progresso.
   */
  private static final int MIN_WIDTH = 300;

  /**
   * Altura mnima para o dilogo de progresso.
   */
  private static final int MIN_HEIGHT = 100;

  /**
   * Texto exibido quando a informao ainda no est disponvel.
   */
  private static final String NA = " ---";

  /**
   * Tempo para habilitar o boto de fechar o dilogo de progresso.
   */
  private int cancelDelay = DEFAULT_CANCEL_DELAY;

  /**
   * Indica o texto do boto de fechar o dilogo de progresso.
   */
  private int cancelButtonType = CLOSE_BUTTON;

  /**
   * Indica se a taxa de progresso pode ser determinada.
   */
  private boolean showProgress = false;

  /**
   * Indica se os passos sero exibidos.
   */
  private boolean showSteps = false;

  /**
   * Imagem a ser usada no dilogo de progresso.
   */
  private ImageIcon image = null;

  /**
   * Janela a partir da qual a tarefa  comandada.
   */
  protected Window parentWindow = null;

  /**
   * Dilogo de progresso.
   */
  protected JDialog progressDialog = null;

  /**
   * Imagem.
   */
  private JLabel imageLabel = null;

  /**
   * Texto contendo mensagem, passo e tempos.
   */
  protected JTextPane textPane = null;

  /**
   * Barra de progresso.
   */
  protected JProgressBar progressBar = null;

  /**
   * Boto de fechar o dilogo de progresso.
   */
  private JButton cancelButton = null;

  /**
   * Formatador de inteiro para (hh:mm:ss) default.
   */
  private static DecimalFormat integerTimeFormatter = new DecimalFormat("00");

  /**
   * Ttulo da tarefa.
   */
  protected String taskTitle = null;

  /**
   * Mensagem para o dilogo da tarefa.
   */
  private String message = null;

  /**
   * Passo sendo executado pela tarefa.
   */
  private String step = null;

  /**
   * Tempo decorrido na execuo da tarefa.
   */
  private String elapsedTime = null;

  /**
   * Tempo estimado para trmino da tarefa.
   */
  private String estimatedTime = null;

  /**
   * Valor do percentual de execuo da tarefa.
   */
  private int percentage = 0;

  /**
   * Indica se o estado  desconhecido.
   */
  private boolean statusUnknown = false;

  /**
   * Decide se o dilogo de progresso ser exibido.
   */
  private boolean progressDialogEnabled = true;

  /**
   * Estilos de formatao dos textos.
   */
  private static SimpleAttributeSet[] attrs = null;

  /**
   * Tiop de bloqueio das janelas.
   */
  private ModalityType modality;

  /**
   * Define os estilos de formatao das mensagens.
   */
  static {
    attrs = new SimpleAttributeSet[4];
    int i = 0;
    // Message:
    attrs[i] = new SimpleAttributeSet();
    StyleConstants.setFontFamily(attrs[i], "SansSerif");
    StyleConstants.setFontSize(attrs[i], 14);
    i++;
    // Step:
    attrs[i] = new SimpleAttributeSet();
    StyleConstants.setFontFamily(attrs[i], "SansSerif");
    StyleConstants.setFontSize(attrs[i], 12);
    i++;
    // Elapsed time:
    attrs[i] = new SimpleAttributeSet();
    StyleConstants.setFontFamily(attrs[i], "Monospaced");
    StyleConstants.setFontSize(attrs[i], 10);
    i++;
    // Estimated time:
    attrs[i] = new SimpleAttributeSet();
    StyleConstants.setFontFamily(attrs[i], "Monospaced");
    StyleConstants.setFontSize(attrs[i], 10);
    i++;
  }

  /**
   * Cria uma task com bloqueio de janelas
   * {@link ModalityType#APPLICATION_MODAL}.
   */
  protected Task() {
    this(ModalityType.APPLICATION_MODAL);
  }

  /**
   * Cria uma task.
   * 
   * @param modality tipo de bloqueio das janelas
   */
  protected Task(ModalityType modality) {
    setModality(modality);
  }

  /**
   * Define o tipo de bloqueio de janelas.
   * 
   * @param modality tipo de bloqueio de janelas
   */
  private void setModality(ModalityType modality) {
    this.modality = modality;
  }

  /**
   * Caso alguma manipulao da interface seja necessria antes da execuo da
   * tarefa (atravs de performTask), ela deve ser realizada neste mtodo. Seu
   * contedo  executado na EDT e, aps seu trmino, a performTask  executada
   * em uma thread prpria.
   */
  protected void beforeTaskUI() {
    // Implementao padro vazia.
  }

  /**
   * Tarefa a ser executada. Esse mtodo deve ser implementado para executar a
   * tarefa.
   * 
   * @throws Exception quando a ocorrncia de um erro (representado pela
   *         exceo) impediu a realizao da tarefa
   */
  protected abstract void performTask() throws Exception;

  /**
   * Caso alguma manipulao da interface seja necessria aps da execuo da
   * tarefa (atravs de performTask), ela deve ser realizada neste mtodo. Seu
   * contedo  executado na EDT aps o trmino de performTask.
   */
  protected void afterTaskUI() {
    // Implementao padro vazia.
  }

  /**
   * Execuo da tarefa. Especializaes dessa classe (tarefas remotas, tarefas
   * locais) realizam o tratamento adequado para erros que possam ocorrer
   * durante a execuo da tarefa.
   */
  @Override
  public final void run() {
    try {
      if (!isNested) {
        taskList = new ArrayList<Task<?>>();
        threadLocal.set(taskList);
        taskList.add(this);
      }
      startTime = System.currentTimeMillis();
      performTask();
      status = true;
      error = null;
      synchronized (taskList) {
        taskList.remove(this);
      }
    }
    catch (Exception exception) {
      status = false;
      error = exception;
      synchronized (taskList) {
        taskList.remove(this);
      }
    }
    finally {
      if (!isNested) {
        threadLocal.remove();
      }
      else {
        // se for uma task aninhada, limpa o flag de interrupted da thread, 
        // pois ele pode ter sido atribudo por cancelar a sub-task e assim uma
        // sub-task seguinte ou a prpria task pai possa continuar sem problemas
        Thread.interrupted();
      }
    }
    finished = true;
  }

  /**
   * Armazena o resultado da tarefa executada, para obteno posterior.
   * 
   * @param result o resultado da tarefa
   */
  protected final void setResult(R result) {
    this.result = result;
  }

  /**
   * Obtm o resultado da tarefa executada.
   * 
   * @return o valor de resultado da tarefa
   */
  public final R getResult() {
    return result;
  }

  /**
   * Obtm o status da tarefa executada.
   * 
   * @return true se a tarefa foi completada false se um erro (exceo) impediu
   *         a realizao da tarefa
   */
  public final boolean getStatus() {
    return status;
  }

  /**
   * Obtm o erro detetado durante a execuo da tarefa.
   * 
   * @return exceo que representa o erro
   */
  public final Exception getError() {
    return error;
  }

  /**
   * Verifica se a tarefa foi cancelada pelo usurio. Esse mtodo pode ser
   * periodicamente chamado por uma implementao de <code>performTask</code>
   * que execute uma tarefa composta de uma sequncia de passos, ou subtarefas.
   * 
   * @return true se o boto de cancelamento foi acionado (cancelando a tarefa)
   */
  public boolean wasCancelled() {
    return cancelled;
  }

  /**
   * Cancela a tarefa, em consequncia do acionamento do boto de cancelamento.
   */
  protected void cancelTask() {
    cancelled = true;
    status = false;
  }

  /**
   * Tratamento de erros especfico para cada tipo de tarefa. Esse tratamento 
   * chamado na EDT. Essa implementao representa um tratamento mnimo, apenas
   * para erros no cliente. Qualquer exceo lanada durante a execuo do
   * mtodo performTask  repassada para este mtodo para tratamento.
   * 
   * @param exception - exceo que representa o erro durante execuo da
   *        tarefa.
   */
  protected void handleError(Exception exception) {
    if (exception != null) {
      StandardDialogs.showErrorDialog(parentWindow, taskTitle, exception
        .getMessage());
    }
  }

  /**
   * Faz pack no dilogo de progresso e ajusta seu tamanho mnimo para que a
   * largura no diminua. Isso  importante para que o dilogo no fique
   * aumentando e diminuindo em largura conforme as tarefas aninhadas vo sendo
   * iniciadas e terminadas.
   * 
   * @param dialog Dilogo de progresso.
   */
  private void pack(JDialog dialog) {
    dialog.pack();
    int width = Math.max(dialog.getWidth(), MIN_WIDTH);
    dialog.setMinimumSize(new Dimension(width, MIN_HEIGHT));
  }

  /**
   * Inicia a execuo da tarefa <b>no bloqueante</b>. Cria uma thread para
   * executar a tarefa, e aguarda o seu trmino durante tempo especificado
   * atravs do mtodo
   * 
   * @param window janela a partir da qual a tarefa foi comandada
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   */
  public final void executeNonBlocking(Window window, String title,
    String description) {
    int cancelDelaySecs = getCancelDelaySecs();
    preExcecute(window, title, description, cancelDelaySecs, CLOSE_BUTTON,
      true, false, null);
    invokePerformTask(false);
  }

  /**
   * Inicia a execuo da tarefa. Cria uma thread para executar a tarefa, e
   * aguarda o seu trmino durante tempo especificado atravs do mtodo
   * setProgressDialogDelay, ou o padro de dois segundo. Se aps esse tempo a
   * tarefa ainda no tiver terminado, cria e exibe o dilogo modal de
   * progresso. Se a tarefa for executada at o final (sem falhas sinalizadas
   * por excees), o valor <code>true</code>  retornado. O resultado da tarefa
   * pode ser obtido atravs da chamada ao mtodo <code>getResult</code>. Se uma
   * exceo for lanada durante a execuo da tarefa (incluindo seu
   * cancelamento), o valor <code>false</code>  retornado, e nenhum resultado
   * estar disponvel. Antes deste retorno, o procedimento para tratamento
   * especfico de erros (definido nas subclasses)  invocado. O dilogo de
   * progresso pode ser configurado de forma a exibir e atualizar os sucessivos
   * passos de uma tarefa. Tambm pode ser configurado para permitir a
   * atualizao do percentual da tarefa j executado (quando o progresso puder
   * ser determinado e for habilitado).
   * 
   * @param window janela a partir da qual a tarefa foi comandada
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   * @param cancelDelayInSecs intervalo em segundos para habilitar o boto de
   *        fechamento (se 0,  imediato)
   * @param cancelButtonType determina o tipo do boto de cancelamento (CANCEL
   *        ou CLOSE)
   * @param showProgress indica se a taxa de progresso pode ser determinada
   * @param showSteps indica se exibe passo corrente
   * @param icon imagem associada
   * 
   * @return true se a tarefa foi executada at o final e h um resultado ou
   *         false se a ocorrncia de um erro impediu a execuo da tarefa e no
   *         h resultado associado.
   */
  public final boolean execute(Window window, String title, String description,
    int cancelDelayInSecs, int cancelButtonType, boolean showProgress,
    boolean showSteps, ImageIcon icon) {
    preExcecute(window, title, description, cancelDelayInSecs,
      cancelButtonType, showProgress, showSteps, icon);
    invokePerformTask(true);
    invokeAfterTaskUI();
    if (!getStatus()) {
      setResult(null);
    }
    return getStatus();
  }

  /**
   * Configura os parmetros da execuo.
   * 
   * @param window janela a partir da qual a tarefa foi comandada. Se este
   *        parmetro for {@code null} a janela me final ser obtida com a
   *        chamada do mtodo protegido {@link #getDefaultParentWindow()}. Este
   *        mtodo pode ser redefinido, mas retorna, por default, {@code null}.
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   * @param cancelDelayInSecs intervalo em segundos para habilitar o boto de
   *        fechamento (se 0,  imediato)
   * @param cancelButtonType determina o tipo do boto de cancelamento (CANCEL
   *        ou CLOSE)
   * @param showProgress indica se a taxa de progresso pode ser determinada
   * @param showSteps indica se exibe passo corrente
   * @param icon imagem associada
   */
  private void preExcecute(Window window, String title, String description,
    int cancelDelayInSecs, int cancelButtonType, boolean showProgress,
    boolean showSteps, ImageIcon icon) {
    this.parentWindow = (window != null ? window : getDefaultParentWindow());
    this.taskTitle = title;
    this.message = description;
    this.cancelDelay = cancelDelayInSecs;
    this.cancelButtonType = cancelButtonType;
    this.showProgress = showProgress;
    this.showSteps = showSteps;
    this.image = icon;
    this.isNested = threadLocal.get() != null;
    if (isNested) {
      taskList = threadLocal.get();
      synchronized (taskList) {
        taskIndex = taskList.size();
        taskList.add(this);
        // Se alguma tarefa superior na hierarquia tiver sido cancelada, essa
        // deve ser cancelada tambm. O cancelamento de uma tarefa que aninha
        // outras dispara o cancelamento de todas as tarefas aninhadas. Porm,
        // pode haver uma condio de corrida: imediatamente aps terminar o
        // lao que chama cancelTask em cada tarefa aninhada, uma nova pode ser
        // criada. Por isso  importante fazer essa verificao adicional aps
        // a incluso da nova tarefa na taskList, ainda dentro do trecho
        // crtico envolvido por "synchronized (taskList)".
        for (Task<?> task : taskList) {
          if (task.wasCancelled()) {
            cancelTask();
          }
        }
      }
    }
    invokeBeforeTaskUI();
  }

  /**
   * Obtm a janela padro de referncia.
   * 
   * @return a janela padro.
   */
  protected Window getDefaultParentWindow() {
    return null;
  }

  /**
   * Ajusta o cursor de espera e executa o mtodo beforeTaskUI dentro da EDT.
   */
  private void invokeBeforeTaskUI() {
    Runnable code = new Runnable() {
      @Override
      public void run() {
        if (!isNested && parentWindow != null) {
          parentWindow
            .setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        }
        beforeTaskUI();
      }
    };
    invokeUI(code);
  }

  /**
   * Executa o performTask. Se esta tarefa estiver aninhada em outra, utiliza a
   * mesma thread para executar e usa o dilogo da tarefa que a aninha para dar
   * o retorno visual ao usurio. Se no estiver, cria uma thread e um dilogo
   * de progresso. Caso alguma exceo seja lanada por performTask, ela 
   * armazenada em error para posterior tratamento.
   * 
   * @param block indicativo de bloqueamento geral da IHC.
   */
  private void invokePerformTask(boolean block) {
    if (isNested) {
      invokeNestedPerformTask();
    }
    else {
      invokeThreadPerformTask(block);
    }
  }

  /**
   * Obtm a tarefa raiz. Retorna null caso ela no exista. Isso pode acontecer
   * em condies de corrida na interface, onde o listener  chamado e a tarefa
   * termina antes que ele execute.
   * 
   * @return a tarefa raiz.
   */
  private Task<?> getRoot() {
    if (taskList == null) {
      return null;
    }
    synchronized (taskList) {
      if (taskList.size() == 0) {
        return null;
      }
      return taskList.get(0);
    }
  }

  /**
   * Executa o performTask na thread corrente. Caso alguma exceo seja lanada
   * por performTask, ela  armazenada em error para posterior tratamento. Note
   * que esse mtodo s  chamado sobre instncias aninhadas de tarefa. Ou seja,
   * esse mtodo nunca  chamado sobre a tarefa raiz de uma sequncia de tarefas
   * aninhadas.
   */
  private void invokeNestedPerformTask() {
    // worker  associado ao worker da task raiz, para ser interrompido caso 
    // a task filha (aninhada) seja cancelada
    worker = getRoot().worker;
    // Executa a tarefa. Sua interface grfica ser criada, se necessrio, e
    // exibida pela tarefa principal, atravs de seu timer de controle. Dessa
    // forma, cada tarefa aninhada s ter sua interface exibida se durar ao
    // menos um segundo. Alm disso, essa interface ficar na tela tambm no
    // mnimo um segundo. Se isso no fosse feito, sucessivas tarefas rpidas
    // iriam piscar rapidamente na tela, criando um visual ruim para o usurio.
    run();
    Runnable code = new Runnable() {
      @Override
      public void run() {
        if (!uiShowing) {
          return;
        }
        Task<?> root = getRoot();
        if (root != null) {
          removeComponents(root.progressDialog.getContentPane());
          pack(root.progressDialog);
        }
        uiShowing = false;
      }
    };
    invokeUI(code);
  }

  /**
   * Executa o performTask em uma thread prpria. Caso alguma exceo seja
   * lanada por performTask, ela  armazenada em error para posterior
   * tratamento. Note que esse mtodo s  chamado sobre a instncia raiz de
   * tarefa. Ou seja, esse mtodo nunca  chamado sobre uma tarefa que esteja
   * aninhada em outra.
   * 
   * @param block indicativo de bloqueamento geral da IHC
   */
  private void invokeThreadPerformTask(final boolean block) {
    if (progressDialogEnabled) {
      Runnable code = new Runnable() {
        @Override
        public void run() {
          createProgressDialog(block);
        }
      };
      invokeUI(code);
    }
    worker = new Thread(this);
    String className = worker.getClass().getSimpleName();
    String myName = worker.toString();
    worker.setName("TASK::" + className + "::" + myName);
    worker.start();
    try {
      if (progressDialogEnabled) {
        if (progressDialogDelay > 0) {
          worker.join(progressDialogDelay * 1000);
        }
      }
      else if (block) {
        worker.join();
      }
    }
    catch (InterruptedException ie) {
      // No fazemos nada nesse caso, pois a Task pode estar rodando e pode
      // disparar outras Tasks aninhadas. Caso isso ocorra, essas Tasks
      // vo manipular a interface grfica de acompanhamento da execuo
      // (i.e., o progressDialog) pois a interrupo no chega a elas.
    }
    if (progressDialogEnabled) {
      Runnable code = new Runnable() {
        @Override
        public void run() {
          if (!worker.isAlive()) {
            progressDialog.dispose();
            return;
          }
          final Timer timer = new Timer(1000, null);
          timer.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
              progress(timer);
            }
          });
          timer.start();
          pack(progressDialog);
          uiShowing = true;
          progressDialog.setVisible(true);
          if (block) {
            progressDialog.dispose();
            uiShowing = false;
          }
        }
      };
      invokeUI(code);
    }
  }

  /**
   * Devolve o cursor padro, executa o mtodo afterTaskUI e o tratador de erro
   * dentro da EDT. O mtodo afterTaskUI s  chamado caso a execuo da tarefa
   * tenha ocorrido com sucesso at esse ponto. O tratador de erro s  chamado
   * se algum erro tiver ocorrido.
   */
  private void invokeAfterTaskUI() {
    Runnable code = new Runnable() {
      @Override
      public void run() {
        if (!isNested && parentWindow != null) {
          parentWindow.setCursor(Cursor
            .getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        }
        afterTaskUI();
        if (!getStatus() && !cancelled) {
          handleError(error);
        }
      }
    };
    invokeUI(code);
  }

  /**
   * Executa o cdigo fornecido dentro da EDT. Como o cdigo no pode lanar
   * excees verificadas, por estender Runnable, quaisquer excees que sejam
   * capturadas sero 'unchecked' e, por isso, no sero passadas para o
   * tratador.
   * 
   * @param code Cdigo a ser executado dentro da EDT.
   */
  private void invokeUI(Runnable code) {
    try {
      if (SwingThreadDispatcher.isEventDispatchThread()) {
        code.run();
      }
      else {
        SwingThreadDispatcher.invokeAndWait(code);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Atualiza a interface de progresso da tarefa. Esse mtodo  chamado em
   * intervalos regulares, por meio de um Timer, e  responsvel por fazer o
   * controle da interface. Note que ele  executado na EDT.<br>
   * O tempo de incio da tarefa  registrado e, a partir dele, esse mtodo deve
   * habilitar o boto de fechamento, exibir mensagens de passos, atualizar a
   * barra de progresso e, ao trmino da tarefa, fechar o dilogo. O mesmo
   * controle deve ser feito sobre as tarefas aninhadas.<br>
   * Note que esse mtodo s  chamado sobre a instncia raiz de tarefa. Ou
   * seja, esse mtodo nunca  chamado sobre uma tarefa que esteja aninhada em
   * outra.<br>
   * Para tarefas aninhadas, o tempo definido para exibio da interface 
   * ignorado, dado que j h a interface da tarefa raiz sendo exibida. Como o
   * Timer  executado a cada segundo, isso garante que as interfaces das
   * tarefas aninhadas no pisquem rapidamente na tela quando elas terminarem
   * rpido.
   * 
   * @param timer Timer responsvel por chamar este mtodo em intervalos
   *        regulares.
   */
  private void progress(Timer timer) {
    if (finished || cancelled) {
      timer.stop();
      progressDialog.setVisible(false);
      progressDialog.dispose();
      return;
    }
    boolean newButton = false;
    if (taskList == null) {
      // Se taskList for nula significa que a thread ainda no foi escalonada
      // para executar. Nesse caso, ainda no h nada a fazer com relao s
      // tarefas aninhadas pois se a principal ainda no executou no h como
      // existir tarefas aninhadas. Ento atua-se somente sobre a principal.
      updateTime();
      newButton = checkShowCancelButton();
    }
    else {
      synchronized (taskList) {
        for (Task<?> task : taskList) {
          if (task.finished || task.cancelled) {
            // Se a tarefa terminou ou foi cancelada, o lao pode ser
            // interrompido, pois no pode haver interface para ela e nem para
            // nenhuma das tarefas aninhadas.
            break;
          }
          if (!task.uiShowing) {
            task.addComponents(progressDialog.getContentPane());
            task.adjustSizes();
            task.pack(progressDialog);
            task.uiShowing = true;
          }
          task.updateTime();
          newButton = newButton || task.checkShowCancelButton();
        }
      }
    }
    if (newButton) {
      adjustSizes();
    }
    pack(progressDialog);
  }

  /**
   * Atualiza o tempo de corrido desde o incio da execuo da tarefa. Deve ser
   * chamado na EDT.
   */
  private void updateTime() {
    if (startTime == 0) {
      return;
    }
    long elapsed = (System.currentTimeMillis() - startTime) / 1000;
    elapsedTime = secToString(elapsed);
    if (showProgress) {
      double perc = progressBar.getValue();
      double p;
      p = perc < 0.0 ? 0.0 : perc;
      p = perc > 100.0 ? 100.0 : perc;
      if (p != 0.0) {
        long tsec = (long) (elapsed * 100D / p);
        estimatedTime = secToString(tsec - elapsed);
      }
      else {
        estimatedTime = NA;
      }
    }
    updateTextPane();
  }

  /**
   * Verifica se o boto de fechamento do dilogo de progresso deve ser exibido
   * e, se for o caso, o exibe.
   * 
   * @return <tt>true</tt> se o boto de fechamento do dilogo foi exibido, ou
   *         <tt>false</tt> caso ele no deva ser exibido ou j estava sendo
   *         exibido.
   */
  private boolean checkShowCancelButton() {
    if (cancelButton != null) {
      return false;
    }
    long now = System.currentTimeMillis();
    if (now - startTime < cancelDelay * 1000) {
      return false;
    }
    cancelButton = new JButton();
    String msg =
      cancelButtonType == CANCEL_BUTTON ? LNG.get(LNGKeys.CANCEL) : LNG
        .get(LNGKeys.CLOSE);
    cancelButton.setText(msg);
    cancelButton.addActionListener(new CancelListener());
    int i = taskIndex;
    int NL = 2;
    Task<?> root = getRoot();
    if (root != null) {
      Container cp = root.progressDialog.getContentPane();
      cp.add(cancelButton, new GBC(2, i * NL, new Insets(5, 0, 5, 5)));
      cp.invalidate();
    }
    return true;
  }

  /**
   * Listener de cancelamento de uma tarefa. Deve cancelar tambm todas as
   * tarefas aninhadas que existirem.
   */
  private class CancelListener implements ActionListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent ev) {
      synchronized (taskList) {
        Task<?> root = getRoot();
        if (root == null) {
          // Se a tarefa terminou, no h mais nada a fazer.
          return;
        }
        boolean cancellingRoot = (Task.this == root);
        for (int i = taskIndex; i < taskList.size(); i++) {
          Task<?> task = taskList.get(i);
          task.cancelTask();
          try {
            task.worker.interrupt();
          }
          catch (Exception ex) {
          }
          if (!cancellingRoot && i > 0) {
            task.removeComponents(root.progressDialog.getContentPane());
          }
        }
        if (cancellingRoot) {
          root.progressDialog.setVisible(false);
          root.progressDialog.dispose();
        }
        else {
          pack(root.progressDialog);
        }
      }
    }
  }

  /**
   * Cria um dilogo modal contendo uma mensagem, o tempo decorrido e uma barra
   * de progresso de durao indeterminada.
   * 
   * @param block . Se for <code>true</code> honra o modo definido para o campo
   *        <code>modality</code>, caso contrrio usa o modo
   *        {@link ModalityType#MODELESS}.
   */
  private void createProgressDialog(boolean block) {
    if (block) {
      if (modality == ModalityType.MODELESS) {
        /*
         * janela no pode ser bloqueante se no for de alguma forma modal;
         * neste caso, adotamos o modo mais "leve" (DOCUMENT_MODAL)
         */
        setModality(ModalityType.DOCUMENT_MODAL);
      }
    }
    else {
      setModality(ModalityType.MODELESS);
    }
    this.progressDialog = new JDialog(parentWindow, taskTitle, modality);
    Container cp = progressDialog.getContentPane();
    cp.setLayout(new GridBagLayout());
    addComponents(cp);

    addCloseCallback(block);

    progressDialog.setMinimumSize(new Dimension(MIN_WIDTH, MIN_HEIGHT));
    progressDialog.setResizable(false);
    pack(progressDialog);
    GUIUtils.centerWindow(progressDialog, parentWindow);
  }

  /**
   * Adio de callback de fechamento da janela da task.
   * 
   * @param block indicativo de bloqueio geral da IHC.
   */
  private void addCloseCallback(boolean block) {
    progressDialog.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);

    if (!block) {
      progressDialog.addWindowListener(new WindowAdapter() {
        @Override
        public void windowClosed(WindowEvent e) {
          invokeAfterTaskUI();
          if (!getStatus()) {
            setResult(null);
          }
          uiShowing = false;
        }
      });
    }
  }

  /**
   * Adiciona os componentes dessa tarefa no dilogo de progresso. Usa o ndice
   * dessa tarefa na lista de tarefas aninhadas para que a ordenao fique certa
   * e os componentes das tarefa apaream empilhados no dilogo, com a primeira
   * tarefa no topo, descendo at a ltima na base.
   * 
   * @param c Container do dilogo de progresso.
   */
  private void addComponents(Container c) {
    int i = taskIndex;
    int NL = 2;
    int NR = 3;
    // Image:
    if (image != null) {
      imageLabel = new JLabel(image);
    }
    else {
      imageLabel = new JLabel();
    }
    c.add(imageLabel, new GBC(0, i * NL, new Insets(5, 5, 5, 0)));
    // textPane:
    textPane = new JTextPane(new DefaultStyledDocument());
    textPane.setEditable(false);
    textPane.setBackground(imageLabel.getBackground());
    updateTextPane();
    c.add(textPane, new GBC(1, i * NL, new Insets(5, 5, 5, 5)).horizontal());
    // Progress bar:
    progressBar = new JProgressBar();
    if (!showProgress || statusUnknown) {
      progressBar.setIndeterminate(true);
      progressBar.setStringPainted(false);
      statusUnknown = false;
    }
    else {
      progressBar.setIndeterminate(false);
      progressBar.setMinimum(0);
      progressBar.setMaximum(100);
      progressBar.setValue(percentage);
      progressBar.setStringPainted(true);
    }
    c.add(progressBar, new GBC(0, i * NL + 1, new Insets(0, 5, 5, 5)).width(NR)
      .horizontal());
    c.invalidate();
  }

  /**
   * Atualiza o texto de progresso, contendo a mensagem, o passo e os tempos.
   * Deve ser chamado na EDT.
   */
  private void updateTextPane() {
    Document doc = textPane.getStyledDocument();
    try {
      doc.remove(0, doc.getLength());
      String text;
      int i = 0;
      // Message:
      text = message;
      doc.insertString(doc.getLength(), text, attrs[i]);
      i++;
      // Step:
      if (showSteps) {
        doc.insertString(doc.getLength(), "\n", null);
        text = LNG.get(LNG_KEY_PREFIX + "stepText") + " ";
        if (step != null) {
          text += step;
        }
        else {
          text += NA;
        }
        doc.insertString(doc.getLength(), text, attrs[i]);
      }
      i++;
      // Total time:
      doc.insertString(doc.getLength(), "\n", null);
      text = LNG.get(LNG_KEY_PREFIX + "elapsedTime");
      if (elapsedTime != null) {
        text += elapsedTime;
      }
      else {
        text += NA;
      }
      doc.insertString(doc.getLength(), text, attrs[i]);
      i++;
      // Estimated time:
      if (showProgress) {
        doc.insertString(doc.getLength(), "\n", null);
        text = LNG.get(LNG_KEY_PREFIX + "totalTime");
        if (estimatedTime != null) {
          text += estimatedTime;
        }
        else {
          text += NA;
        }
        doc.insertString(doc.getLength(), text, attrs[i]);
      }
      i++;
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
    Task<?> root = getRoot();
    if (root != null) {
      Container cp = root.progressDialog.getContentPane();
      cp.invalidate();
    }
  }

  /**
   * Remove os componentes de interface de uma tarefa do dilogo de progresso.
   * Deve ser chamado da EDT.
   * 
   * @param c Container do dilogo de progresso.
   */
  private void removeComponents(Container c) {
    if (imageLabel != null) {
      c.remove(imageLabel);
    }
    if (textPane != null) {
      c.remove(textPane);
    }
    if (progressBar != null) {
      c.remove(progressBar);
    }
    if (cancelButton != null) {
      c.remove(cancelButton);
    }
    c.invalidate();
  }

  /**
   * Ajusta os tamanhos das imagens e dos botes para que fiquem iguais e,
   * assim, a interface fique alinhada.
   */
  private void adjustSizes() {
    synchronized (taskList) {
      List<JComponent> widgets = new ArrayList<JComponent>();
      for (Task<?> task : taskList) {
        if (task.imageLabel != null) {
          widgets.add(task.imageLabel);
        }
      }
      adjustSizes(widgets);
      widgets.clear();
      for (Task<?> task : taskList) {
        if (task.cancelButton != null) {
          widgets.add(task.cancelButton);
        }
      }
      adjustSizes(widgets);
    }
  }

  /**
   * Ajusta os tamanhos dos elementos fornecidos de forma que todos fiquem com o
   * mesmo tamanho, usando as maiores dimenses entre todos.
   * 
   * @param widgets Componentes cujos tamanhos devem ser ajustados.
   */
  private void adjustSizes(List<JComponent> widgets) {
    double w = 0;
    double h = 0;
    for (JComponent widget : widgets) {
      widget.setPreferredSize(null);
      Dimension wdim = widget.getPreferredSize();
      w = Math.max(w, wdim.getWidth());
      h = Math.max(h, wdim.getHeight());
    }
    Dimension dim = new Dimension();
    dim.setSize(w, h);
    for (JComponent widget : widgets) {
      widget.setPreferredSize(dim);
    }
  }

  /**
   * Atualiza o texto do passo corrente.
   * 
   * @param step identificao do passo corrente
   */
  public void setStepText(final String step) {
    //  importante armazenar a informao mesmo que ela no seja usada nesse
    // momento, pois esse mtodo pode ser chamado antes da interface ser
    // exibida. Nesse caso, a informao precisa estar disponvel quando ela
    // for exibida.
    this.step = step;
    if (showSteps) {
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          if (finished || cancelled || !uiShowing) {
            return;
          }
          updateTextPane();
          Task<?> root = getRoot();
          if (root != null) {
            pack(root.progressDialog);
          }
        }
      });
    }
  }

  /**
   * Atualiza a barra de progresso.
   * 
   * @param percentage percentual da tarefa j completada
   */
  public void setProgressStatus(final int percentage) {
    //  importante armazenar a informao mesmo que ela no seja usada nesse
    // momento, pois esse mtodo pode ser chamado antes da interface ser
    // exibida. Nesse caso, a informao precisa estar disponvel quando ela
    // for exibida.
    this.percentage = percentage;
    if (showProgress) {
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          if (finished || cancelled || !uiShowing) {
            return;
          }
          int p;
          p = percentage < 0 ? 0 : percentage;
          p = percentage > 100 ? 100 : percentage;
          progressBar.setIndeterminate(false);
          progressBar.setStringPainted(true);
          progressBar.setValue(p);
          int c = 200 - (int) (200.0 * p / 100.0);
          int b = 255 - (int) (150.0 * p / 100.0);
          progressBar.setForeground(new Color(c, c, b));
        }
      });
    }
  }

  /**
   * Atualiza a barra de progresso para desconhecido.
   */
  public void setUnknownStatus() {
    //  importante armazenar a informao mesmo que ela no seja usada nesse
    // momento, pois esse mtodo pode ser chamado antes da interface ser
    // exibida. Nesse caso, a informao precisa estar disponvel quando ela
    // for exibida.
    this.statusUnknown = true;
    if (showProgress) {
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          if (finished || cancelled || !uiShowing) {
            return;
          }
          progressBar.setIndeterminate(true);
          progressBar.setStringPainted(false);
        }
      });
    }
  }

  /**
   * Converso de inteiro para string com dois dgitos.
   * 
   * @param v a quantidade.
   * @return uma string formatada.
   */
  private static String timeFormat(long v) {
    return integerTimeFormatter.format(v);
  }

  /**
   * Converso de segundos para um texto de tempo.
   * 
   * @param seconds a quantidade de segundos.
   * @return um texto formatado.
   */
  private static String secToString(long seconds) {
    if (seconds < 0) {
      return "-";
    }
    long s = seconds;
    long m = 0;
    long h = 0;
    if (s >= 60) {
      m = (int) s / 60;
      s = s - (m * 60);
    }
    if (m < 60) {
      return " " + timeFormat(m) + ":" + timeFormat(s);
    }
    h = (int) m / 60;
    m = m - (h * 60);
    return " " + timeFormat(h) + ":" + timeFormat(m) + ":" + timeFormat(s);
  }

  /**
   * Consulta do tempo (em segundos) para aparecimento do boto de cancelamento
   * aps a task entrar.
   * 
   * @return o tempo
   */
  protected int getCancelDelaySecs() {
    return DEFAULT_CANCEL_DELAY;
  }

  /**
   * Sobrecarga de execute assumindo que no h imagem a ser exibida no dilogo
   * de progresso.
   * 
   * @param window janela a partir da qual a tarefa foi comandada
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   * @param cancelDelayInSecs intervalo em segundos para habilitar o boto de
   *        fechamento (se 0,  imediato)
   * @param cancelButtonType determina o tipo do boto de cancelamento (CANCEL
   *        ou CLOSE)
   * @param showProgress indica se a taxa de progresso pode ser determinada
   * @param showSteps indica se exibe passo corrente
   * 
   * @return true se a tarefa foi executada at o final e h um resultado ou
   *         false se a ocorrncia de um erro impediu a execuo da tarefa e no
   *         h resultado associado.
   */
  public final boolean execute(Window window, String title, String description,
    int cancelDelayInSecs, int cancelButtonType, boolean showProgress,
    boolean showSteps) {
    return execute(window, title, description, cancelDelayInSecs,
      cancelButtonType, showProgress, showSteps, null);
  }

  /**
   * Sobrecarga de execute assumindo o tempo padro para habilitar o boto de
   * fechamento, o texto padro (Fechar) para esse boto, que no  possvel
   * determinar a taxa de progresso, que no sero exibidos os passos e que no
   * h imagem a ser exibida no dilogo de progresso.
   * 
   * @param window janela a partir da qual a tarefa foi comandada
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   * 
   * @return true se a tarefa foi executada at o final e h um resultado ou
   *         false se a ocorrncia de um erro impediu a execuo da tarefa e no
   *         h resultado associado.
   */
  public final boolean execute(Window window, String title, String description) {
    final int delay = getCancelDelaySecs();
    return execute(window, title, description, delay, CLOSE_BUTTON, false,
      false, null);
  }

  /**
   * Sobrecarga de execute assumindo o tempo padro para habilitar o boto de
   * fechamento, o texto padro (Fechar) para esse boto, que no  possvel
   * determinar a taxa de progresso e que no sero exibidos os passos.
   * 
   * @param window janela a partir da qual a tarefa foi comandada
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   * @param icon imagem associada
   * 
   * @return true se a tarefa foi executada at o final e h um resultado ou
   *         false se a ocorrncia de um erro impediu a execuo da tarefa e no
   *         h resultado associado.
   */
  public final boolean execute(Window window, String title, String description,
    ImageIcon icon) {
    final int delay = getCancelDelaySecs();
    return execute(window, title, description, delay, CLOSE_BUTTON, false,
      false, icon);
  }

  /**
   * Sobrecarga de execute assumindo o tempo padro para habilitar o boto de
   * fechamento, o texto padro (Fechar) para esse boto e que no h imagem a
   * ser exibida no dilogo de progresso.
   * 
   * @param window janela a partir da qual a tarefa foi comandada
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   * @param showProgress indica se a taxa de progresso pode ser determinada
   * @param showSteps indica se exibe passo corrente
   * 
   * @return true se a tarefa foi executada at o final e h um resultado ou
   *         false se a ocorrncia de um erro impediu a execuo da tarefa e no
   *         h resultado associado.
   */
  public final boolean execute(Window window, String title, String description,
    boolean showProgress, boolean showSteps) {
    final int delay = getCancelDelaySecs();
    return execute(window, title, description, delay, CLOSE_BUTTON,
      showProgress, showSteps);
  }

  /**
   * Sobrecarga de execute assumindo que no  possvel determinar a taxa de
   * progresso, que no sero exibidos os passos e que no h imagem a ser
   * exibida no dilogo de progresso.
   * 
   * @param window janela a partir da qual a tarefa foi comandada
   * @param title ttulo do dilogo de progresso
   * @param description mensagem no dilogo de progresso
   * @param cancelDelayInSecs intervalo em segundos para habilitar o boto de
   *        fechamento (se 0,  imediato)
   * @param cancelButtonType determina o tipo do boto de cancelamento (CANCEL
   *        ou CLOSE)
   * 
   * @return true se a tarefa foi executada at o final e h um resultado ou
   *         false se a ocorrncia de um erro impediu a execuo da tarefa e no
   *         h resultado associado.
   */
  public final boolean execute(Window window, String title, String description,
    int cancelDelayInSecs, int cancelButtonType) {
    return execute(window, title, description, cancelDelayInSecs,
      cancelButtonType, false, false);
  }

  /**
   * Tempo de espera para exibio do dilogo
   * 
   * @return Tempo em segundos
   */
  public final int getProgressDialogDelay() {
    return progressDialogDelay;
  }

  /**
   * Configura tempo de espera para exibio do dilogo
   * 
   * @param progressDialogDelay tempo em segundos
   */
  public final void setProgressDialogDelay(int progressDialogDelay) {
    this.progressDialogDelay = progressDialogDelay;
  }

  /**
   * Desabilita exibio do dilogo de progresso durante execuo da tarefa.
   * 
   * @param b indicativo de progresso habilitado.
   */
  public final void setProgressDialogEnabled(boolean b) {
    this.progressDialogEnabled = b;
  }

  /**
   * @return a mensagem que aparece no dilogo de progresso dessa tarefa.
   */
  public String getTaskMessage() {
    return message;
  }

  /**
   * @return lista de informaes adicionais que vo aparecer nos dilogos de
   *         erro para essa tarefa, e sero enviadas ao email do administrador
   *         se necessrio. Ex: 'Nome do projeto', 'Caminho do arquivo'. Cada
   *         informao ser mostrada em uma linha. Retornar NULL para nenhuma
   *         informao adicional.
   */
  protected String[] getAdditionalInfo() {
    return null;
  }
}