package tecgraf.javautils.gui.print;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.ImageObserver;
import java.awt.print.PageFormat;

import javax.swing.ImageIcon;

/**
 * Imagem a ser impressa. Pode ser alinhada  esquerda,  direita ou
 * centralizada. Se a largura da imagem for maior do que a largura da pgina, ou
 * a altura da imagem for maior do que a altura da pgina, a imagem ser
 * cortada.
 */
public class PrintableImage implements PrintableReportItem, ImageObserver {

  /** Imagem a ser impressa */
  private ImageIcon image;
  /** Altura da imagem */
  private float height;
  /** Indica se o carregamento da imagem foi completado */
  private boolean done;
  /** Alinhamento horizontal da imagem */
  private HorizontalAlignment alignment;

  /** Constantes para alinhamento horizontal da imagem */
  public enum HorizontalAlignment {
    /** Alinhada  esquerda */
    LEFT,
    /** Alinhada  direita */
    RIGHT,
    /** Centralizada */
    CENTER
  }

  /**
   * Construtor.
   * 
   * @param image imagem a ser impressa.
   * @param alignment alinhamento horizontal da imagem.
   */
  public PrintableImage(ImageIcon image, HorizontalAlignment alignment) {
    this.image = image;
    this.alignment = alignment;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean simulatePrint(Graphics2D g2, PageFormat format, int pageIndex) {
    /*
     * Mesmo que a imagem no caiba e seja cortada, consideramos que ela foi
     * totalmente impressa.
     */
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean print(Graphics2D g2, PageFormat format, int pageIndex) {
    height = image.getIconHeight();
    int x = (int) format.getImageableX();

    switch (alignment) {
      case LEFT:
        break;
      case RIGHT:
        x += ((int) format.getImageableWidth() - image.getIconWidth());
        break;
      case CENTER:
        x += ((int) format.getImageableWidth() - image.getIconWidth()) / 2;
        break;
      default:
        throw new IllegalArgumentException("alignment " + alignment
          + " desconhecido.");
    }
    int y = (int) format.getImageableY();
    drawImage(image, g2, x, y);
    return true;
  }

  /**
   * Desenha uma imagem nas coordenadas passadas.
   * 
   * @param image imagem a ser desenhada.
   * @param g2 contexto grfico.
   * @param x coordenada horizontal.
   * @param y coordenada vertical.
   */
  private void drawImage(ImageIcon image, Graphics2D g2, int x, int y) {
    int height = image.getIconHeight();
    int width = image.getIconWidth();
    Image img = image.getImage();
    done = false;
    boolean drawn = g2.drawImage(img, x, y, width, height, this);
    if (!drawn) {
      while (true) {
        synchronized (this) {
          if (done) {
            break;
          }
        }
        try {
          Thread.sleep(100);
        }
        catch (InterruptedException e) {
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getHeight() {
    return height;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean imageUpdate(Image img, int infoflags, int x,
    int y, int width, int height) {
    done = ((infoflags & (ERROR | FRAMEBITS | ALLBITS)) != 0);
    return !done;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initPrinting(PrintConfiguration configuration) {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getWidth() {
    return image.getIconWidth();
  }
}
