package tecgraf.javautils.gui.wizard;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.SwingConstants;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNGKeys;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIResources;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.wizard.Step.ConfirmOperation;

/**
 * Representa um wizard (assistente) para a obteno de dados do usurio e,
 * posteriormente, execuo de uma ao.
 *
 * @see WizardSample
 *
 * @author Tecgraf/PUC-Rio
 */
public final class Wizard extends JPanel {

  /**
   * Define os tipos de finalizao suportados.
   *
   * @author Tecgraf/PUC-Rio
   */
  public enum FinishingType {
    /**
     * Significa que o wizard pode ser cancelado. Um boto Cancelar poder ser
     * exibido.
     */
    CANCELABLE,

    /**
     * Significa que o wizard pode ser fechado. Um boto Fechar poder ser
     * exibido.
     */
    CLOSEABLE,

    /**
     * Significa que o wizard pode ser cancelado e fechado. Um boto de Cancelar
     * e/ou um botao de fechar podero ser exibidos.
     */
    CANCELABLE_AND_CLOSEABLE;
  }

  /** Prefixo para o nome das chaves de traduo. */
  static final String LNG_KEY_PREFFIX = Wizard.class.getName() + ".";

  /**
   * Representa a ao de cancelar.
   *
   * @author Tecgraf/PUC-Rio
   */
  private final class CancelAction extends AbstractAction {
    /**
     * Tag de descrio para i18n.
     */
    private static final String CANCEL_SHORT_DESCRIPTION =
      "cancelShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      cancel();
    }

    /**
     * Construtor
     */
    CancelAction() {
      putValue(Action.NAME, LNG.get(LNGKeys.CANCEL));
      putValue(Action.SHORT_DESCRIPTION, LNG.get(LNG_KEY_PREFFIX
        + CANCEL_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa a ao de fechar.
   *
   * @author Tecgraf/PUC-Rio
   */
  private final class CloseAction extends AbstractAction {
    /**
     * Tag de descrio para i18n.
     */
    private static final String CLOSE_SHORT_DESCRIPTION =
      "closeShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      close();
    }

    /**
     * Construtor
     */
    private CloseAction() {
      putValue(Action.NAME, LNG.get(LNGKeys.CLOSE));
      putValue(Action.SHORT_DESCRIPTION, LNG.get(LNG_KEY_PREFFIX
        + CLOSE_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa a ao de confirmao.
   *
   * @author Tecgraf/PUC-Rio
   */
  private final class ConfirmAction extends AbstractAction {
    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      confirm();
    }

    /**
     * Construtor
     *
     * @param step passo
     */
    ConfirmAction(Step step) {
      putValue(Action.NAME, step.getConfirmActionName());
      putValue(Action.SHORT_DESCRIPTION, step.getConfirmActionTooltip());
    }
  }

  /**
   * Representa a ao de ir para o prximo passo.
   *
   * @author Tecgraf/PUC-Rio
   */
  private final class GoNextAction extends AbstractAction {
    /**
     * Tag de texto para i18n
     */
    private static final String NEXT = "next";

    /**
     * Tag de descrio para i18n.
     */
    private static final String NEXT_SHORT_DESCRIPTION = "nextShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      Step oldStep = Wizard.this.currentStep;
      if (!oldStep.validate()) {
        return;
      }
      if (!exitCurrentStep()) {
        return;
      }
      goNext();
      enterCurrentStep();
      adjustStackToGoNext(oldStep);
      notifyGoneNext();
    }

    /**
     * Construtor
     */
    GoNextAction() {
      putValue(Action.SMALL_ICON, GUIResources.FORWARD_ICON);
      putValue(Action.NAME, LNG.get(LNG_KEY_PREFFIX + NEXT));
      putValue(Action.SHORT_DESCRIPTION, LNG.get(LNG_KEY_PREFFIX
        + NEXT_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa a ao de ir para o passo anterior.
   *
   * @author Tecgraf/PUC-Rio
   */
  private final class GoPreviousAction extends AbstractAction {
    /**
     * Tag de texto para i18n
     */
    private static final String PREVIOUS = "previous";

    /**
     * Tag de descrio para i18n.
     */
    private static final String NEXT_SHORT_DESCRIPTION =
      "previousShortDescription";

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      Step oldStep = Wizard.this.currentStep;
      if (!exitCurrentStep()) {
        return;
      }
      goPrevious();
      enterCurrentStep();
      adjustStackToGoPrevious(oldStep);
      notifyGonePrevious();
    }

    /**
     * Construtor
     */
    GoPreviousAction() {
      putValue(Action.SMALL_ICON, GUIResources.BACK_ICON);
      putValue(Action.NAME, LNG.get(LNG_KEY_PREFFIX + PREVIOUS));
      putValue(Action.SHORT_DESCRIPTION, LNG.get(LNG_KEY_PREFFIX
        + NEXT_SHORT_DESCRIPTION));
    }
  }

  /**
   * Representa um ouvinte de passos.
   *
   * @author Tecgraf/PUC-Rio
   */
  private final class WizardStepListener implements IStepListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void wasChanged(Step step) {
      updateProgressButtons();
    }
  }

  /** A largura mnima do wizard. */
  private int minimum_width = 500;

  /** A altura mnima do wizard. */
  private int minimum_height = 250;

  /** O boto para cancelamento do wizard.  opcional. */
  private JButton cancelButton;

  /** O boto para fechamento do wizard.  opcional. */
  private JButton closeButton;

  /**
   * O boto de confirmao do wizard. S  exibido em passos onde  possvel
   * confirmar o wizard.
   */
  private JButton confirmButton;

  /**
   * O boto que leva para o prximo passo.
   */
  private JButton nextButton;

  /**
   * O boto que leva para o passo anterior.
   */
  private JButton previousButton;

  /**
   * Serve para exibir o ttulo do passo corrente.
   */
  final private JLabel stepTitleLabel = new JLabel();

  /**
   * Server para exibir as instrues do passo corrente.
   */
  final private JTextArea stepInstructionTextArea = new JTextArea();

  /**
   * O histrico utilizado para armazenar os dados globais e os dados obtidos em
   * cada passo.
   */
  private History history;

  /**
   * O conjunto de listeners interessados em eventos do wizard.
   */
  private Set<IWizardListener> listeners;

  /**
   * O ouvinte de passos.  cadastrado no passo corrente.
   */
  private WizardStepListener wasChangedStepListener;

  /**
   * Representa o painel onde ser inserido o painel de cada passo.
   */
  final private JPanel stepPanel = new JPanel();

  /**
   * Lista com os grupos de passos que compem o wizard.
   */
  private List<StepGroup> stepGroupList;

  /**
   * Tipo de finalizao suportado pelo wizard.
   */
  private FinishingType finishingType;

  /**
   * O passo corrente.
   */
  private Step currentStep;

  /**
   * O grupo de passos corrente.
   */
  private StepGroup currentGroup;

  /**
   * Indica se os botes de finilizao, Fechar, Cancelar e Confirmar
   * permanecero visivis quando desabilitados.
   */
  private boolean hideFinishButtonsWhenDisabled;

  /**
   * Indica se o histrico ser limpo em caso de retrocesso do wizard e mudana
   * dos dados
   */
  private boolean resetHistoryOnChange = false;

  /**
   * Indica se  possvel saltar um conjunto de passos
   */
  private boolean canSkipSteps;

  /**
   * SplitPane usado no wizard
   */
  private JSplitPane splitPane;

  /**
   * Componente a ser usado do lado esquerdo.
   */
  private ILeftDecoration leftDecoration;

  /**
   * Pilha com os passos executados.
   */
  final private Stack<Class<? extends Step>> stack;

  /**
   * Adiciona um ouvinte  lista de ouvintes do wizard.
   *
   * @param listener O ouvinte a ser adicionado.
   */
  public void addListener(IWizardListener listener) {
    this.listeners.add(listener);
  }

  /**
   * Remove um ouvinte da lista de ouvintes do wizard.
   *
   * @param listener O ouvinte a ser removido.
   */
  public void removeListener(IWizardListener listener) {
    this.listeners.remove(listener);
  }

  /**
   * Cancela o wizrd.
   */
  void cancel() {
    try {
      this.currentStep.cancel();
      notifyCancelled();
    }
    catch (WizardException exception) {
      manageException(exception);
    }
  }

  /**
   * Fecha o wizard.
   */
  private void close() {
    notifyClosed();
  }

  /**
   * Confirma o wizard.
   */
  private void confirm() {
    try {
      this.history.setResult(this.currentStep.confirm(this.history));
      notifyConfirmed();
      final ConfirmOperation cnfOperation = currentStep.getConfirmOperation();
      if (cnfOperation.equals(Step.ConfirmOperation.GO_NEXT)) {
        final Step oldStep = Wizard.this.currentStep;
        if (!oldStep.validate()) {
          return;
        }

        if (!exitCurrentStep()) {
          return;
        }
        goNext();
        enterCurrentStep();
        adjustStackToGoNext(oldStep);
        notifyGoneNext();
      }
    }
    catch (WizardException exception) {
      manageException(exception);
    }
  }

  /**
   * Preenche a interface grfica de usurio. So criados o painel de sequncia
   * de passos e o painel de controle.
   */
  private void fillGui() {
    JPanel controlStepPanel = createControlStepPane();

    if (leftDecoration != null) {
      addListener(leftDecoration);
      final Component component = leftDecoration.getComponent();
      if (leftDecoration.isSplitPaneNeeded()) {
        splitPane = new JSplitPane();
        splitPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
        splitPane.setLeftComponent(component);
        splitPane.setRightComponent(controlStepPanel);
        splitPane.setOneTouchExpandable(true);
        splitPane.setContinuousLayout(true);
        splitPane.getLeftComponent().setMinimumSize(new Dimension(0, 0));
        splitPane.getRightComponent().setMinimumSize(new Dimension(0, 0));
        splitPane.setDividerLocation(component.getPreferredSize().width);
        splitPane.setLastDividerLocation(0);
        setLayout(new BorderLayout());
        add(splitPane, BorderLayout.CENTER);
      }
      else {
        setLayout(new GridBagLayout());
        add(component, new GBC(0, 0).center().both(0.1, 1.0));
        add(controlStepPanel, new GBC(1, 0).center().both(0.9, 1.0));
      }
    }
    else {
      splitPane = null;
      setLayout(new BorderLayout());
      add(controlStepPanel, BorderLayout.CENTER);
    }

    Dimension wizardDimension = this.getPreferredDimension();
    this.setPreferredSize(wizardDimension);
  }

  /**
   * Retorna o tamanho inicial do wizard. O tamanho do wizard  calculado a
   * partir do tamanho do maior passo.
   *
   * @return O tamanho do wizard.
   */
  private Dimension getPreferredDimension() {
    Iterator<?> stepGroupIterator = this.stepGroupList.iterator();
    int maximumStepHeight = 0;
    int maximumStepWidth = 0;
    while (stepGroupIterator.hasNext()) {
      StepGroup stepGroup = (StepGroup) stepGroupIterator.next();
      Set<?> stepSet = stepGroup.getAll();
      Iterator<?> stepSetIterator = stepSet.iterator();
      while (stepSetIterator.hasNext()) {
        Step step = (Step) stepSetIterator.next();
        int currentStepPreferredHeight = (int) step.getContainer()
          .getPreferredSize().getHeight();
        if (currentStepPreferredHeight > maximumStepHeight) {
          maximumStepHeight = currentStepPreferredHeight;
        }
        int currentStepPreferredWidth = (int) step.getContainer()
          .getPreferredSize().getWidth();
        if (currentStepPreferredWidth > maximumStepWidth) {
          maximumStepWidth = currentStepPreferredWidth;
        }
      }
    }
    if (maximumStepHeight < minimum_height) {
      maximumStepHeight = minimum_height;
    }
    if (maximumStepWidth < minimum_width) {
      maximumStepWidth = minimum_width;
    }

    int leftCmpWidth = 0;
    if (leftDecoration != null) {
      final Component leftComponent = leftDecoration.getComponent();
      leftCmpWidth = leftComponent.getPreferredSize().width;
    }
    return new Dimension(leftCmpWidth + maximumStepWidth, maximumStepHeight);
  }

  /**
   * Atualiza o estado dos botes desabilitando-os ou tornando-os invisveis,
   * quando necessrio.
   */
  private void updateButtons() {
    this.previousButton.setEnabled(this.currentStep.hasPrevious()
      && this.currentStep.canGoPrevious());
    this.confirmButton.setVisible(this.currentStep
      .getConfirmActionName() != null);
    this.confirmButton.setAction(new ConfirmAction(this.currentStep));
    updateProgressButtons();
    updateFinishButtons();
  }

  /**
   * Atualiza os botes de finalizao (cancelar e fechar).
   */
  private void updateFinishButtons() {
    if (this.cancelButton != null) {
      this.cancelButton.setEnabled(this.currentStep.canCancel());
      if (hideFinishButtonsWhenDisabled) {
        this.cancelButton.setVisible(this.currentStep.canCancel());
      }
    }
    if (this.closeButton != null) {
      this.closeButton.setEnabled(this.currentStep.canClose());
      if (hideFinishButtonsWhenDisabled) {
        this.closeButton.setVisible(this.currentStep.canClose());
      }
    }
  }

  /**
   * Entra no passo corrente.
   */
  private void enterCurrentStep() {
    try {
      this.currentStep.addListener(this.wasChangedStepListener);
      this.currentStep.addListener(this.history);
      this.currentStep.enter(this.history);
      this.stepTitleLabel.setText(this.currentStep.getTitle());
      this.stepInstructionTextArea.setText(this.currentStep.getInstruction());
      this.stepInstructionTextArea.setLineWrap(true);
      this.stepInstructionTextArea.setWrapStyleWord(true);
      this.stepPanel.removeAll();
      this.stepPanel.setLayout(new BorderLayout());
      this.stepPanel.add(this.currentStep.getContainer(), BorderLayout.CENTER);
      this.updateButtons();
      this.stepPanel.revalidate();
      this.repaint();
    }
    catch (WizardException exception) {
      manageException(exception);
    }
  }

  /**
   * Sai do passo corrente.
   *
   * @return indicativo de que o exit foi realmente feito (a step da aplicao
   *         no levantou exceo).
   */
  private boolean exitCurrentStep() {
    try {
      this.currentStep.exit(this.history);
      this.currentStep.removeListener(this.wasChangedStepListener);
      this.currentStep.removeListener(this.history);
      return true;
    }
    catch (WizardException we) {
      manageException(we);
      return false;
    }
  }

  /**
   * Cria o painel de controle dos passos. Este painel consiste no painel de
   * cabealho, no painel do passo eno painel de botes.
   *
   * @return O painel de controle dos passos.
   */
  private JPanel createControlStepPane() {
    JPanel controlStepPanel = new JPanel();
    controlStepPanel.setLayout(new BorderLayout());

    JPanel headerPanel = createHeaderPanel();
    JPanel buttonPanel = createButtonPanel();

    controlStepPanel.add(headerPanel, BorderLayout.NORTH);
    controlStepPanel.add(stepPanel, BorderLayout.CENTER);
    controlStepPanel.add(buttonPanel, BorderLayout.SOUTH);
    return controlStepPanel;
  }

  /**
   * Cria o painel de botes.
   *
   * @return O painel de botes.
   */
  private JPanel createButtonPanel() {
    ArrayList<JButton> buttons = new ArrayList<JButton>();
    JPanel browserButtonPanel = new JPanel();
    this.previousButton = new JButton(new GoPreviousAction());
    buttons.add(this.previousButton);
    browserButtonPanel.add(this.previousButton);

    this.nextButton = new JButton(new GoNextAction());
    buttons.add(this.nextButton);
    this.nextButton.setHorizontalTextPosition(SwingConstants.LEFT);
    browserButtonPanel.add(this.nextButton);

    JPanel actionButtonPanel = new JPanel();
    this.confirmButton = new JButton();
    buttons.add(this.confirmButton);
    actionButtonPanel.add(this.confirmButton);

    if (this.finishingType.equals(FinishingType.CANCELABLE)
      || this.finishingType.equals(FinishingType.CANCELABLE_AND_CLOSEABLE)) {
      this.cancelButton = new JButton(new CancelAction());
      buttons.add(this.cancelButton);
      actionButtonPanel.add(this.cancelButton);
    }
    if (this.finishingType.equals(FinishingType.CLOSEABLE) || this.finishingType
      .equals(FinishingType.CANCELABLE_AND_CLOSEABLE)) {
      this.closeButton = new JButton(new CloseAction());
      buttons.add(this.closeButton);
      actionButtonPanel.add(this.closeButton);
    }

    JButton[] array = new JButton[buttons.size()];
    GUIUtils.matchPreferredSizes(buttons.toArray(array));

    JSeparator separator = new JSeparator(SwingConstants.HORIZONTAL);

    final Insets insets = new Insets(2, 2, 2, 2);
    JPanel buttonPanel = new JPanel();
    buttonPanel.setLayout(new GridBagLayout());
    buttonPanel.add(separator, new GBC(0, 0).horizontal().width(2).center()
      .insets(insets));
    buttonPanel.add(browserButtonPanel, new GBC(0, 1).none().west().insets(
      insets));
    buttonPanel.add(actionButtonPanel, new GBC(1, 1).none().east().insets(
      insets));

    return buttonPanel;
  }

  /**
   * Cria o painel com o cabealho, onde so exibidos o ttulo e as instrues
   * do passo corrente.
   *
   * @return O painel com o cabealho.
   */
  private JPanel createHeaderPanel() {
    this.stepInstructionTextArea.setBackground(this.getBackground());
    this.stepInstructionTextArea.setEditable(false);

    final Font defaultFont = stepTitleLabel.getFont();
    stepTitleLabel.setFont(defaultFont.deriveFont(Font.BOLD).deriveFont(16.0F));

    JSeparator separator = new JSeparator(SwingConstants.HORIZONTAL);

    final Insets insets = new Insets(2, 2, 2, 2);
    JPanel headerPanel = new JPanel();
    headerPanel.setLayout(new GridBagLayout());

    int y = 0;
    headerPanel.add(stepTitleLabel, new GBC(0, y++).horizontal().insets(
      insets));
    headerPanel.add(stepInstructionTextArea, new GBC(0, y++).horizontal()
      .insets(insets));
    headerPanel.add(separator, new GBC(0, y++).horizontal().insets(insets));

    return headerPanel;
  }

  /**
   * Configura a fonte utilizada na caixa de texto da instruo do passo
   *
   * @param font nova fonte a ser utilizada na caixa de texto de instrues
   * @return wizard;
   */
  public Wizard setStepInstructionTextAreaFont(Font font) {
    this.stepInstructionTextArea.setFont(font);
    return this;
  }

  /**
   * Vai para o prximo passo.
   *
   * @throws IllegalStateException Caso o passo no seja encontrado no grupo
   *         corrente e no prximo grupo.
   */
  private void goNext() {
    /*
     * obtemos a classe do prximo passo
     */
    Class<? extends Step> stepClass = this.currentStep.getNext(this.history);
    if (stepClass.equals(this.currentStep.getClass())) {
      /*
       * Caso o prximo passo seja igual ao passo corrente, no  necessrio
       * verificar o stepGroup ou atualizar valores.
       *
       * OBS.: o fluxo de processamento prosseguir como se tivesse havido
       * realmente uma mudana de passo, ou seja, o mtodo enter() do passo
       * corrente ser novamente executado.
       */
      return;
    }
    Step nextStep;
    StepGroup nextStepGroup;
    if (!this.canSkipSteps) {
      /*
       * se no  possvel pular passos, o prximo passo s pode ser do mesmo
       * grupo que o passo corrente
       */
      nextStepGroup = this.currentGroup;
      nextStep = nextStepGroup.get(stepClass);
      if (nextStep == null) {
        int groupIndex = this.stepGroupList.indexOf(nextStepGroup);
        if (groupIndex == this.stepGroupList.size() - 1) {
          throw new IllegalStateException(MessageFormat.format(
            "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
            new Object[] { this.currentStep.getClass(), stepClass }));
        }
        nextStepGroup = this.stepGroupList.get(groupIndex + 1);
        nextStep = nextStepGroup.get(stepClass);
        if (nextStep == null) {
          throw new IllegalStateException(MessageFormat.format(
            "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
            new Object[] { this.currentStep.getClass(), stepClass }));
        }
      }
    }
    else {
      /*
       * se podemos pular passos,  necessrio primeiro descobrir a qual grupo
       * pertence o prximo passo
       */
      nextStep = null;
      nextStepGroup = null;

      for (Object o : stepGroupList) {
        nextStepGroup = (StepGroup) o;
        nextStep = nextStepGroup.get(stepClass);
        if (nextStep != null) {
          break;
        }
      }

      if (nextStep == null) {
        throw new IllegalStateException(MessageFormat.format(
          "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
          new Object[] { this.currentStep.getClass(), stepClass }));
      }
    }
    nextStep.setPrevious(currentStep);
    currentStep = nextStep;
    currentGroup = nextStepGroup;
  }

  /**
   * Recupera um passo
   *
   * @param stepClass A classe do passo
   * @return A instncia do passo, null se o passo no existir
   */
  protected Step getStep(Class<? extends Step> stepClass) {
    for (Object o : stepGroupList) {
      StepGroup stepGroup = (StepGroup) o;
      Step step = stepGroup.get(stepClass);
      if (step != null) {
        return step;
      }
    }
    return null;
  }

  /**
   * Vai para o passo anterior.
   *
   * @throws IllegalStateException Caso o passo no seja encontrado no grupo
   *         corrente e no grupo anterior.
   */
  private void goPrevious() {
    if (!this.canSkipSteps) {
      Step previousStep = this.currentStep.getPrevious();
      StepGroup stepGroup = this.currentGroup;
      if (!stepGroup.contains(previousStep)) {
        int groupIndex = this.stepGroupList.indexOf(stepGroup);
        if (groupIndex == 0) {
          throw new IllegalStateException(MessageFormat.format(
            "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
            new Object[] { this.currentStep.getClass(), previousStep
              .getClass() }));
        }
        stepGroup = this.stepGroupList.get(groupIndex - 1);
        if (!stepGroup.contains(previousStep)) {
          throw new IllegalStateException(MessageFormat.format(
            "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
            new Object[] { this.currentStep.getClass(), previousStep
              .getClass() }));
        }
      }
      this.currentStep = previousStep;
      this.currentGroup = stepGroup;
    }
    else {
      Step nextStep = null;
      StepGroup nextStepGroup = null;
      Class<? extends Step> stepClass = this.currentStep.getPrevious()
        .getClass();

      for (StepGroup sg : stepGroupList) {
        nextStepGroup = sg;
        nextStep = nextStepGroup.get(stepClass);
        if (nextStep != null) {
          break;
        }
      }

      if (nextStep == null) {
        throw new IllegalStateException(MessageFormat.format(
          "Tentativa de obter um passo inexistente.\nO passo atual  da classe {0} e o passo requerido  da classe {1}.",
          new Object[] { this.currentStep.getClass(), stepClass }));
      }

      this.currentStep = nextStep;
      this.currentGroup = nextStepGroup;
    }
  }

  /**
   * Faz o tratamento de uma exceo ocorrida. O wizard  fechado aps
   *
   * @param exception A exceo ocorrida.
   */
  private void manageException(WizardException exception) {
    notifyException(exception);
    close();
  }

  /**
   * Gera uma notificao de que o wizard foi cancelado.
   */
  private void notifyCancelled() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      final IWizardListener listener = (IWizardListener) listenerIterator
        .next();
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          listener.wasCancelled(currentStep);
        }
      });
    }
  }

  /**
   * Gera uma notificao de que o wizard foi fechado.
   */
  private void notifyClosed() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      final IWizardListener listener = (IWizardListener) listenerIterator
        .next();
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          listener.wasClosed(currentStep);
        }
      });
    }
  }

  /**
   * Gera uma notificao de que o wizard foi confirmado.
   */
  private void notifyConfirmed() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      IWizardListener listener = (IWizardListener) listenerIterator.next();
      listener.wasConfirmed(this.currentStep, this.history.getResult());
    }
  }

  /**
   * Gera uma notificao de que uma exceo foi levantada.
   *
   * @param exception A exceo que foi levantada.
   */
  private void notifyException(WizardException exception) {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      IWizardListener listener = (IWizardListener) listenerIterator.next();
      listener.wasHappenedException(this.currentStep, exception);
    }
  }

  /**
   * Ajusta a pilha de passos executado para a operao de next.
   *
   * @param oldStep passo original de onde o next ser executado.
   */
  private void adjustStackToGoNext(Step oldStep) {
    clearStack(oldStep.getClass());
    stack.push(currentStep.getClass());
    //    final Class[] array = stack.toArray(new Class[1]);
    //    System.out.println(Arrays.toString(array));
  }

  /**
   * Gera uma notificao de que o wizard foi para o prximo passo.
   */
  private void notifyGoneNext() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      IWizardListener listener = (IWizardListener) listenerIterator.next();
      listener.wasGoneNext(this.currentStep, this.history);
    }
  }

  /**
   * Ajusta a pilha de passos executado para a operao de previous.
   *
   * @param oldStep passo original de onde o previous ser executado.
   */
  private void adjustStackToGoPrevious(Step oldStep) {
    if (!resetHistoryOnChange) {
      stack.pop();
    }
    //    final Class[] array = stack.toArray(new Class[1]);
    //    System.out.println(Arrays.toString(array));
  }

  /**
   * Gera uma notificao de que o wizard voltou ao passo anterior.
   */
  private void notifyGonePrevious() {
    Iterator<?> listenerIterator = this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      IWizardListener listener = (IWizardListener) listenerIterator.next();
      listener.wasGonePrevious(this.currentStep, this.history);
    }
  }

  /**
   * Limpa o histrico a partir do passo inicial informado
   *
   * @param stepClass A classe do passo inicial
   */
  void reset(Class<? extends Step> stepClass) {
    while (!stack.isEmpty() && stepClass != stack.peek()) {
      final Class<? extends Step> poppedStepClass = stack.pop();
      final Step poppedStep = getStep(poppedStepClass);
      history.clear(poppedStep);
    }
    if (!stack.isEmpty()) {
      final Class<? extends Step> topStepClass = stack.peek();
      final Step topStep = getStep(topStepClass);
      history.clear(topStep);
    }
  }

  /**
   * Atualiza os botes de progresso. Os botes de progresso so o prximo e o
   * de confirmao.
   */
  private void updateProgressButtons() {
    this.confirmButton.setEnabled(this.currentStep.canConfirm());
    this.nextButton.setEnabled(this.currentStep.hasNext(this.history)
      && this.currentStep.canGoNext(this.history));
  }

  /**
   *
   * Faz uma iterao por todos os passos do wizard traduzindo-os. O processo de
   * traduo  efetuado pelo passo a partir dos dados existentes em data. Ao
   * terminar o mtodo, o passo corrente ser o primeiro passo cuja traduo no
   * tenha sido concluda.
   *
   * @param data Os dados que sero utilizados pelo passos para se traduzirem.
   */
  private void translateSteps(Object data) {
    Iterator<?> stepGroupIterator = stepGroupList.iterator();
    while (stepGroupIterator.hasNext()) {
      StepGroup stepGroup = (StepGroup) stepGroupIterator.next();
      Set<?> stepSet = stepGroup.getAll();
      Iterator<?> stepIterator = stepSet.iterator();
      while (stepIterator.hasNext()) {
        Step step = (Step) stepIterator.next();
        step.translate(history, data);
      }
    }
    while (currentStep.isTranslationCompleted(history)) {
      final Step oldStep = currentStep;
      goNext();
      adjustStackToGoNext(oldStep);
      notifyGoneNext();
    }
  }

  /**
   * Cria o wizard.
   *
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList Uma lista com os grupos de passos que sero exibidos
   *        pelo wizard.
   * @param globalData Dados globais do wizard que podero ser acessados por
   *        qualquer passo.
   */
  private void createWizard(Class<? extends Step> firstStepClass,
    List<? extends StepGroup> stepGroupList, Map<String, Object> globalData) {
    this.wasChangedStepListener = new WizardStepListener();
    this.listeners = new HashSet<IWizardListener>();
    this.history = new History(this, firstStepClass, this.resetHistoryOnChange);
    this.history.addGlobal(globalData);
    this.addListener(this.history);
    this.stepGroupList = Collections.unmodifiableList(stepGroupList);
    this.currentGroup = this.stepGroupList.get(0);
    this.currentStep = this.currentGroup.get(firstStepClass);

    fillGui();
  }

  /**
   * Cria um wizard.
   *
   * @param firstStepClass A classe do primeiro passo.
   * @param stepGroupList A lista de grupos de passos.
   *
   * @param globalData Dados globais a serem inseridos no histrico do wizard
   *        (se for passado null, assume-se um mapa de dados vazio).
   * @param stepData Os dados a serem traduzidos pelos passos (pode ser null e
   *        significa que no sero feitas tradues).
   * @param finishingType O tipo de finalizao suportada pelo wizard (no pode
   *        ser {@code null}).
   * @param leftDecoration objeto de decorao  esquerda do wizard que pode
   *        substituir o tpico da classe {@link ILeftDecoration} (pode ser
   *        {@code null} o que significa que no haver decorao.
   * @param resetHistoryOnChange indica se o histrico ser limpo em caso de
   *        retrocesso do wizard e mudana dos dados
   */
  public Wizard(Class<? extends Step> firstStepClass,
    List<StepGroup> stepGroupList, Map<String, Object> globalData,
    Object stepData, FinishingType finishingType,
    ILeftDecoration leftDecoration, boolean resetHistoryOnChange) {
    setHideFinishButtonsWhenDisabled(false);
    setCanSkipSteps(false);
    this.resetHistoryOnChange = resetHistoryOnChange;
    this.leftDecoration = leftDecoration;

    stack = new Stack<Class<? extends Step>>();
    stack.push(firstStepClass);

    if (finishingType == null) {
      throw new IllegalArgumentException("Finishing type cannot be null");
    }
    this.finishingType = finishingType;

    if (globalData == null) {
      globalData = Collections.emptyMap();
    }

    createWizard(firstStepClass, stepGroupList, globalData);

    if (stepData != null) {
      this.translateSteps(stepData);
    }
    this.enterCurrentStep();
  }

  /**
   * Limpa o histrico a partir do passo inicial informado
   *
   * @param oldStepClass A classe do passo inicial
   */
  private void clearStack(Class<? extends Step> oldStepClass) {
    while (!stack.isEmpty() && oldStepClass != stack.peek()) {
      stack.pop();
    }
  }

  /**
   * Habilita/desabilita o salto de passos.
   *
   * @param canSkipSteps
   * @return wizard
   */
  public Wizard setCanSkipSteps(boolean canSkipSteps) {
    this.canSkipSteps = canSkipSteps;
    return this;
  }

  /**
   * Consulta o salto de passos.
   *
   * @return indicativo
   */
  public boolean canSkipSteps() {
    return this.canSkipSteps;
  }

  /**
   * Atribui uma nova altura mnima para o wizard
   *
   * @param height a nova altura
   * @return wizard
   */
  public Wizard setMinHeight(int height) {
    minimum_height = height;
    this.setPreferredSize(this.getPreferredDimension());
    return this;
  }

  /**
   * Atribui uma nova largura mnima para o wizard
   *
   * @param width a nova largura
   * @return wizard
   */
  public Wizard setMinWidth(int width) {
    minimum_width = width;
    this.setPreferredSize(this.getPreferredDimension());
    return this;
  }

  /**
   * @return true se o painel de instrues estiver recolhido, false caso
   *         contrrio
   */
  public boolean isLeftDecorationlHidden() {
    if (!hasSplitPaneDecoration()) {
      return true;
    }
    return splitPane.getLeftComponent().getSize().width <= 0;
  }

  /**
   * Recolhe o painel de instrues
   *
   * @return wizard
   */
  public Wizard hideLeftDecoration() {
    if (hasSplitPaneDecoration()) {
      this.splitPane.setDividerLocation(0);
      final Component leftComponent = this.splitPane.getLeftComponent();
      this.splitPane.setLastDividerLocation(leftComponent
        .getPreferredSize().width);
    }
    return this;
  }

  /**
   * Exibe o painel de instrues
   *
   * @return wizard
   */
  public Wizard showLeftDecoration() {
    if (hasSplitPaneDecoration()) {
      final Component leftComponent = this.splitPane.getLeftComponent();
      this.splitPane.setDividerLocation(leftComponent.getPreferredSize().width);
    }
    return this;
  }

  /**
   * Consulta se a decorao  esquerda do wizard tem um split-pane.
   *
   * @return indicativo
   */
  public boolean hasSplitPaneDecoration() {
    return (splitPane != null);
  }

  /**
   * Consulta indicativo de ocultao de botes de trmino no caso de ficarem
   * desabilitados.
   *
   * @return indicativo
   */
  public boolean isHideFinishButtonsWhenDisabled() {
    return hideFinishButtonsWhenDisabled;
  }

  /**
   * Ajusta indicativo de ocultao de botes de trmino no caso de ficarem
   * desabilitados.
   *
   * @param flag indicativo
   * @return wizard
   */
  public Wizard setHideFinishButtonsWhenDisabled(boolean flag) {
    this.hideFinishButtonsWhenDisabled = flag;
    updateFinishButtons();
    return this;
  }
}
