package tecgraf.javautils.jexpression.parser;

import java.util.ArrayList;
import java.util.List;

import tecgraf.javautils.jexpression.exception.JExpressionSyntaxErrorException;
import tecgraf.javautils.jexpression.parser.model.BinaryOp;
import tecgraf.javautils.jexpression.parser.model.DoubleValue;
import tecgraf.javautils.jexpression.parser.model.Exp;
import tecgraf.javautils.jexpression.parser.model.Field;
import tecgraf.javautils.jexpression.parser.model.FunctionCall;
import tecgraf.javautils.jexpression.parser.model.Group;
import tecgraf.javautils.jexpression.parser.model.Index;
import tecgraf.javautils.jexpression.parser.model.Question;
import tecgraf.javautils.jexpression.parser.model.UnaryOp;
import tecgraf.javautils.jexpression.parser.model.Var;
import tecgraf.javautils.jexpression.scanner.JScanner;
import tecgraf.javautils.jexpression.scanner.Symbol;
import tecgraf.javautils.jexpression.scanner.Token;

/**
 * Analisador sinttico de expresses. Abaixo uma gramtica informal das
 * expresses:
 *
 * <ol>
 * <li>
 * {@code exp -> exp binop exp | unop exp | doubleValue | funcioncall | var |
 * group | question}
 * <li>{@code binop -> '<' | '>' | '<=' | '>=' | '!=' | '==' | '+' |
 * '-' | '*' | '/' | '-' | '.' | '||' | '&&'}
 * <li>{@code unop -> '!' | '-'}
 * <li>{@code question -> exp '?' exp ':' exp}
 * <li>{@code functioncall -> name '(' [exp << ',' exp >>] ')'}}
 * <li>{@code group -> '(' exp ')'}
 * </ol>
 *
 * <p>
 * A anlise sinttica  feita atravs do algoritmo tradicional de
 * parser-top-down <a
 * href=http://www.cs.engr.uky.edu/~lewis/essays/compilers/rec-des.html>
 * "Recursive Descent Parser"</a>. Para simplificar a prescedncia dos
 * operadores, a gramtica implementada  sutilmente diferente.
 * </p>
 *
 * <p>
 * Veja abaixo:
 *
 * <ul>
 * <li>{@code exp -> exp1 ['?' exp ':' exp]}
 * <li>{@code exp1 -> exp2 <<'||' exp2>>}
 * <li>{@code exp2 -> * exp3 <<'&&' exp3>>}
 * <li>{@code exp3 -> exp4 <<('<' | '>' | '<=' | '>=' | '!=' | '==')
 * exp4>>}
 * <li>{@code exp4 -> exp5 <<('+' | '-') exp5>>}
 * <li>{@code exp5 -> exp6 <<('*' | '/') exp6>>}
 * <li>{@code exp6 -> ('!' | '-') exp6 | exp7}
 * <li>{@code exp7 -> exp8 <<'^' exp6>>}
 * <li>{@code exp8 -> exp9 ( <<'.' name>> | <<'[' exp ']'>>)}
 * <li>{@code exp9 -> doubleValue | functioncall | var | group}
 * <li>{@code functioncall -> name '(' [exp <<',' exp>>] ')'}
 * <li>{@code var -> name}
 * <li>{@code group -> '(' exp ')'}
 * </ul>
 *
 *
 *
 * @see TokenMemoization
 * @see JScanner
 *
 * @author Tecgraf
 */
public class JParser {

  /** Construtor. */
  public JParser() {
  }

  /**
   * Faz a anlise sinttica da entrada e retorna a rvore sinttica.
   *
   * @param input entrada.
   * @return rvore sinttica.
   * @throws JExpressionSyntaxErrorException caso a expresso seja invlida.
   */
  public Exp parse(String input) throws JExpressionSyntaxErrorException {
    if (input == null) {
      throw new IllegalArgumentException("input no pode ser nulo.");
    }
    TokenMemoization m = new TokenMemoization(input);
    Exp exp = exp(m);

    if (exp == null) {
      String f = "'%s' No  uma expresso vlida";
      String msg = String.format(f, input);
      throw new JExpressionSyntaxErrorException(msg, 0);
    }

    if (m.hasMoreTokens()) {
      Token token = m.token();
      String f = "Expresso invlida a partir de '%s'";
      String msg = String.format(f, token.getValue());
      throw new JExpressionSyntaxErrorException(msg, token.getLineNumber());
    }
    return exp;
  }

  /**
   * Anlise do no-terminal raz da gramtica.
   * <p>
   * exp &rarr; exp1 [? exp : exp]
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp1(m);

    if (m.isTerminal(Symbol.QUESTION)) {
      Token question = m.token();
      m.commit();
      Exp then = exp(m);

      if (exp == null || then == null) {
        String msg = "Faltam operandos para o operador '?'";
        throw new JExpressionSyntaxErrorException(msg, question
          .getLineNumber());
      }

      if (!m.isTerminal(Symbol.COLON)) {
        String msg = "Falta o operador ':' para o dado '?'";
        throw new JExpressionSyntaxErrorException(msg, question
          .getLineNumber());
      }
      m.commit();

      Exp otherwise = exp(m);
      if (otherwise == null) {
        String msg = "Faltam operandos para o operador ':'";
        throw new JExpressionSyntaxErrorException(msg, question
          .getLineNumber());
      }

      exp = new Question(exp, then, otherwise);
    }

    return exp;
  }

  /**
   * Anlise de expresses com '||'.
   * <p>
   * exp1 &rarr; exp2 {'||' exp2}
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp1(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp2(m);

    while (m.isTerminal(Symbol.OR)) {
      Token or = m.token();
      m.commit();
      Exp second = exp2(m);
      if (exp == null || second == null) {
        String msg = "Faltam operandos para o operador '||'";
        throw new JExpressionSyntaxErrorException(msg, or.getLineNumber());
      }
      exp = new BinaryOp(exp, Symbol.OR, second);
    }

    return exp;
  }

  /**
   * Anlise de expresses com '&amp;&amp;'.
   * <p>
   * exp2 &rarr; exp3 {'&amp;&amp;' exp3}
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp2(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp3(m);

    while (m.isTerminal(Symbol.AND)) {
      Token and = m.token();
      m.commit();
      Exp second = exp3(m);
      if (exp == null || second == null) {
        String msg = "Faltam operandos para o operador '&&'";
        throw new JExpressionSyntaxErrorException(msg, and.getLineNumber());
      }
      exp = new BinaryOp(exp, Symbol.AND, second);
    }

    return exp;
  }

  /**
   * Anlise de expresses relacionais.
   * <p>
   * exp3 &rarr; exp4 {('&lt;' | '&gt;' | '&lt;=' | '&gt;=' | '!=' | '==') exp4}
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp3(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp4(m);

    while (m.isTerminal(Symbol.LOWER, Symbol.GREATER, Symbol.LOWER_EQ,
      Symbol.GREATER_EQ, Symbol.NOT_EQUAL, Symbol.EQUAL)) {
      Token op = m.token();
      m.commit();
      Exp second = exp4(m);
      if (exp == null || second == null) {
        String f = "Faltam operandos para o operador '%s'";
        String graphic = op.getType().graphic;
        String msg = String.format(f, graphic);
        throw new JExpressionSyntaxErrorException(msg, op.getLineNumber());
      }

      exp = new BinaryOp(exp, op.getType(), second);
    }
    return exp;
  }

  /**
   * Anlise das expresses aritmticas '+' e '-'.
   * <p>
   * exp4 &rarr; exp5 {('+' | '-') exp5}
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp4(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp5(m);

    while (m.isTerminal(Symbol.PLUS, Symbol.MINUS)) {
      Token op = m.token();
      m.commit();
      Exp second = exp5(m);
      if (exp == null || second == null) {
        String f = "Faltam operandos para o operador '%s'";
        String graphic = op.getType().graphic;
        String msg = String.format(f, graphic);
        throw new JExpressionSyntaxErrorException(msg, op.getLineNumber());
      }

      exp = new BinaryOp(exp, op.getType(), second);
    }
    return exp;
  }

  /**
   * Anlise das expresses aritmticas '*' e '/'.
   * <p>
   * exp5 &rarr; exp6 {('*' | '/') exp6}
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp5(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp6(m);

    while (m.isTerminal(Symbol.TIMES, Symbol.DIVIDE)) {
      Token op = m.token();
      m.commit();
      Exp second = exp6(m);
      if (exp == null || second == null) {
        String f = "Faltam operandos para o operador '%s'";
        String graphic = op.getType().graphic;
        String msg = String.format(f, graphic);
        throw new JExpressionSyntaxErrorException(msg, op.getLineNumber());
      }

      exp = new BinaryOp(exp, op.getType(), second);
    }
    return exp;
  }

  /**
   * Anlise de expresses com operadores unrios.
   * <p>
   * exp6 &rarr; ('!' | '-') exp6 | exp7
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp6(TokenMemoization m) throws JExpressionSyntaxErrorException {
    if (m.isTerminal(Symbol.NOT, Symbol.MINUS)) {
      Token op = m.token();
      m.commit();

      Exp exp = exp6(m);
      if (exp == null) {
        String f = "Falta operando para o operador unrio '%s'";
        String graphic = op.getType().graphic;
        String msg = String.format(f, graphic);
        throw new JExpressionSyntaxErrorException(msg, op.getLineNumber());
      }
      return new UnaryOp(op.getType(), exp);
    }
    return exp7(m);
  }

  /**
   * Anlise de expresses aritmticas '^'.
   * <p>
   *
   * <b>Nota:</b> O operador '^' possui associatividade a direita.
   * <p>
   * exp7 &rarr; exp8 {'^' exp6}
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp7(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp8(m);
    if (exp == null) {
      return null;
    }

    List<Exp> exps = new ArrayList<>();
    exps.add(exp);

    while (m.isTerminal(Symbol.POW)) {
      Token pow = m.token();
      m.commit();
      Exp exponent = exp6(m);
      if (exponent == null) {
        String msg = "Faltam operandos para o operador '^'";
        throw new JExpressionSyntaxErrorException(msg, pow.getLineNumber());
      }
      exps.add(exponent);
    }

    if (exps.size() == 1) {
      return exps.get(0);
    }

    // associando a direita.
    exp = null;
    for (int i = exps.size() - 1; i > 0; i--) {
      Exp second = exps.get(i);
      Exp first = exps.get(i - 1);
      exp = new BinaryOp(first, Symbol.POW, second);
      exps.set(i - 1, exp);
    }
    return exp;
  }

  /**
   * Anlise de acessos a campos.
   * <p>
   * exp8 &rarr; exp9 ({'.' name} | {'[' exp ']'})
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp8(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = exp9(m);
    if (exp == null) {
      return null;
    }

    while (m.isTerminal(Symbol.DOT, Symbol.L_BRACKET)) {
      while (m.isTerminal(Symbol.DOT)) {
        int lineNumber = m.token().getLineNumber();
        m.commit();

        if (!m.isTerminal(Symbol.NAME)) {
          String msg = "Faltam operandos para o operador '.'";
          throw new JExpressionSyntaxErrorException(msg, lineNumber);
        }
        Token name = m.token();
        m.commit();
        exp = new Field(exp, name.getValue());
      }

      while (m.isTerminal(Symbol.L_BRACKET)) {
        Token op = m.token();
        m.commit();
        Exp second = exp(m);
        if (exp == null || second == null) {
          String f = "Faltam ndice do valor '%s'";
          String msg = String.format(f, exp);
          throw new JExpressionSyntaxErrorException(msg, op.getLineNumber());
        }

        if (!m.isTerminal(Symbol.R_BRACKET)) {
          String msg = "Colchetes desbalanceados ']'";
          throw new JExpressionSyntaxErrorException(msg, op.getLineNumber());
        }
        m.commit();
        exp = new Index(exp, second);
      }
    }

    return exp;
  }

  /**
   * Anlise dos terminais da linguagem.
   * <p>
   * exp9 &rarr; doubleValue | functioncall | var | group
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp exp9(TokenMemoization m) throws JExpressionSyntaxErrorException {
    Exp exp = doubleValue(m);
    if (exp != null) {
      return exp;
    }

    exp = functioncall(m);
    if (exp != null) {
      return exp;
    }

    exp = var(m);
    if (exp != null) {
      return exp;
    }
    exp = group(m);
    if (exp != null) {
      return exp;
    }

    return null;
  }

  /**
   * Anlise de valores numricos do tipo double.
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp doubleValue(TokenMemoization m)
    throws JExpressionSyntaxErrorException {
    if (!m.isTerminal(Symbol.DOUBLE)) {
      return null;
    }
    Token token = m.token();
    m.commit();
    String value = token.getValue();
    return new DoubleValue(Double.valueOf(value));
  }

  /**
   * Anlise de chamadas de funo.
   * <p>
   * functioncall &rarr; name '(' [exp {',' exp}] ')'
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp functioncall(TokenMemoization m)
    throws JExpressionSyntaxErrorException {
    if (!m.isTerminal(Symbol.NAME)) {
      return null;
    }

    Token name = m.token();
    m.commit();

    if (!m.isTerminal(Symbol.L_PAREN)) {
      m.rollback();
      return null;
    }
    m.commit();

    //verificando lista de parametros.
    List<Exp> params = new ArrayList<>();
    Exp exp = exp(m);
    if (exp != null) {
      params.add(exp);
      while (m.isTerminal(Symbol.COMMA)) {
        m.commit();
        exp = exp(m);

        if (exp == null) {
          String f = "Falta parmetro aps ',' na chamada de funo '%s'";
          String msg = String.format(f, name.getValue());
          throw new JExpressionSyntaxErrorException(msg, name.getLineNumber());
        }
        params.add(exp);
      }
    }

    if (!m.isTerminal(Symbol.R_PAREN)) {
      String f = "Falta ')' para a dada chamada de funo '%s'";
      String msg = String.format(f, name.getValue());
      throw new JExpressionSyntaxErrorException(msg, name.getLineNumber());
    }
    m.commit();

    return new FunctionCall(name.getValue(), params);
  }

  /**
   * Anlise de variveis.
   * <p>
   * var &rarr; name
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Var var(TokenMemoization m) throws JExpressionSyntaxErrorException {
    if (!m.isTerminal(Symbol.NAME)) {
      return null;
    }
    Token name = m.token();
    m.commit();

    return new Var(name.getValue());
  }

  /**
   * Anlise de agrupamentos.
   * <p>
   * group &rarr; '(' exp ')'
   *
   * @param m memorizador de tokens.
   * @return expresso resultante.
   * @throws JExpressionSyntaxErrorException em caso de erro de sintaxe.
   */
  private Exp group(TokenMemoization m) throws JExpressionSyntaxErrorException {
    if (!m.isTerminal(Symbol.L_PAREN)) {
      return null;
    }
    int lineNumber = m.token().getLineNumber();
    m.commit();

    Exp exp = exp(m);
    if (exp == null) {
      String msg = "Expresso esperada aps '('";
      throw new JExpressionSyntaxErrorException(msg, lineNumber);
    }

    if (!m.isTerminal(Symbol.R_PAREN)) {
      String msg = "Parnteses desbalanceados ')'";
      throw new JExpressionSyntaxErrorException(msg, lineNumber);
    }
    m.commit();

    return new Group(exp);
  }
}
