package tecgraf.javautils.jexpression.scanner;

import java.util.regex.Matcher;

import tecgraf.javautils.jexpression.parser.JParser;

/**
 * Analisador lxico das expresses. Para saber a ordem dos tokens a serem
 * verificados, consulte a enumerao {@link Symbol}.
 * 
 * Em caso de tokens desconhecidos, o {@link JScanner} constri um token com o
 * tipo {@link Symbol#UNKNOWN} ao invs de lanar excees. Essa estratgia
 * permite a gerao de erros mais detalhados durante a anlise sinttica feita
 * pelo {@link JParser}.
 * 
 * @see Symbol
 * @see JParser
 * 
 * @author Tecgraf
 */
public class JScanner {

  /** Texto de entrada. */
  private String input;

  /** Nmero da linha atual. */
  private int lineNumber;

  /** ndice do texto. */
  private int index;

  /**
   * Construtor.
   * 
   * @param input entrada.
   */
  public JScanner(String input) {
    if (input == null) {
      throw new IllegalArgumentException("input no pode ser nulo.");
    }
    this.input = input;
    this.lineNumber = 1;
    this.index = 0;
  }

  /**
   * Retorna o prximo token ou null caso a entrada tenha terminado.
   * 
   * @return prximo token ou null.
   */
  public Token nextToken() {
    normalize();

    if (over()) {
      return null;
    }

    for (Symbol type : Symbol.values()) {
      Matcher m = type.pattern.matcher(input);

      m.region(index, input.length());
      if (m.find()) {
        index = m.end();

        return new Token(type, lineNumber, m.group());
      }
    }
    return null;
  }

  /**
   * Mtodo que pula os espaos em branco, quebra de linha e tabulao. A cada
   * quebra de linha incrementamos o nmero da linha.
   */
  private void normalize() {
    if (over()) {
      return;
    }

    char c = input.charAt(index);
    if (c == ' ' || c == '\n' || c == '\t') {
      index++;
      if (c == '\n') {
        lineNumber++;
      }
      normalize();
    }
  }

  /**
   * True se j chegamos ao final da entrada, false caso contrrio.
   * 
   * @return flag que define se chegamos ao final da entrada.
   */
  public boolean over() {
    return input.length() == 0 || index == input.length();
  }
}
