package tecgraf.javautils.pdfviewer.core;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingWorker;

import com.sun.pdfview.PDFPage;

/**
 * Painel que renderiza uma pgina PDF dentro de um {@link JScrollPane}
 * 
 * @author Tecgraf
 */
class PDFRendererPagePanel extends JPanel {

  /**
   * Pinta a imagem do PDF em um JPanel
   * 
   * @author Tecgraf
   */
  class PDFImagePanel extends JPanel {
    /** Image corrente no painel */
    Image currentImage;

    /** offset x para centralizar a imagem */
    int offsetX;

    /** offset Y para centralizar a imagem */
    int offsetY;

    /**
     * Atribui a imagem corrente, pode ser nulo
     * 
     * @param image a ser exibida ou null
     */
    public void setImage(Image image) {
      if (currentImage != image) {
        currentImage = image;
        if (currentImage != null) {
          setPreferredSize(new Dimension(image.getWidth(null), image
            .getHeight(null)));
          //          System.out.println("Width: " + image.getWidth(null) + " height "
          //            + image.getHeight(null));
        }
        else {
          setPreferredSize(new Dimension(1, 1));
        }
        revalidate();
        repaint();

      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void paint(Graphics g) {
      super.paint(g);
      if (currentImage == null) {
        return;
      }

        // desenha a imagem
        int imageWidth = currentImage.getWidth(null);
        int imageHeight = currentImage.getHeight(null);

        // desenha centralizado na pgina, caso o Panel for maior que a imagem
        if (getSize().width > imageWidth) {
          offsetX = (getSize().width - imageWidth) / 2;
        }
        else {
          offsetX = 0;
        }

        if (getSize().height > imageHeight) {
          offsetY = (getSize().height - imageHeight) / 2;
        }
        else {
          offsetY = 0;
        }

        g.drawImage(currentImage, offsetX, offsetY, this);

        // desenha as linhas da borda
        g.drawLine(offsetX, offsetY, offsetX, offsetY + imageHeight);
        g.drawLine(offsetX, offsetY, offsetX + imageWidth, offsetY);
        g.drawLine(offsetX, offsetY + imageHeight, offsetX + imageWidth,
          offsetY + imageHeight);
        g.drawLine(offsetX + imageWidth, offsetY, offsetX + imageWidth, offsetY
          + imageHeight);
    }
  }

  /**
   * SwingWorker que gera a imagem do PDF e atribui ao PDFImagePanel corrente
   * 
   * @author Tecgraf
   */
  class LoadImageWorker extends SwingWorker<Image, Void> {

    /** Dimenso da imagem a ser gerada */
    final Dimension dimension;

    /**
     * Construtor principal
     * 
     * @param dimension dimenso da imagem a ser gerada
     */
    public LoadImageWorker(Dimension dimension) {
      super();
      this.dimension = dimension;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Image doInBackground() throws Exception {

      return currentPage.getImage(dimension.width, dimension.height, null,
        null, true, true);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void done() {
      try {
        imagePanel.setImage(get());
      }
      catch (Exception ignore) {
        // exceo caso o worker tenha sido cancelado, ento deve ser ignorada
      }
    }

  }

  /** PDFImagePanel que desenha a imagem do PDF */
  final private PDFImagePanel imagePanel = new PDFImagePanel();

  /** Pgina corrente */
  private PDFPage currentPage;

  /** Tamanho corrente da pgina */
  private Dimension pageSize;

  /** Tamanho do zoom de 100% */
  private Dimension defaultPageSize;

  /** Porcentagem do zoom */
  private double zoomPercentage;

  /** ltimo LoadImageWorker utilizado */
  private LoadImageWorker lastImageLoader = null;

  /**
   * Construtor default
   */
  PDFRendererPagePanel() {
    super(new BorderLayout());
    add(imagePanel, BorderLayout.CENTER);
    //scrollPane.setViewportView(imagePanel);
  }

  /**
   * Exibe uma pgina ou nulo, para no exibir nenhuma pgina
   * 
   * @param page pgina a ser exibida ou null
   */
  void showPage(PDFPage page) {
    currentPage = page;

    if (currentPage != null) {
      if (pageSize == null) {
        defaultPageSize =
          new Dimension((int) page.getPageBox().getWidth(), (int) page
            .getPageBox().getHeight());
        pageSize = new Dimension(defaultPageSize.width, defaultPageSize.height);
        zoomPercentage = 100;

      }
      loadCurrentPageWithCurrentPageSize();
    }
    else {
      pageSize = null;
      imagePanel.setImage(null);
    }
  }

  /**
   * Cria um worker para gerar uma imagem da pgina corrente com o tamanho
   * corrente
   */
  protected void loadCurrentPageWithCurrentPageSize() {
    if (lastImageLoader != null && !lastImageLoader.isDone()) {
      lastImageLoader.cancel(true);
    }

    if (pageSize.width > 0 && pageSize.height > 0) {
      lastImageLoader = new LoadImageWorker(pageSize);
      lastImageLoader.execute();
    }
  }

  /**
   * Faz o zoom de acordo com um fator. 0 fator deve ser maior que zero. Se o
   * fator for maior que 1,  feito o ZoomIn. Se o fator for menor que 1, 
   * feito o ZoomOut.
   * 
   * @param zoomFactor fator de zoom
   */
  protected void zoomByFactor(double zoomFactor) {
    pageSize.width = (int) (pageSize.width * zoomFactor);
    pageSize.height = (int) (pageSize.height * zoomFactor);
    zoomPercentage = 100d * defaultPageSize.width / pageSize.width;
    loadCurrentPageWithCurrentPageSize();
  }

  /**
   * FAz o zoom de acordo com uma porcentagem
   * 
   * @param percentage porcentagem para ser feita o zoom
   */
  protected void setZoom(double percentage) {
    zoomPercentage = percentage;
    applyZoom();
  }

  /**
   * Aplica o zoom definido por zoomPercentage
   */
  protected void applyZoom() {
    double normalizedZoom = zoomPercentage / 100;
    pageSize.width = (int) (defaultPageSize.width * normalizedZoom);
    pageSize.height = (int) (defaultPageSize.height * normalizedZoom);
    loadCurrentPageWithCurrentPageSize();
  }

  /**
   * Retorna o valor do zoom corrente em percentual
   * 
   * @return o valor do zoom corrente em percentual
   */
  protected double getZoom() {
    return zoomPercentage;
  }

  /**
   * Atribui o tamanho corrente
   * 
   * @param pageSize tamanho a ser atribuido
   */
  protected void setPageSize(Dimension pageSize) {
    // atualiza o zoom
    zoomPercentage = 100d * defaultPageSize.width / pageSize.width;
    this.pageSize = pageSize;
  }

  /**
   * Retorna o tamanho da pgina ou nulo
   * 
   * @return o tamanho da pgina ou nulo
   */
  protected Dimension getPageSize() {
    return pageSize;
  }

  /**
   * Retorna o tamanho do zoom 100%
   * 
   * @return o tamanho do zoom 100%
   */
  Dimension getDefaultPageSize() {
    return defaultPageSize;
  }

}
