package tecgraf.javautils.pdfviewer.viewer;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;

import tecgraf.javautils.pdfviewer.core.PDFCorePanel;
import tecgraf.javautils.pdfviewer.core.PDFCorePanel.ZoomFitPolicy;
import tecgraf.javautils.pdfviewer.core.PDFCoreThumbnailPanel;
import tecgraf.javautils.pdfviewer.core.PDFDocument;
import tecgraf.javautils.pdfviewer.core.actions.PDFAbstractDocumentOpenAwareAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFAbstractPageChangeAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFFitHeightAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFFitWholePageAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFFitWidthAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFNextPageAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFPrevPageAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFZoomInAction;
import tecgraf.javautils.pdfviewer.core.actions.PDFZoomOutAction;
import tecgraf.javautils.pdfviewer.core.listeners.PDFDocumentOpenCloseListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFPageChangeRequestedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFPageChangedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomChangeRequestedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomChangedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomFitPolicyChangedListener;
import tecgraf.javautils.pdfviewer.viewer.actions.PDFCloseAction;
import tecgraf.javautils.pdfviewer.viewer.actions.PDFShowThumbnailsAction;
import tecgraf.javautils.pdfviewer.viewer.listeners.PDFThumbnailsVisibilityListener;

/**
 * Painel que exibe um PDF dado um {@link InputStream}. Contm uma
 * {@link JToolBar} para manipular o PDF.
 * 
 * A toolbar pode ser acessada pelo mtodo {@link #getPDFToolBar()}, podendo ser
 * adicionada a um painel externo, fazendo com que seja removida deste painel ou
 * podendo ser ocultada
 * 
 * @author Tecgraf
 */
public class PDFViewerPanel extends JPanel {

  /** Painel que exibe o PDF */
  final private PDFCorePanel pdfCorePanel;

  /** Painel de ThumbNails */
  final private PDFCoreThumbnailPanel coreThumbnailPanel =
    new PDFCoreThumbnailPanel();

  /** ToolBar para manipular o PDF */
  final private JToolBar pdfToolBar = new JToolBar();

  /** SideBar splitPane */
  JSplitPane sideBarSplitPane = new JSplitPane();

  /** Ao de fit de WHOLE_PAGE */
  final private PDFAbstractDocumentOpenAwareAction fitWholePageAction;

  /** Ao de fit de WIDTH */
  final private PDFAbstractDocumentOpenAwareAction fitWidthAction;

  /** Ao de fit de HEIGHT */
  final private PDFAbstractDocumentOpenAwareAction fitHeightAction;

  /** Ao de zoom in */
  final private PDFAbstractDocumentOpenAwareAction zoomInAction;

  /** Ao de zoom out */
  final private PDFAbstractDocumentOpenAwareAction zoomOutAction;

  /** Ao de ir para a pgina anterior */
  final private PDFAbstractPageChangeAction prevPageAction;

  /** Ao de ir para a pgina seguinte */
  final private PDFAbstractPageChangeAction nextPageAction;

  /** Ao que fecha o PDF corrente */
  final private PDFCloseAction pdfCloseAction;

  /** Ao que exibe ou oculta a sidebar com Thubnails */
  final private PDFShowThumbnailsAction showThumbnailsAction;

  /** Painel que permite ir direto a uma pgina pelo nmero */
  final private PageNumberPanel pageNumberPanel;

  /** Grupo de botes das aes de FIT */
  final private ButtonGroup fitButtonGroup = new ButtonGroup();

  /** ComboBox com o valor do zoom corrente */
  final private ZoomValueComboBox zoomValueCombo = new ZoomValueComboBox();

  /** Lista de {@link PDFThumbnailsVisibilityListener} */
  final private List<PDFThumbnailsVisibilityListener> thumbnailsVisibilityListeners =
          new ArrayList<>();

  /** atributo que controla se os thubnails esto visveis */
  boolean thumbnailsVisible = false;

  /**
   * Construtor
   * 
   * @param locale usado para as mensagens
   */
  public PDFViewerPanel(Locale locale) {
    this(new PDFViewerBundle(locale));
  }

  /**
   * Construtor padro
   * 
   * @param bundle recurso de i18n.
   */
  public PDFViewerPanel(ResourceBundle bundle) {
    this.pdfCorePanel = new PDFCorePanel(bundle);
    // aes inicializadas aqui pois so final
    fitWholePageAction = new PDFFitWholePageAction(pdfCorePanel);
    fitWidthAction = new PDFFitWidthAction(pdfCorePanel);
    fitHeightAction = new PDFFitHeightAction(pdfCorePanel);
    zoomInAction = new PDFZoomInAction(pdfCorePanel);
    zoomOutAction = new PDFZoomOutAction(pdfCorePanel);
    prevPageAction = new PDFPrevPageAction(pdfCorePanel);
    nextPageAction = new PDFNextPageAction(pdfCorePanel);
    pdfCloseAction = new PDFCloseAction(pdfCorePanel, this);
    showThumbnailsAction = new PDFShowThumbnailsAction(pdfCorePanel, this);
    // fim da inicializao das aes
    setPreferredSize(new Dimension(600, 400));
    this.pageNumberPanel = new PageNumberPanel(pdfCorePanel);
    addListeners();
    addActionsTo(pdfToolBar);
    layout(pdfCorePanel, pdfToolBar, coreThumbnailPanel);

  }

  /**
   * Adiciona os listeners ao {@link PDFCorePanel}
   */
  private void addListeners() {
    addPageChangeListener(nextPageAction);
    addPageChangeListener(prevPageAction);
    addPageChangeListener(pageNumberPanel);
    addPDFDocumentOpenCloseListener(pageNumberPanel);
    addPDFDocumentOpenCloseListener(prevPageAction);
    addPDFDocumentOpenCloseListener(nextPageAction);
    addPDFDocumentOpenCloseListener(zoomInAction);
    addPDFDocumentOpenCloseListener(zoomOutAction);
    addPDFDocumentOpenCloseListener(fitWholePageAction);
    addPDFDocumentOpenCloseListener(fitHeightAction);
    addPDFDocumentOpenCloseListener(fitWidthAction);
    addPDFZoomChangedListener(zoomValueCombo);
    addPDFDocumentOpenCloseListener(zoomValueCombo);
    addPDFDocumentOpenCloseListener(pdfCloseAction);
    addPDFDocumentOpenCloseListener(showThumbnailsAction);
    addPDFDocumentOpenCloseListener(coreThumbnailPanel);
    addPDFDocumentOpenCloseListener(new PDFDocumentOpenCloseListener() {
      @Override
      public void documentOpened(PDFCorePanel pdfCorePanel) {
        applyThumbsVisible();
      }

      @Override
      public void documentClosed(PDFCorePanel source) {
        hideThumbnails();
      }
    });

    addPageChangeListener(coreThumbnailPanel);

    coreThumbnailPanel
      .addPDFPageChangeRequestListener(this::setPage);
    zoomValueCombo
      .addPDFZoomChangeRequestedListener(pdfCorePanel::setZoom);

  }

  /**
   * Adiciona as actions  {@link JToolBar}
   * 
   * @param toolBar a ser adicionadas as actions
   */
  protected void addActionsTo(JToolBar toolBar) {
    toolBar.setFloatable(false);
    toolBar.add(getPrevPageAction());
    toolBar.add(getNextPageAction());
    toolBar.addSeparator();
    toolBar.add(pageNumberPanel);
    toolBar.addSeparator();
    toolBar.add(getZoomInAction());
    toolBar.add(getZoomOutAction());
    toolBar.add(zoomValueCombo);
    toolBar.addSeparator();

    final JToggleButton fitWholePageButton =
      new JToggleButton(getFitWholePageAction());
    final JToggleButton fitWidthButton = new JToggleButton(getFitWidthAction());
    final JToggleButton fitHeightButton =
      new JToggleButton(getFitHeightAction());
    fitButtonGroup.add(fitHeightButton);
    fitButtonGroup.add(fitWidthButton);
    fitButtonGroup.add(fitWholePageButton);

    pdfCorePanel
      .addPDFZoomFitPolicyChangedListener(new PDFZoomFitPolicyChangedListener() {

        @Override
        public void zoomFitPolicyChanged(ZoomFitPolicy policy) {
          if (policy == ZoomFitPolicy.FREE) {
            fitButtonGroup.clearSelection();
          }
          else if (policy == ZoomFitPolicy.WIDTH) {
            fitWidthButton.setSelected(true);
          }
          else if (policy == ZoomFitPolicy.HEIGHT) {
            fitHeightButton.setSelected(true);
          }
          else if (policy == ZoomFitPolicy.WHOLE_PAGE) {
            fitWholePageButton.setSelected(true);
          }

        }
      });

    toolBar.add(fitWholePageButton);
    toolBar.add(fitWidthButton);
    toolBar.add(fitHeightButton);

    fitWholePageButton.setText(null);
    fitWidthButton.setText(null);
    fitHeightButton.setText(null);

    toolBar.addSeparator();

    final JToggleButton showThumbnailsButton =
      new JToggleButton(showThumbnailsAction);
    showThumbnailsButton.setText(null);
    //    showThumbnailsButton.addItemListener(showThumbnailsAction);
    toolBar.add(showThumbnailsButton);

    addPDFThumbnailsVisibilityListeners(showThumbnailsButton::setSelected);

  }

  /**
   * Faz o layout dos componentes neste JPanel
   * 
   * @param pageView visualizao da pgina
   * @param pdfToolBar toolbar para manipular o PDF
   * @param thumbnailPanel painel de thumbnail,
   */
  protected void layout(Component pageView, JToolBar pdfToolBar,
    Component thumbnailPanel) {
    setLayout(new BorderLayout());
    add(pdfToolBar, BorderLayout.NORTH);
  }

  /**
   * Carrega um PDF de um {@link InputStream}. Este {@link InputStream} no 
   * fechado pelo {@link PDFViewerPanel} e deve ser fechado por quem abriu. Uma
   * cpia em memria  feita pelo {@link PDFCorePanel} e o {@link InputStream}
   * no ser mais necessrio quando este mtodo for retornado.
   * 
   * @param inputStream inputStream para ler o PDF
   * @throws IOException
   */
  /**
   * Carrega um documento PDF no visualizador
   * 
   * @param document documento PDF
   */
  public void loadDocument(PDFDocument document) {
    pdfCorePanel.loadDocument(document);
  }

  /**
   * Consulta tamanho da pgina do cdocumento.
   * 
   * @return tamanho
   */
  public Dimension getDocumentPageSize() {
    final Dimension size = pdfCorePanel.getDocumentPageSize();
    return size;
  }

  /**
   * Adiciona um {@link PDFPageChangedListener}  lista de listeners
   * 
   * @param listener {@link PDFPageChangedListener} a ser adicionado
   */
  protected void addPageChangeListener(PDFPageChangedListener listener) {
    pdfCorePanel.addPageChangeListener(listener);
  }

  /**
   * Adiciona um {@link PDFDocumentOpenCloseListener}  lista de listeners
   * 
   * @param listener {@link PDFDocumentOpenCloseListener} a ser adicionado
   */
  public void addPDFDocumentOpenCloseListener(
    PDFDocumentOpenCloseListener listener) {
    pdfCorePanel.addPDFDocumentOpenCloseListener(listener);
  }

  /**
   * Adiciona um {@link PDFThumbnailsVisibilityListener}  lista de listeners
   * 
   * @param listener {@link PDFThumbnailsVisibilityListener} a ser adicionado
   */
  public void addPDFThumbnailsVisibilityListeners(
    PDFThumbnailsVisibilityListener listener) {
    thumbnailsVisibilityListeners.add(listener);
  }

  /**
   * Atribui a pgina ser exibida, este nmero  _1_ based, isto , comea em 1.
   * 
   * @param pageNumber nmero da pgina, deve ser maior ou igual a 1 e menor ou
   *        igual ao nmero de pginas total
   */
  public void setPage(int pageNumber) {
    pdfCorePanel.setPage(pageNumber);
  }

  /**
   * Retorna verdadeiro se  possvel ir para a prxima pgina, isto , a pgina
   * corrente no  ltima, falso caso contrrio.
   * 
   * @return verdadeiro se  possvel ir para a prxima pgina, isto , a pgina
   *         corrente no  ltima, falso caso contrrio.
   */
  public boolean canGoToNextPage() {
    return pdfCorePanel.canGoToNextPage();
  }

  /**
   * Retorna verdadeiro se  possvel ir para a pgina anterior, isto , a
   * pgina corrente no  primeira, falso caso contrrio.
   * 
   * @return verdadeiro se  possvel ir para a pgina anterior, isto , a
   *         pgina corrente no  primeira, falso caso contrrio.
   */
  public boolean canGoToPrevPage() {
    return pdfCorePanel.canGoToPrevPage();
  }

  /**
   * Vai para prxima pgina
   * 
   * @return verdadeiro se foi possvel mudar de pgina, falso caso contrrio
   */
  public boolean goToNextPage() {
    return pdfCorePanel.goToNextPage();
  }

  /**
   * Vai para a pgina anterior
   * 
   * @return verdade se foi possvel mudar de pgina, falso caso contrrio
   */
  public boolean goToPrevPage() {
    return pdfCorePanel.goToPrevPage();
  }

  /**
   * Atribui e aplica a poltica de zoom
   * 
   * @param fitPolicy {@link ZoomFitPolicy} que representa a poltica de zoom
   */
  public void setZoomFitPolicy(ZoomFitPolicy fitPolicy) {
    pdfCorePanel.setZoomFitPolicy(fitPolicy);
  }

  /**
   * Retorna o nmero total de pginas do documento corrente
   * 
   * @return o nmero total de pginas do documento corrente
   */
  public int getTotalPageNumber() {
    return pdfCorePanel.getTotalPageNumber();
  }

  /**
   * Fecha o PDF.
   */
  public void closePDF() {
    if (pdfCorePanel.hasOpenFile()) {
      pdfCorePanel.closePDF();
    }
  }

  /**
   * Retorna a ao de fit de WIDTH
   * 
   * @return a ao de fit de WIDTH
   */
  public Action getFitWidthAction() {
    return fitWidthAction;
  }

  /**
   * Retorna a ao de fit de HEIGHT
   * 
   * @return a ao de fit de HEIGHT
   */
  public Action getFitHeightAction() {
    return fitHeightAction;
  }

  /**
   * Retorna a ao de fit de WHOLE_PAGE
   * 
   * @return a ao de fit de WHOLE_PAGE
   */
  public Action getFitWholePageAction() {
    return fitWholePageAction;
  }

  /**
   * s Retorna a ao de zoom in
   * 
   * @return a ao de zoom in
   */
  public Action getZoomInAction() {
    return zoomInAction;
  }

  /**
   * Retorna a ao de zoom out
   * 
   * @return a ao de zoom out
   */
  public Action getZoomOutAction() {
    return zoomOutAction;
  }

  /**
   * Retorna a ao de ir para a pgina anterior
   * 
   * @return a ao de ir para a pgina anterior
   */
  public Action getPrevPageAction() {
    return prevPageAction;
  }

  /**
   * Retorna a ao de ir para a prxima pgina
   * 
   * @return a ao de ir para a prxima pgina
   */
  public Action getNextPageAction() {
    return nextPageAction;
  }

  /**
   * Retorna a ao de fechar o documento corrente
   * 
   * @return a ao de fechar o documento corrente
   */
  public Action getPDFCloseAction() {
    return pdfCloseAction;
  }

  /**
   * Retorna a ao de exibir/ocultar o sidebar com thumbnails
   * 
   * @return a ao de exibir/ocultar o sidebar com thumbnails
   */
  public Action getPDFShowThumbnailsAction() {
    return showThumbnailsAction;
  }

  /**
   * Retorna a {@link JToolBar} usada para manipular o PDF
   * 
   * @return a {@link JToolBar} usada para manipular o PDF
   */
  public JToolBar getPDFToolBar() {
    return pdfToolBar;
  }

  /**
   * Adiciona um {@link PDFZoomChangedListener}  lista de listeners
   * 
   * @param listener novo listener
   */
  public void addPDFZoomChangedListener(PDFZoomChangedListener listener) {
    pdfCorePanel.addPDFZoomChangedListener(listener);
  }

  /**
   * Adiciona um {@link PDFZoomFitPolicyChangedListener}  lista de listeners
   * 
   * @param listener novo listener
   */
  public void addPDFZoomFitPolicyChangedListener(
    PDFZoomFitPolicyChangedListener listener) {
    pdfCorePanel.addPDFZoomFitPolicyChangedListener(listener);
  }

  /**
   * Mostra o sidebar com os thumbnails
   */
  protected void showThumbnails() {
    remove(pdfCorePanel);
    sideBarSplitPane.setLeftComponent(coreThumbnailPanel);
    sideBarSplitPane.setRightComponent(pdfCorePanel);
    sideBarSplitPane.resetToPreferredSizes();
    add(sideBarSplitPane, BorderLayout.CENTER);
    revalidate();
  }

  /**
   * Oculta o sidebar com os thumbnails
   */
  protected void hideThumbnails() {
    remove(sideBarSplitPane);
    add(pdfCorePanel, BorderLayout.CENTER);
    revalidate();
  }

  /**
   * Muda a visibilidade dos thumbnails
   * 
   * @param visible true para exibir, false para ocultar
   */
  public void setThumbnailsVisible(boolean visible) {
    if (visible != thumbnailsVisible) {
      thumbnailsVisible = visible;
      applyThumbsVisible();
      notifyThumbnailsSidebarVisibilityListeners(thumbnailsVisible);
    }
  }

  /**
   * Notifica os listeners ativos se o thumbnails esto visveis ou no
   * 
   * @param thumbnailsVisible true se estiverem visveis, false caso contrrio
   */
  private void notifyThumbnailsSidebarVisibilityListeners(
    boolean thumbnailsVisible) {

    for (PDFThumbnailsVisibilityListener listener : thumbnailsVisibilityListeners) {
      listener.onThumbnailsVisibilityChanged(thumbnailsVisible);
    }

  }

  /**
   * Retorna se a sidebar com thumbnails est visvel
   * 
   * @return se a sidebar com thumbnails est visvel
   */
  public boolean isThumbnailsVisible() {
    return thumbnailsVisible;
  }

  /**
   * Exibe ou ocults os thumbnails de acordo com o atributo de
   * thumbsnailsVisible
   */
  protected void applyThumbsVisible() {
    if (thumbnailsVisible) {
      showThumbnails();
    }
    else {
      hideThumbnails();
    }
  }

  /**
   * Retorna uma String internacionalizada para dada a chave
   * 
   * @param key chave da string de
   * @return uma String internacionalizada para dada a chave
   */
  public String getString(String key) {
    return pdfCorePanel.getString(key);
  }

}
