/*
 * $Id: ImprovedXMLListIterator.java 84592 2008-10-22 16:05:15Z costa $
 */
package tecgraf.javautils.xml;

import java.util.ArrayList;
import java.util.List;

/**
 * A classe <code>NewXMLListIterator</code> implementa uma nova verso do
 * iterador sobre elementos XML coletados pelo parser.
 * <p>
 * Esta verso  baseada em um array (ao invs de uma lista, como na
 * implementao original), o que simplifica muito a implementao no s das
 * operaes definidas na interface como operaes extra como
 * {@link #hasAny(String)} e {@link #howManyLeft(String)}.
 */
public class ImprovedXMLListIterator implements XMLListIteratorInterface {

  /**
   * Lista que ser percorrida.
   */
  private List<XMLElementInterface> list;
  /**
   * ndice do elemento corrente na lista.
   */
  private int index = 0;
  /**
   * Tag que est sendo procurada.
   */
  private String searchTag = ANY;
  /**
   * Flag que indica se qualquer tag deve ser considerada como match para a
   * busca.
   */
  private boolean matchAny = true;

  /**
   * Construtor. Usa como tag de percorrimento a tag default <i>ANY</i>.
   * 
   * @param list - lista de elementos coletados pelo parser
   */
  public ImprovedXMLListIterator(List<XMLElementInterface> list) {
    this.list = list != null ? list : new ArrayList<>();
  }

  /**
   * Construtor. Usa uma tag especfica para percorrimento.
   * 
   * @param list - lista de elementos coletados pelo parser
   * @param tag - tag para percorrimento
   */
  public ImprovedXMLListIterator(List<XMLElementInterface> list, String tag) {
    this(list);
    newTag(tag);
  }

  /**
   * {@inheritDoc}
   */
  public void reset(String tag) {
    index = 0;
    newTag(tag);
  }

  /**
   * Redefine a tag de busca.
   * 
   * @param tag - nova tag de busca
   */
  private void newTag(String tag) {
    searchTag = tag;
    matchAny = tag.equals(ANY);
  }

  /**
   * {@inheritDoc}
   */
  public void reset() {
    index = 0;
    newTag(ANY);
  }

  /**
   * {@inheritDoc}
   */
  public XMLElementInterface next() {
    for (int i = index; i < list.size(); i++) {
      final XMLElementInterface element = list.get(i);
      if (tagMatches(element)) {
        index = i + 1;
        return element;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  public XMLElementInterface next(String tag) {
    newTag(tag);
    return next();
  }

  /**
   * {@inheritDoc}
   */
  public boolean hasNext() {
    if (index == list.size())
      return false;
    return tagMatches(list.get(index));
  }

  /**
   * Indica se um determinado elemento est associado  tag de busca.
   * 
   * @param element - elemento a ser comparado
   * @return true se o elemento est associado  tag de busca
   */
  private boolean tagMatches(XMLElementInterface element) {
    return matchAny || searchTag.equals(element.getTag());
  }

  /**
   * Indica se um determinado elemento est associado a uma tag.
   * 
   * @param element - elemento a ser comparado
   * @param tag - tag
   * @return true se o elemento est associado  tag
   */
  private boolean tagMatches(XMLElementInterface element, String tag) {
    return tag.equals(ANY) || tag.equals(element.getTag());
  }

  /**
   * {@inheritDoc}
   */
  public String getTag() {
    return searchTag;
  }

  /**
   * Indica se ainda h algum elemento com uma determinada tag. Apenas os
   * elementos restantes so consultados, e o cursor no  reposicionado,
   * independente do resultado.
   * <p>
   * Todos os elementos restantes so consultados, at que se encontre algum do
   * tipo especificado ou se chegue ao fim da lista.
   * 
   * @param tag - tag
   * @return true se ainda h um elemento com a tag
   */
  public boolean hasAny(String tag) {
    for (int i = index; i < list.size(); i++) {
      if (tagMatches(list.get(i), tag)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Determina se a lista ainda possui o elemento em questo. Apenas os
   * elementos restantes so consultados, e o cursor no  reposicionado,
   * independente do resultado.
   * <p>
   * Todos os elementos restantes so consultados, at que se encontre algum do
   * tipo especificado ou se chegue ao fim da lista.
   * 
   * @see #hasAny(String)
   * @param element - elemento
   * @return true se ainda h alguma instncia do elemento na lista
   */
  public boolean hasAny(XMLElementInterface element) {
    return hasAny(element.getTag());
  }

  /**
   * Retorna o nmero de instncias de um determinado elemento na lista, a
   * partir da posio corrente. O cursor no  reposicionado, indepentende do
   * resultado.
   * 
   * @param element - elemento
   * @return nmero de ocorrncias restantes do elemento
   */
  public int howManyLeft(XMLElementInterface element) {
    final String tag = element.getTag();
    return howManyLeft(tag);
  }

  /**
   * Retorna o nmero de instncias de um determinado elemento na lista, a
   * partir da posio corrente. O cursor no  reposicionado, indepentende do
   * resultado.
   * 
   * @param tag - tag do elemento
   * @return nmero de ocorrncias restantes do elemento
   */
  public int howManyLeft(final String tag) {
    int count = 0;
    for (int i = index; i < list.size(); i++) {
      if (tagMatches(list.get(i), tag)) {
        count++;
      }
    }
    return count;
  }

  /**
   * Verifica se o prximo elemento  de um tipo especfico, baseado em uma tag.
   * O cursor no  reposicionado.
   * 
   * @param tag - tag
   * @return true se o prximo elemento  do tipo especificado, false se no for
   *         ou se o percorrimento j tiver terminado
   */
  public boolean isNext(final String tag) {
    if (index >= list.size())
      return false;
    return tag.equalsIgnoreCase(list.get(index).getTag());
  }

  /**
   * Verifica se o percorrimento j terminou.
   * 
   * @return true se j percorremos toda a lista
   */
  public boolean hasEnded() {
    return index >= list.size();
  }
}
