/**
 * $Id: InternalHandler.java 100297 2009-12-22 21:39:16Z costa $
 */
package tecgraf.javautils.xml;

import java.util.LinkedList;
import java.util.List;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

import tecgraf.javautils.xml.exception.XMLException;
import tecgraf.javautils.xml.exception.XMLParseException;

/**
 * Handler SAX de eventos. Esta classe intercepta os eventos relevantes e
 * executa os mtodos pertinentes em um handler externo, para que os detalhes do
 * processamento de eventos SAX no tenham que ser expostos  aplicao.
 * 
 * @author Andre Oliveira da Costa.
 */
class InternalHandler extends DefaultHandler {

  /** handler externo da biblioteca */
  final XMLHandlerInterface xmlHandler;

  /**
   * Callback executada quando o fim do documento  atingido.
   * 
   * @see org.xml.sax.ContentHandler#endDocument()
   */
  @Override
  final public void endDocument() throws SAXException {
    try {
      xmlHandler.endDocument();
    }
    catch (XMLException e) {
      throw new SAXException(e);
    }
  }

  /**
   * Evento associado  leitura de caracteres associados ao valor do elemento.
   * Esta leitura pode ser parcial (i.e. outros eventos semelhantes se seguiro
   * a este).
   * 
   * @see org.xml.sax.ContentHandler#characters(char[], int, int)
   */
  @Override
  final public void characters(final char[] ch, final int start,
    final int length) throws SAXException {
    try {
      xmlHandler.characters(ch, start, length);
    }
    catch (XMLException e) {
      throw new SAXException(e);
    }
  }

  /**
   * Evento associado  tag de abertura de um elemento.
   * 
   * @see org.xml.sax.ContentHandler#startElement(java.lang.String,java.lang.String,java.lang.String,org.xml.sax.Attributes)
   */
  @Override
  final public void startElement(final String uri, final String localName,
    final String qName, final Attributes attributes) throws SAXException {
    try {
      xmlHandler.startElement(uri, localName, qName,
        unpackSAXAttributes(attributes));
    }
    catch (XMLException e) {
      throw new SAXException(e);
    }
  }

  /**
   * Evento associado  tag de fechamento de um elemento.
   * 
   * @see org.xml.sax.ContentHandler#endElement(java.lang.String,java.lang.String,java.lang.String)
   */
  @Override
  final public void endElement(final String uri, final String localName,
    final String qName) throws SAXException {
    try {
      xmlHandler.endElement(uri, localName, qName);
    }
    catch (XMLException e) {
      throw new SAXException(e);
    }
  }

  /**
   * Callback executada quando o parser tenta resolver uma entidade externa
   * (p.ex. a URL associada ao DTD).
   * 
   * @see DefaultHandler#resolveEntity(java.lang.String, java.lang.String)
   */
  @Override
  final public InputSource resolveEntity(final String publicId,
    final String systemId) throws SAXException {
    try {
      return xmlHandler.resolveEntity(publicId, systemId);
    }
    catch (XMLException e) {
      throw new SAXException(e);
    }
  }

  /**
   * Evento associado a um alerta ocorrido durante o processamento.
   * 
   * @see org.xml.sax.ErrorHandler#warning(org.xml.sax.SAXParseException)
   */
  @Override
  final public void warning(final SAXParseException e) throws SAXException {
    try {
      xmlHandler.warning(new XMLParseException(e));
    }
    catch (XMLException xe) {
      throw new SAXException(xe);
    }
  }

  /**
   * Evento associado a um erro fatal de processamento.
   * 
   * @see org.xml.sax.ErrorHandler#fatalError(org.xml.sax.SAXParseException)
   */
  @Override
  final public void fatalError(final SAXParseException e) throws SAXException {
    try {
      xmlHandler.fatalError(new XMLParseException(e));
    }
    catch (XMLException xe) {
      throw new SAXException(xe);
    }
  }

  /**
   * Evento associado a um erro de processamento.
   * 
   * @see org.xml.sax.ErrorHandler#error(org.xml.sax.SAXParseException)
   */
  @Override
  final public void error(final SAXParseException e) throws SAXException {
    try {
      xmlHandler.error(new XMLParseException(e));
    }
    catch (XMLException xe) {
      throw new SAXException(xe);
    }
  }

  /**
   * Converte atributos SAX em uma lista encadeada de objetos do tipo
   * <code>XMLAttribute</code>.
   * 
   * @param saxAttrs atributos SAX
   * @return lista encadeada com objetos do tipo <code>XMLAttribute</code>
   */
  List<XMLAttribute> unpackSAXAttributes(final Attributes saxAttrs) {
    final LinkedList<XMLAttribute> attrs = new LinkedList<>();
    for (int i = 0; i < saxAttrs.getLength(); i++) {
      attrs.add(new XMLAttribute(saxAttrs.getQName(i), saxAttrs.getValue(i)));
    }
    return attrs;
  }

  /**
   * Construtor. Armazena referncia para o handler externo.
   * 
   * @param xmlHandler handler externo da biblioteca XML.
   */
  InternalHandler(final XMLHandlerInterface xmlHandler) {
    super();
    this.xmlHandler = xmlHandler;
  }
}
