/* $Id: XMLBasicElement.java 123004 2011-10-13 14:35:40Z costa $ */

package tecgraf.javautils.xml;

import java.io.IOException;
import java.io.Writer;
import java.util.List;

/**
 * A classe <code>XMLBasicElement</code> representa elementos XML que tm
 * filhos.
 * 
 * @author Andre Oliveira da Costa
 */
public abstract class XMLBasicElement extends XMLElement {

  /**
   * Construtor vazio. Existe apenas como "ponte" para o construtor da
   * superclasse.
   */
  protected XMLBasicElement() {
    super();
  }

  /**
   * Cria um novo elemento com a tag especificada. Existe apenas como "ponte"
   * para o construtor da superclasse.
   * 
   * @param tag a tag
   */
  protected XMLBasicElement(String tag) {
    super(tag);
  }

  /**
   * Escreve uma tag de abertura arbitrria (i.e. no a do elemento) sem
   * atributos.
   * 
   * @param tagName tag
   * @param stream stream de sada
   * @param ident identao
   * 
   * @throws IOException se houve algum erro de I/O durante a escrita
   */
  public static void writeStartTagln(final String tagName, final Writer stream,
    final String ident) throws IOException {
    stream.write(ident + "<" + tagName + ">\n");
  }

  /**
   * Escreve uma tag de abertura arbitrria (com "\n" no final), com uma lista
   * de atributos (<code>XMLAttribute</code>) tambm arbitrria.
   * 
   * @param tagName tag
   * @param writer stream de sada
   * @param ident identao
   * @param attrs lista de attributos (<code>XMLAttribute</code>)
   * @throws IOException se houve algum erro de I/O durante a escrita
   */
  public static void writeStartTagln(final String tagName, final Writer writer,
    final String ident, final List<? extends XMLAttribute> attrs)
    throws IOException {
    writeStartTag(tagName, writer, ident, attrs);
    writer.write("\n");
  }

  /**
   * Escreve uma tag de abertura arbitrria (sem "\n" no final), com uma lista
   * de atributos (<code>XMLAttribute</code>) tambm arbitrria.
   * 
   * @param tagName tag
   * @param writer stream de sada
   * @param ident identao
   * @param attrs lista de attributos (<code>XMLAttribute</code>)
   * @throws IOException se houve algum erro de I/O durante a escrita
   */
  public static void writeStartTag(final String tagName, final Writer writer,
    final String ident, final List<? extends XMLAttribute> attrs)
    throws IOException {
    writer.write(ident + "<" + tagName);
    writeAttributes(writer, attrs);
    writer.write(">");
  }

  /**
   * Escreve uma tag de abertura arbitrria, sem "\n" no final e sem atributos.
   * 
   * @param tagName tag
   * @param writer stream de sada
   * @param ident identao
   * @throws IOException se houve algum erro de I/O na escrita
   */
  public static void writeStartTag(final String tagName, final Writer writer,
    final String ident) throws IOException {
    writer.write(ident + "<" + tagName + ">");
  }

  /**
   * Escreve uma tag de fechamento arbitrria, com identao.
   * 
   * @param tagName tag
   * @param stream stream de sada
   * @param ident identao
   * 
   * @throws IOException se houve algum erro de I/O durante a escrita
   */
  public static void writeEndTag(final String tagName, final Writer stream,
    final String ident) throws IOException {
    stream.write(ident + "</" + tagName + ">\n");
  }

  /**
   * Retorna a string correspondente ao prximo nvel de identao baseado na
   * identao corrente e na identao default definido em
   * <code>XMLWriter</code>.
   * 
   * @param ident identao corrente
   * @return string correspondente  identao no prximo nvel
   */
  public static String getNextIdentation(final String ident) {
    return ident + XMLWriter.getDefaultIdent();
  }

  /**
   * Mtodo esttico para buscar a primeira ocorrncia de um elemento XML em uma
   * lista de elementos pela sua tag. Esta implementao faz busca sequencial.
   * 
   * @param children lista de elementos
   * @param tagName tag a ser buscada
   * 
   * @return o elemento caso este exista na lista, ou <code>null</code> caso
   *         contrrio
   */
  public static XMLElementInterface getFirstChildWithTag(
    final List<XMLElementInterface> children, final String tagName) {
    XMLListIteratorInterface iterator =
      new ImprovedXMLListIterator(children, tagName);
    return iterator.next();
  }
}
