package tecgraf.javautils.xml;

import java.lang.reflect.Constructor;
import java.util.Map;

import tecgraf.javautils.xml.exception.XMLInternalErrorException;

/**
 * Fbrica de elementos. Disponibiliza funes para instanciar classes XML a
 * partir de uma tag ou de um objeto da aplicao que possua classe XML
 * correspondente.<br>
 * O mapeamento tag --&gt; classe de elemento XML  usado quando um evento de
 * nova tag  recebido pelo handler. O mapeamento classe de aplicao --&gt; tag
 *  usado por XMLWriter para a criao de um elemento XML a partir de um objeto
 * da aplicao.
 * 
 * @author Andre Oliveira da Costa
 */
public abstract class XMLElementFactory implements XMLElementFactoryInterface {

  /**
   * Construtor protegido.
   */
  protected XMLElementFactory() {
    fillConversionTables();
  }

  /**
   * Cria um elemento XML a partir de uma tag. Usa o mapeamento
   * <code>tagNamesToXMLClasses</code>, preenchido pelo mtodo
   * <code>fillConversionTables</code> para obter o nome da classe associada 
   * tag.
   * 
   * Este mtodo tipicamente no precisa ser redefinido, basta implementar o
   * mtodo abstrato <code>fillConversionTables</code>.
   * 
   * @see tecgraf.javautils.xml.XMLElementFactoryInterface#createXMLElementFromTag(java.lang.String)
   * @throws XMLInternalErrorException nos casos:
   *         <ol>
   *         <li>ClassNotFoundException: no existe o mapeamento de uma tag para
   *         a classe especificada
   *         <li>NoSuchMethodException: no existe um construtor que receba
   *         apenas uma tag como parmetro
   *         <li>SecurityException: permisses de acesso do construtor no
   *         permitem sua execuo
   *         <li>InstantiationException: classe especificada corresponde a uma
   *         classe abstrata IllegalAccessException: construtor inacessvel
   *         <li>IllegalArgumentException: assinatura do construtor est errada
   *         <li>InvocationTargetException: construtor gerou uma exceo
   *         </ol>
   */
  @Override
  public XMLElementInterface createXMLElementFromTag(final String tagName) {
    XMLElementInterface element = null;
    Class<? extends XMLElementInterface> xmlClass = null;
    Map<String, Class<? extends XMLElementInterface>> tagNamesToXMLClasses =
      getTagToXMLMap();
    try {
      xmlClass = tagNamesToXMLClasses.get(tagName);
      if (xmlClass != null) {
        element = xmlClass.newInstance();
        element.setAppContextObject(getAppContextObject());
        element.setTag(tagName);
      }
    }
    catch (IllegalAccessException e) {
      throw new XMLInternalErrorException(tagName,
        "construtor sem parmetros da classe " + xmlClass.getName()
          + " no est acessvel");
    }
    catch (InstantiationException e) {
      throw new XMLInternalErrorException(tagName, "classe "
        + xmlClass.getName() + " no possui construtor sem parmetros");
    }
    catch (Exception e) {
      throw new XMLInternalErrorException(tagName, e);
    }
    return element;
  }

  /**
   * Cria um elemento XML a partir de um objeto da aplicao. Usa o mapeamento
   * <code>appClassesToTags</code>, preenchido pelo mtodo
   * <code>fillConversionTables</code> para obter o nome da tag associada ao
   * objeto.<br>
   * 
   * Este mtodo tipicamente no precisa ser redefinido, basta implementar o
   * mtodo abstrato <code>fillConversionTables</code>.
   * 
   * @param appObject objeto raiz da aplicacao.
   * 
   * @see tecgraf.javautils.xml.XMLElementFactoryInterface#createXMLElementFromApp(java.lang.Object)
   */
  @Override
  public XMLElementInterface createXMLElementFromApp(final Object appObject) {
    XMLElementInterface element = null;
    if (appObject == null) {
      final String err = "objeto nulo na criacao de elemento XML";
      throw new XMLInternalErrorException(null, err);
    }
    Map<String, Class<? extends XMLElementInterface>> tagNamesToXMLClasses =
      getTagToXMLMap();
    Map<Class<?>, String> appClassesToTags = getAppObjectToTagMap();
    final Class<?> appClass = appObject.getClass();
    final String tagName = appClassesToTags.get(appClass);

    if (tagName == null) {
      final String err =
        "tag nula para objeto da classe: " + appClass.getName();
      throw new XMLInternalErrorException(null, err);
    }

    final Class<? extends XMLElementInterface> xmlClass =
      tagNamesToXMLClasses.get(tagName);
    if (xmlClass == null) {
      final String err = "objeto nulo no mapeamento para a tag";
      throw new XMLInternalErrorException(tagName, err);
    }

    Constructor<? extends XMLElementInterface> xmlElementConstructor = null;
    final Class<?>[] noParams = {};
    try {
      xmlElementConstructor = xmlClass.getConstructor(noParams);
    }
    catch (Exception e) {
      final String err = "erro acessando construtor de " + xmlClass.toString();
      throw new XMLInternalErrorException(tagName, e, err);
    }
    final Object[] noObjs = {};
    try {
      element = xmlElementConstructor.newInstance(noObjs);
      element.setAppObject(appObject);
      element.setAppContextObject(getAppContextObject());
      element.setTag(tagName);
    }
    catch (Exception e) {
      final String err = "erro executando construtor de " + xmlClass;
      throw new XMLInternalErrorException(tagName, e, err);
    }
    return element;
  }

  /**
   * Associa uma tag a uma classe que representa o elemento XML correspondente.
   * Recebe o nome da classe como parmetro.
   * 
   * @param className nome da classe
   * @param tagName nome da tag
   * @throws XMLInternalErrorException se a tag  nula, se o nome da classe 
   *         nulo ou se j existe mapeamento para a tag
   * @deprecated use {@link #mapTagToXMLClass(String, Class)}
   */
  @SuppressWarnings("unchecked")
  @Deprecated
  final public void mapTagToXMLClass(final String tagName,
    final String className) {
    if (className == null) {
      final String err = "Nome de classe nulo em mapeamento TAG -> XML";
      throw new XMLInternalErrorException(tagName, err);
    }
    try {
      mapTagToXMLClass(tagName, (Class<? extends XMLElementInterface>) Class
        .forName(className));
    }
    catch (ClassNotFoundException e1) {
      throw new XMLInternalErrorException(tagName, "Classe '" + className
        + "' invlida em mapeamento TAG -> XML");
    }
  }

  /**
   * Associa uma tag a uma classe que representa o elemnto XML correspondente.
   * Recebe a prpria classe como parmetro.
   * 
   * @param tagName nome da tag
   * @param cls classe
   * @throws XMLInternalErrorException se a tag  nula, se a classe  nula ou se
   *         j existe mapeamento para a tag
   */
  final public void mapTagToXMLClass(final String tagName,
    final Class<? extends XMLElementInterface> cls) {
    if (tagName == null) {
      final String err = "Tag nulo em mapeamento TAG -> XML";
      throw new XMLInternalErrorException(null, err);
    }
    if (cls == null) {
      final String err = "Classe nula em mapeamento TAG -> XML";
      throw new XMLInternalErrorException(tagName, err);
    }
    Map<String, Class<? extends XMLElementInterface>> tagNamesToXMLClasses =
      getTagToXMLMap();
    /*
     * se j existe um mapeamento para a tag em questo e  para uma classe
     * diferente da que estamos mapeando, geramos exceo
     */
    if (tagNamesToXMLClasses.containsKey(tagName)) {
      if (tagNamesToXMLClasses.get(tagName) != cls) {
        throw new XMLInternalErrorException(tagName,
          "redefinio de mapeamento da tag");
      }
    }
    else {
      tagNamesToXMLClasses.put(tagName, cls);
    }
  }

  /**
   * Associa uma classe da aplicao a uma tag. Recebe o nome da classe.
   * 
   * @param className nome da classe
   * @param tagName nome da tag
   * @deprecated use {@link #mapAppClassToTag(Class, String)}
   */
  @Deprecated
  final public void mapAppClassToTag(final String className,
    final String tagName) {
    if (tagName == null) {
      throw new XMLInternalErrorException(null,
        "Tag nulo em mapeamento CLASS -> TAG");
    }
    if (className == null) {
      throw new XMLInternalErrorException(tagName,
        "Nome de classe nulo em mapeamento CLASS -> TAG");
    }
    Map<Class<?>, String> appClassesToTags = getAppObjectToTagMap();
    try {
      appClassesToTags.put(Class.forName(className), tagName);
    }
    catch (ClassNotFoundException e) {
      throw new XMLInternalErrorException(tagName, "Classe '" + className
        + "' invlida em mapeamento CLASS -> TAG");
    }
  }

  /**
   * Associa uma classe da aplicao a uma tag. Recebe a prpria classe como
   * parmetro.
   * 
   * @param cls a classe
   * @param tagName nome da tag
   */
  final public void mapAppClassToTag(final Class<?> cls, final String tagName) {
    if (tagName == null) {
      final String err = "Tag nulo em mapeamento CLASS -> TAG";
      throw new XMLInternalErrorException(null, err);
    }
    if (cls == null) {
      final String err = "Classe nula em mapeamento CLASS -> TAG";
      throw new XMLInternalErrorException(tagName, err);
    }
    Map<Class<?>, String> appClassesToTags = getAppObjectToTagMap();
    /*
     * se j existe um mapeamento para a classe em questo e  para uma tag
     * diferente da que estamos mapeando, geramos exceo
     */
    if (appClassesToTags.containsKey(cls)) {
      if (!appClassesToTags.get(cls).equals(tagName)) {
        throw new XMLInternalErrorException(cls.getName(),
          "redefinio de mapeamento da classe");
      }
    }
    else {
      appClassesToTags.put(cls, tagName);
    }
  }

  /**
   * Preenche as tabelas de converso de tag para nome de classe. As tabelas de
   * converso so:
   * <ul>
   * <li><b>appClassesToTags</b> - mapeamento de classes da aplicao para tags.
   * Usar os mtodos <code>mapAppClassToTag</code></li>
   * <li><b>tagNamesToXMLClasses</b> - mapeamento de tags para classes que
   * modelam elementos XML. Usar os mtodos <code>mapTagToXMLClass</code></li>
   * </ul>
   */
  protected abstract void fillConversionTables();

  /**
   * Retorna o mapeamento de tags para elementos XML. Este mtodo  abstrato
   * para permitir que as subclasses implementem os mapas da forma que for mais
   * conveniente (p.ex. como campos estticos).
   * 
   * @return mapa definido pela subclasse para associar tags para elementos XML
   */
  protected abstract Map<String, Class<? extends XMLElementInterface>> getTagToXMLMap();

  /**
   * Retorna o mapeamento de objetos da aplicao para tags. Este mtodo 
   * abstrato para permitir que as subclasses implementem os mapas da forma que
   * for mais conveniente (p.ex. como campos estticos).
   * 
   * @return mapa definido pela subclasse para associar objetos da aplicao a
   *         tags
   */
  protected abstract Map<Class<?>, String> getAppObjectToTagMap();
}
