/*
 * $Id: XMLElementInterface.java 2016 2006-02-10 16:47:43 +0000 (Fri, 10 Feb
 * 2006) costa $
 */

package tecgraf.javautils.xml;

import java.io.IOException;
import java.io.Writer;
import java.util.List;

/**
 * Elemento bsico XML.
 * 
 * @author Andre Oliveira da Costa
 */
public interface XMLElementInterface {
  /**
   * Callback associada ao evento characters(), gerado quando um elemento CDATA
   *  capturado.
   * 
   * @param charArray os caracteres
   * @param start posio inicial no array de caracteres
   * @param length nmero de caracteres a serem recuperados
   */
  public void newCharsEvent(final char[] charArray, final int start,
    final int length);

  /**
   * Define o valor do elemento XML a partir de uma <code>String</code>.
   * 
   * @param value valor
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface setValue(final String value);

  /**
   * Define o valor do elemento XML a partir de um <code>double</code>.
   * 
   * @param value valor
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface setValue(final double value);

  /**
   * Define valor nulo para o elemento.
   * 
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface resetValue();

  /**
   * Adiciona um valor textual (append) ao elemento XML.
   * 
   * @param value - valor como texto (string).
   * @param newLine - indica se deve ser anexado um fim-de-linha
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface appendValue(final String value,
    final boolean newLine);

  /**
   * Indica se o valor associado ao elemento  diferente da string vazia.
   * Observar que qualquer caracter (incluindo espaos em branco e terminadores
   * de linha) configura a existncia de valor.
   * 
   * @return <code>true</code> se o o valor do elemento  diferente da string
   *         vazia
   */
  public boolean hasValue();

  /**
   * Indica se o valor do elemento  uma string vazia ou  constitudo apenas
   * por espaos ou terminadores de linha.
   * 
   * @return true se o valor do elemento  uma string vazia ou  constitudo
   *         apenas por espaos ou terminadores de linha
   */
  public boolean isEmpty();

  /**
   * Retorna o valor do elemento como string.
   * 
   * @return valor associado ao elemento
   */
  public String getStrValue();

  /**
   * Retorna a tag associada ao elemento.
   * 
   * @return tag associada ao elemento
   */
  public String getTag();

  /**
   * Ajusta a tag associada ao elemento.
   * 
   * @param tag tag associada ao elemento
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface setTag(final String tag);

  /**
   * Instancia um novo atributo XML para o objeto.
   * 
   * @param name nome do atributo
   * @param value valor do atributo
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface newAttribute(final String name, final String value);

  /**
   * Instancia um novo atributo que tem como valor um nmero inteiro.
   * 
   * @param name nome do atributo
   * @param value valor
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface newAttribute(final String name, final int value);

  /**
   * Instancia um novo atributo que tem como valor um nmero inteiro (long).
   * 
   * @param name nome do atributo
   * @param value valor
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface newAttribute(final String name, final long value);

  /**
   * Instancia um novo atributo que tem como valor um booleano. O valor do
   * atributo ser uma dentre as strings <code>TRUE</code> ou <code>FALSE</code>
   * .
   * 
   * @param name nome do atributo
   * @param value valor
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface newAttribute(final String name, final boolean value);

  /**
   * Instancia um novo atributo que tem como valor um <code>double</code>.
   * 
   * @param name nome do atributo
   * @param value valor
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface newAttribute(final String name, final double value);

  /**
   * Instancia um novo atributo que tem como valor uma constante de um
   * {@link Enum}. O mtodo {@link Enum#toString()}  usado para obter o valor a
   * ser associado ao atributo.
   * 
   * @param name - nome do atributo
   * @param value - valor de enumerao
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface newAttribute(final String name, final Enum<?> value);

  /**
   * Redefine os atributos do elemento baseado em um novo mapa.
   * 
   * @param newAttrs mapa com os novos atributos na forma {nome, valor}
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface newAttributeList(final List<XMLAttribute> newAttrs);

  /**
   * Limpa a lista de atributos.
   */
  public void resetAttributeList();

  /**
   * Indica se o elemento possui o atributo dado.
   * 
   * @param name Nome do atributo
   * 
   * @return true caso o elemento possua o atributo. False caso contrrio.
   */
  public boolean hasAttribute(final String name);

  /**
   * Recupera o valor associado a um atributo.
   * 
   * @param name nome do atributo
   * 
   * @return valor (string) do atributo
   */
  public String getAttributeStrValue(final String name);

  /**
   * Retorna o valor inteiro associado a um atributo.
   * 
   * @param name nome do atributo
   * 
   * @return valor inteiro associado ao atributo
   */
  public int getAttributeIntValue(final String name);

  /**
   * Retorna o valor real associado a um atributo.
   * 
   * @param name nome do atributo
   * 
   * @return valor real associado ao atributo
   */
  public double getAttributeDoubleValue(final String name);

  /**
   * Retorna <code>true</code> se o valor do atributo  igual a
   * <code>"true"</code> (case-insensitive), <code>false</code> caso contrrio.
   * 
   * @param name nome do atributo
   * @return <code>true</code> se o valor do atributo  igual  string
   *         <code>"true"</code> (case-insensitive), <code>false</code> caso
   *         contrrio.
   */
  public boolean getAttributeBooleanValue(final String name);

  /**
   * Mtodo executado quando a tag de fim do elemento XML  encontrada. Neste
   * momento devemos criar o objeto correspondente na camada da aplicao.
   * Observar que no necessariamente cada elemento XML filho do elemento em
   * questo ser mapeado para um objeto na aplicao (em muitos casos, sero
   * gerados atributos).
   * 
   * @param childrenList lista com os filhos do elemento em questo, ou uma
   *        lista vazia caso o elemento no possua filhos
   */
  public void endTag(final List<XMLElementInterface> childrenList);

  /**
   * Converte o valor armazenado, trocando &amp;lt; por &lt; etc.
   */
  public void convertValueFromXML();

  /**
   * Mtodo complementar a convertValueFromXML().
   * 
   * @param text texto a ser convertido
   * 
   * @return texto convertido para XML
   */
  @Deprecated
  public String convertTextToXML(final String text);

  /**
   * Retorna a lista de atributos do elemento.
   * 
   * @return a lista de atributos.
   */
  public List<? extends XMLAttribute> getAttributes();

  /**
   * Escreve o elemento em um stream.
   * 
   * @param writer stream de sada
   * @param ident identao
   * 
   * @throws IOException erro de escrita no stream
   */
  public void write(Writer writer, final String ident) throws IOException;

  /**
   * Escreve a tag de abertura do elemento. No  anexado um fim-de-linha ao
   * final da mesma.
   * 
   * @param stream stream de sada
   * @param ident identao
   * 
   * @throws IOException erro de escrita no stream
   */
  public void writeStartTag(final Writer stream, final String ident)
    throws IOException;

  /**
   * Escreve a tag de abertura do elemento, com fim-de-linha.
   * 
   * @param stream stream de sada
   * @param ident identao
   * 
   * @throws IOException erro de escrita no stream
   */
  public void writeStartTagln(final Writer stream, final String ident)
    throws IOException;

  /**
   * Escreve a tag de fechamento do elemento, usando identao.  usada para
   * fechar um elemento em uma linha diferente daquela que contm a tag de
   * abertura.
   * 
   * @param stream stream de sada
   * @param ident identao
   * 
   * @throws IOException erro de escrita no stream
   */
  public void writeEndTag(final Writer stream, final String ident)
    throws IOException;

  /**
   * Escreve a tag de fechamento do elemento, sem identao. Usado quando se
   * quer fechar uma tag na mesma linha em que esta foi iniciada.
   * 
   * @param stream stream de sada
   * 
   * @throws IOException erro de escrita no stream
   */
  public void writeEndTag(final Writer stream) throws IOException;

  /**
   * Escreve os atributos do elemento, na mesma ordem em que foram coletados.
   * 
   * @param stream stream de sada
   * 
   * @throws IOException erro de escrita no stream
   */
  public void writeAttributes(final Writer stream) throws IOException;

  /**
   * Escreve o valor do elemento.
   * 
   * @param stream stream de sada
   * 
   * @throws IOException erro de escrita no stream
   */
  public void writeValue(final Writer stream) throws IOException;

  /**
   * Sinaliza para o elemento que ele acaba de ser criado. Neste momento seus
   * atributos j foram coletados e se encontram disponveis para manipulao. A
   * implementao default deste mtodo tipicamente no faz nada.
   */
  public void startTag();

  /**
   * Retorna o valor associado ao elemento como um <code>float</code>.
   * 
   * @return <code>float</code> associado ao elemento
   */
  public float getFloatValue();

  /**
   * Retorna o valor associado ao elemento como um <code>int</code>.
   * 
   * @return <code>int</code> associado ao elemento
   */
  public int getIntValue();

  /**
   * Retorna o valor associado ao elemento como um <code>long</code>.
   * 
   * @return <code>long</code> associado ao elemento
   */
  public long getLongValue();

  /**
   * Retona o valor associado ao elemento como um <code>float</code>.
   * 
   * @return <code>double</code> associado ao elemento
   */
  public double getDoubleValue();

  /**
   * Define o contexto da aplicao.
   * 
   * @param contextObject objeto representativo de contexto da aplicacao.
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface setAppContextObject(final Object contextObject);

  /**
   * Retorna o contexto da aplicao.
   * 
   * @return o objeto representativo de contexto da aplicacao.
   */
  public Object getAppContextObject();

  /**
   * Define o objeto da aplicao associado a este elemento.
   * 
   * @param appObject objeto analogo na aplicacao.
   * @return o prprio objeto (para encadeamento de operaes)
   */
  public XMLElementInterface setAppObject(final Object appObject);

  /**
   * Retorna o objeto da aplicao associado a este elemento.
   * 
   * @return o objeto da aplicao associado ao elemento (pode ser
   *         <code>null</code>).
   */
  public Object getAppObject();
}
