/* $Id: XMLReader.java 123860 2011-11-04 23:11:41Z costa $ */

package tecgraf.javautils.xml;

import java.io.Closeable;
import java.io.IOException;
import java.io.Reader;
import java.util.Map;

/**
 * Leitor de documentos XML.
 * 
 * @author Andr Oliveira da Costa
 */
public class XMLReader implements Closeable {

  /** Parser */
  private XMLParser parser;

  /** Elemento raiz */
  private XMLElementInterface xmlRoot;

  /** Caminho do arquivo de entrada */
  private String xmlFilePath;

  /** Codificacao */
  private String encoding;

  /** Reader para leitura do XML */
  private Reader reader;

  /**
   * Tratador de eventos gerados pelo parser.
   */
  private XMLHandlerInterface handler;

  /**
   * Flag que indica se o arquivo XML j foi lido.
   */
  private boolean alreadyRead;

  /**
   * Encoding default usado na leitura do documento. FIXME pegar de alguma
   * propriedade, ou talvez assumir UTF-8
   */
  public static final String DEFAULT_ENCODING = "ISO-8859-1";

  /**
   * Define na fbrica o contexto da aplicao.
   * 
   * @param appContext contexto da aplicao
   */
  private void setAppContext(Object appContext) {
    // TODO precisamos verificar se xmlFactory != null?
    handler.getXMLFactory().setAppContextObject(appContext);
  }

  /**
   * Processa o arquivo XML associado ao documento.
   * 
   * @throws IOException se houve algum erro de I/O
   * @throws IllegalStateException se o arquivo j foi processado
   */
  final public void read() throws IOException {
    if (alreadyRead) {
      throw new IllegalStateException("O arquivo j foi processado");
    }
    xmlRoot = parser.parse(reader);
    alreadyRead = true;
  }

  /**
   * Retorna o encoding associado ao documento.
   * 
   * @return encoding associado ao documento
   */
  final public String getEncoding() {
    return encoding;
  }

  /**
   * Retorna a fbrica de elementos XML usada pelo <i>reader</i>.
   * 
   * @return o objeto que implementa <code>XMLElementFactoryInterface</code>
   * @deprecated use o mtodo {@link #getXMLFactory()}
   */
  @Deprecated
  final public XMLElementFactoryInterface getXmlFactory() {
    return handler.getXMLFactory();
  }

  /**
   * Retorna a fbrica de elementos XML usada pelo <i>reader</i>.
   * 
   * @return o objeto que implementa <code>XMLElementFactoryInterface</code>
   */
  final public XMLElementFactoryInterface getXMLFactory() {
    return handler.getXMLFactory();
  }

  /**
   * Define o encoding a ser usado na leitura do documento.
   * 
   * @param encoding - novo encoding
   */
  final public void setEncoding(final String encoding) {
    this.encoding = encoding;
  }

  /**
   * Retorna o DTD assoacido ao documento.
   * 
   * @return DTD associado ao documento.
   */
  final public String getDTD() {
    return parser.getDTD();
  }

  /**
   * Retorna o objeto XML associado  raiz do documento XML.
   * 
   * @return objeto associado  raiz do documento XML
   */
  final public XMLElementInterface getXMLRootObject() {
    return xmlRoot;
  }

  /**
   * Retorna o objeto da aplicao associado  raiz do documento XML. Cabe 
   * aplicao garantir que o elemento XML correspondente  raiz retorne um
   * objeto do tipo correto.
   * 
   * @return objeto da aplicao associado  raiz do documento XML
   */
  final public Object getAppRootObject() {
    return xmlRoot.getAppObject();
  }

  /**
   * Retorna o path para o arquivo XML. Pode ser nulo.
   * 
   * @return o path para o arquivo XML (nulo quando o XML for lido a partir de
   *         um <code>InputStream</code>)
   * @deprecated use {@link #getXMLFilePath()}
   */
  @Deprecated
  final public String getXmlFilePath() {
    return xmlFilePath;
  }

  /**
   * Retorna o path para o arquivo XML. Pode ser nulo.
   * 
   * @return o path para o arquivo XML (nulo quando o XML for lido a partir de
   *         um <code>InputStream</code>)
   */
  final public String getXMLFilePath() {
    return xmlFilePath;
  }

  /**
   * Cria um leitor que usa um mapa para criar os objetos.
   * 
   * @param reader reader para leitura do XML
   * @param tagToObjMap mapa relacionando tags a objetos XML
   * @param validate <code>true</code> para validar o documento com relao ao
   *        DTD
   */
  public XMLReader(final Reader reader,
    Map<String, Class<? extends XMLElementInterface>> tagToObjMap,
    final boolean validate) {
    this(reader, new XMLBasicHandler(tagToObjMap), validate);
  }

  /**
   * Cria um leitor que mapeia todas as tags para um nico tipo de objeto XML.
   * 
   * @param reader reader para leitura do XML
   * @param cls classe do objeto XML para o qual todas as tags sero mapeadas
   * @param validate <code>true</code> para validar o documento com relao ao
   *        DTD
   */
  public XMLReader(final Reader reader,
    Class<? extends XMLElementInterface> cls, final boolean validate) {
    this(reader, new XMLBasicHandler(cls), validate);
  }

  /**
   * Cria um leitor que usa uma fbrica para criar objetos XML a partir das
   * tags.
   * 
   * @param reader reader para leitura do XML
   * @param factory fbrica para criar objetos XML a partir das tags
   * @param validate <code>true</code> para validar o documento com relao ao
   *        DTD
   */
  public XMLReader(final Reader reader, XMLElementFactoryInterface factory,
    final boolean validate) {
    this(reader, new XMLBasicHandler(factory), validate);
  }

  /**
   * Cria um leitor que usa um {@link XMLHandlerInterface handler} para criar
   * objetos XML a partir das tags.
   * 
   * @param reader reader para leitura do XML
   * @param handler handler de eventos da aplicao
   * @param validate <code>true</code> para validar o documento com relao ao
   *        DTD
   */
  public XMLReader(final Reader reader, final XMLHandlerInterface handler,
    final boolean validate) {
    this.reader = reader;
    this.handler = handler;
    parser = new XMLParser(handler, validate);
    alreadyRead = false;
  }

  /**
   * Cria um leitor que usa um {@link XMLHandlerInterface handler} para criar
   * objetos XML a partir das tags.
   * 
   * @param reader reader para leitura do XML
   * @param handler handler de eventos da aplicao
   * @param validate <code>true</code> para validar o documento com relao ao
   *        DTD
   * @param appContext contexto da aplicao. Este objeto pode ser acessado por
   *        todos os elementos XML durante seu processamento, de forma a
   *        permitir acesso a informaes da aplicao
   */
  public XMLReader(final Reader reader, final XMLHandlerInterface handler,
    final boolean validate, final Object appContext) {
    this(reader, handler, validate);
    setAppContext(appContext);
  }

  /**
   * Fecha o {@link Reader} usado para leitura do XML.
   * <p>
   * IMPORTANTE: este mtodo s deve ser executado aps o trmino da leitura
   * 
   * @throws IOException se houve algum erro de I/O
   */
  @Override
  public void close() throws IOException {
    reader.close();
  }

}
