/* $Id: XMLWriter.java 123860 2011-11-04 23:11:41Z costa $ */

package tecgraf.javautils.xml;

import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.Map;

import tecgraf.javautils.xml.exception.XMLException;

/**
 * Grava em um stream o XML gerado a partir de objetos da aplicao.
 * 
 * @author Andre Oliveira da Costa
 */
public class XMLWriter implements Closeable {

  /** Stream de sada */
  private Writer writer;

  /** Codificacao */
  private Charset charset;

  /** Elemento raiz do XML */
  private XMLElementInterface xmlRoot;

  /** DTD */
  private String dtd;

  /**
   * Charset default (ISO-8859-1).
   */
  public static final Charset DEFAULT_CHARSET = Charset.forName("ISO-8859-1");

  /** Indentacao default */
  private static String DEFAULT_IDENT = "  ";

  /** Fabrica */
  private XMLElementFactoryInterface xmlFactory;

  /**
   * Escreve efetivamente o XML (os passos anteriores foram apenas preparao).
   * Escreve o header XML e em seguida a rvore, partindo do elemento-raiz.
   * 
   * @throws XMLException em caso de erro.
   */
  private void doWrite() throws XMLException {
    if (xmlRoot == null) {
      throw new XMLException("Elemento <root> nulo na escrita XML!");
    }
    try {
      writeXMLHeader();
      /*
       * a escrita do n-raiz automaticamente dispara a escrita dos demais ns
       * recursivamente
       */
      xmlRoot.write(writer, "");
      writer.flush();
    }
    catch (IOException e) {
      throw new XMLException(e);
    }
  }

  /**
   * Escreve o cabealho xml do documento. O teste para garantir que o documento
   * j foi processado foi feito em doWrite().
   * 
   * @throws IOException gerada pelo mtodo Writer.write()
   * @see Writer
   */
  private void writeXMLHeader() throws IOException {
    writer.write("<?xml version=\"1.0\" encoding=\"" + charset.name()
      + "\"?>\n");
    if (dtd != null) {
      // documentos podem no ter DTDs associados
      writer.write("<!DOCTYPE " + xmlRoot.getTag() + " SYSTEM \"" + dtd
        + "\">\n\n");
    }
  }

  /**
   * Define o contexto da aplicao diretamente na fbrica de elementos. O
   * objeto j foi inicializado, portanto tanto <code>xmlFactory</code> quanto
   * <code>xmlRoot</code> j foram criados.
   * 
   * @param appContext contexto da aplicao
   */
  private void setappContext(Object appContext) {
    xmlFactory.setAppContextObject(appContext);
    xmlRoot.setAppContextObject(appContext);
  }

  /**
   * Retorna a string de identao default usada pelo writer.
   * 
   * @return string de identao default.
   */
  public static String getDefaultIdent() {
    return DEFAULT_IDENT;
  }

  /**
   * Retorna a fbrica de elementos usada pelo writer.
   * 
   * @return fbrica de elementos usada pelo writer.
   */
  final public XMLElementFactoryInterface getXmlFactory() {
    return xmlFactory;
  }

  /**
   * Define a string de identao a ser usada pelo writer.
   * 
   * @param ident string de identao.
   */
  final public void setDefaultIdent(final String ident) {
    DEFAULT_IDENT = ident;
  }

  /**
   * Escreve o documento na sada padro (System.out).
   * 
   * @throws XMLException em caso de erro.
   */
  final public void write() throws XMLException {
    writer = new BufferedWriter(new PrintWriter(System.out));
    doWrite();
  }

  /**
   * Escreve o documento em um {@link Writer}.
   * <p>
   * <b>IMPORTANTE:</b> o <code>Writer</code> recebido como parmetro pode ter
   * definido um charset diferente daquele especificado para este
   * <code>XMLWriter</code>. Para garantir que o charset correto seja usado, use
   * {@link #write(OutputStream)}.
   * 
   * @param w stream de sada
   * 
   * @see #write(OutputStream)
   */
  final public void write(Writer w) {
    this.writer = w;
    doWrite();
  }

  /**
   * Escreve o documento em um {@link OutputStream}, usando o charset
   * especificado para este <code>XMLWriter</code>.
   * 
   * @param stream o stream de sada.
   * 
   * @see #write(Writer)
   */
  final public void write(OutputStream stream) {
    write(new BufferedWriter(new OutputStreamWriter(stream, charset)));
  }

  /**
   * Cria um escritor que usa um mapa para criar tags a partir de objetos da
   * aplicao.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param objToTagMap mapa relacionando objetos a tags
   * @param dtd DTD (pode ser <code>null</code>)
   * @param charset codificao de caracteres do documento
   * @throws XMLException se o elemento XML retornado pela fbrica para o objeto
   *         fornecido for null.
   */
  public XMLWriter(final Object rootObject, Map<Class<?>, String> objToTagMap,
    final String dtd, final Charset charset) throws XMLException {
    this(rootObject, new XMLBasicElementFactory(null, objToTagMap), dtd,
      charset);
  }

  /**
   * Cria um escritor que usa um mapa para criar tags a partir de objetos da
   * aplicao.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param objToTagMap mapa relacionando objetos a tags
   * @param dtd DTD (pode ser <code>null</code>)
   * @param charset nome do charset usado no documento
   * @throws XMLException se o elemento XML retornado pela fbrica para o objeto
   *         fornecido for null.
   */
  public XMLWriter(final Object rootObject, Map<Class<?>, String> objToTagMap,
    final String dtd, final String charset) throws XMLException {
    this(rootObject, new XMLBasicElementFactory(null, objToTagMap), dtd,
      charset);
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param xmlFactory fbrica de elementos XML
   * @param dtd DTD a ser usado
   * @param charset codificao de caracteres do documento
   * @throws XMLException se o elemento XML retornado pela fbrica para o objeto
   *         fornecido for null.
   */
  public XMLWriter(final Object rootObject,
    final XMLElementFactoryInterface xmlFactory, final String dtd,
    final Charset charset) throws XMLException {
    if (rootObject == null) {
      throw new IllegalArgumentException(
        "objeto da aplicao no pode ser nulo");
    }
    if (xmlFactory == null) {
      throw new IllegalArgumentException("fbrica XML no pode ser nula");
    }
    this.xmlFactory = xmlFactory;
    this.charset = charset;
    this.dtd = dtd;
    xmlRoot = xmlFactory.createXMLElementFromApp(rootObject);
    if (xmlRoot == null) {
      throw new XMLException("Elemento <root> retornado da fbrica nulo!");
    }
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param xmlFactory fbrica de elementos XML
   * @param dtd DTD a ser usado
   * @param encoding codificao de caracteres do documento
   * @throws XMLException se houver algum erro na gerao do documento
   */
  public XMLWriter(final Object rootObject,
    final XMLElementFactoryInterface xmlFactory, final String dtd,
    final String encoding) throws XMLException {
    this(rootObject, xmlFactory, dtd, Charset.forName(encoding));
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param appContext contexto da aplicao, pode ser acessado pelos elementos
   *        XML quando estes esto sendo processados
   * @param xmlFactory fbrica de elementos XML
   * @param dtd DTD a ser usado
   * @param encoding codificao de caracteres do documento
   * @throws XMLException se houver algum erro na gerao do documento
   */
  public XMLWriter(final Object rootObject, final Object appContext,
    final XMLElementFactoryInterface xmlFactory, final String dtd,
    final String encoding) throws XMLException {
    this(rootObject, xmlFactory, dtd, Charset.forName(encoding));
    setappContext(appContext);
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param appContext contexto da aplicao, pode ser acessado pelos elementos
   *        XML quando estes esto sendo processados
   * @param xmlFactory fbrica de elementos XML
   * @param dtd DTD a ser usado
   * @param charset charset a ser usado na gerao do documento
   * @throws XMLException se houver algum erro na gerao do documento
   */
  public XMLWriter(final Object rootObject, final Object appContext,
    final XMLElementFactoryInterface xmlFactory, final String dtd,
    final Charset charset) throws XMLException {
    this(rootObject, xmlFactory, dtd, charset);
    setappContext(appContext);
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao. Usa o encoding default.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param xmlFactory fbrica de elementos XML
   * @param dtd DTD a ser usado
   * 
   * @see #XMLWriter(Object, XMLElementFactoryInterface, String, Charset)
   */
  public XMLWriter(final Object rootObject,
    final XMLElementFactoryInterface xmlFactory, final String dtd) {
    this(rootObject, xmlFactory, dtd, DEFAULT_CHARSET);
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao. Usa o encoding default.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param appContext contexto da aplicao, pode ser acessado pelos elementos
   *        XML quando estes esto sendo processados
   * @param xmlFactory fbrica de elementos XML
   * @param dtd DTD a ser usado
   * @see #XMLWriter(Object, XMLElementFactoryInterface, String, Charset)
   */
  public XMLWriter(final Object rootObject, final Object appContext,
    final XMLElementFactoryInterface xmlFactory, final String dtd) {
    this(rootObject, xmlFactory, dtd, DEFAULT_CHARSET);
    setappContext(appContext);
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao. Usa o encoding default.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param xmlFactory fbrica de elementos XML
   */
  public XMLWriter(final Object rootObject,
    final XMLElementFactoryInterface xmlFactory) {
    this(rootObject, xmlFactory, (String) null, DEFAULT_CHARSET);
  }

  /**
   * Cria um escritor que usa uma {@link XMLElementFactoryInterface fbrica}
   * para criar tags a partir de objetos da aplicao. Usa o encoding default.
   * 
   * @param rootObject objeto da aplicao correspondente  raiz do XML
   * @param appContext contexto da aplicao, pode ser acessado pelos elementos
   *        XML quando estes esto sendo processados
   * @param xmlFactory fbrica de elementos XML
   */
  public XMLWriter(final Object rootObject, final Object appContext,
    final XMLElementFactoryInterface xmlFactory) {
    this(rootObject, xmlFactory);
    setappContext(appContext);
  }

  /**
   * Fecha o writer.
   * 
   * @throws IOException se houver algum erro no fechamento do writer.
   */
  @Override
  public void close() throws IOException {
    XMLUtils.close(writer);
  }
}
