package tecgraf.javautils.xml.conversion;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.validation.Schema;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.DefaultHandler2;

import tecgraf.javautils.xml.XMLUtils;

/**
 * Fachada para um {@link DocumentBuilder}. Esta classe define um tratador de
 * erros que anexa as mensagens de erro em uma lista para posterior anlise se
 * necessrio. Alm disso, define um <i>entity resolver</i> para garantir que o
 * prefixo do DTD seja aplicado.
 */
public class ConversionDocumentBuilder extends DocumentBuilder {
  /**
   * {@link DocumentBuilder} encapsulado por esta classe.
   */
  private DocumentBuilder documentBuilder;
  /**
   * Lista de erros.
   */
  private List<String> errors;

  /**
   * Cria um construtor de documentos.
   * 
   * @param validate <code>true</code> para habilitar validao
   * @param dtdPrefix prefixo para o DTD (neste caso o prefixo se refere
   * @throws ParserConfigurationException se houver algum erro na configurao
   *         do parser
   */
  public ConversionDocumentBuilder(boolean validate, String dtdPrefix)
    throws ParserConfigurationException {
    this(validate, dtdPrefix, false);
  }

  /**
   * Cria um construtor de documentos.
   * 
   * @param validate <code>true</code> para habilitar validao
   * @param dtdPrefix prefixo para o DTD
   * @param fromCodebase <code>true</code> para indicar que o prefixo  relativo
   *        ao <i>codebase</i>
   * @throws ParserConfigurationException se houver algum erro na configurao do parser
   */
  public ConversionDocumentBuilder(boolean validate, String dtdPrefix,
    boolean fromCodebase) throws ParserConfigurationException {
    this(validate, new ConversionBaseHandler(dtdPrefix, fromCodebase));
  }

  /**
   * Construtor.
   *
   * @param validate indicativo de validao
   * @param resolver o resolver.
   * @throws ParserConfigurationException se houver algum erro na configurao do parser
   */
  public ConversionDocumentBuilder(boolean validate, EntityResolver resolver)
    throws ParserConfigurationException {
    errors = new ArrayList<>();
    DocumentBuilderFactory docBuilderFactory =
      DocumentBuilderFactory.newInstance();
    docBuilderFactory.setValidating(validate);
    documentBuilder = docBuilderFactory.newDocumentBuilder();
    documentBuilder.setEntityResolver(resolver);
    documentBuilder.setErrorHandler(new ConversionErrorHandler(errors));
  }

  /**
   * Retorna a lista de erros de processamento (coletados pelo tratdor de
   * erros).
   * 
   * @see ConversionErrorHandler
   * 
   * @return lista de erros de processamento.
   */
  public List<String> getErrorList() {
    return errors;
  }

  /**
   * Indica se houve erros durante o processamento do XML.
   * 
   * @return flag indicando se houve erros durante o processamento do XML
   */
  public boolean hasErrors() {
    return errors.size() > 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    return documentBuilder.equals(obj);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DOMImplementation getDOMImplementation() {
    return documentBuilder.getDOMImplementation();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Schema getSchema() {
    return documentBuilder.getSchema();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return documentBuilder.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isNamespaceAware() {
    return documentBuilder.isNamespaceAware();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isValidating() {
    return documentBuilder.isValidating();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isXIncludeAware() {
    return documentBuilder.isXIncludeAware();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Document newDocument() {
    return documentBuilder.newDocument();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Document parse(final File f) throws SAXException, IOException {
    return documentBuilder.parse(f);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Document parse(InputSource is) throws SAXException, IOException {
    return documentBuilder.parse(is);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Document parse(InputStream is, String systemId) throws SAXException,
    IOException {
    return documentBuilder.parse(is, systemId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Document parse(InputStream is) throws SAXException, IOException {
    return documentBuilder.parse(is);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Document parse(String uri) throws SAXException, IOException {
    return documentBuilder.parse(uri);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void reset() {
    documentBuilder.reset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setEntityResolver(EntityResolver er) {
    documentBuilder.setEntityResolver(er);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setErrorHandler(ErrorHandler eh) {
    documentBuilder.setErrorHandler(eh);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return documentBuilder.toString();
  }

}

/**
 * Handler de eventos para converses XML. Existe apenas para que possamos
 * interceptar a resoluo do DTD para prefix-lo com uma string arbitrria.
 */
class ConversionBaseHandler extends DefaultHandler2 {
  /**
   * Prefixo para o DTD (pode ser <code>null</code>).
   */
  private String dtdPrefix;

  /**
   * Indica se o prefixo do DTD  relativo ao JAR corrente.
   */
  private final boolean fromJar;

  /**
   * Instancia um handler para obteno dos DTDs de uma URL ou do filesystem
   * local.
   * 
   * @param dtdPrefix prefixo para os DTDs. Pode ser uma URL, um path do
   *        filesystem local ou <code>null</code>
   */
  public ConversionBaseHandler(String dtdPrefix) {
    this(dtdPrefix, false);
  }

  /**
   * Instancia um handler.
   * 
   * @param dtdPrefix prefixo para os DTDs. Pode ser uma URL, um path do
   *        filesystem local ou <code>null</code>. Se
   *        <code>fromCodebase == true</code> o prefixo deve ser relativo ao JAR
   * @param fromJar <code>true</code> para indicar que os DTDs sero obtidos do
   *        <i>codebase</i> corrente (tipicamente para uso em JARs),
   *        <code>false</code> para indicar que o prefixo  externo (URL).
   */
  public ConversionBaseHandler(String dtdPrefix, boolean fromJar) {
    this.fromJar = fromJar;
    this.dtdPrefix = dtdPrefix;
  }

  /**
   * Remove a '/' no incio de um path.
   * 
   * @param path path (pode ser <code>null</code>)
   * @return o path sem a '/' no incio ou <code>null</code> se o path era nulo
   */
  private static String removeLeadingSlash(String path) {
    if (path == null) {
      return null;
    }
    if (path.startsWith("/")) {
      return path.substring(1);
    }
    return path;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputSource resolveEntity(String name, String publicId,
    String baseURI, String systemId) throws SAXException, IOException {
    if (fromJar) {
      InputStream inputStream;
      String dtd = removeLeadingSlash(systemId);
      if (dtdPrefix == null) {
        inputStream = getClass().getResourceAsStream(dtd);
      }
      else {
        String dtdResource = dtdPrefix + '/' + dtd;
        inputStream = getClass().getResourceAsStream(dtdResource);
        if (inputStream == null) {
          throw new IOException("resource invlido: " + dtdResource);
        }
      }
      return new InputSource(inputStream);
    }
    String fixedURL = XMLUtils.fixUrlPrefix(systemId, dtdPrefix);
    return new InputSource(fixedURL);
  }
}

/**
 * Tratador de erros. Anexa as descries dos erros em uma lista de strings
 * recebida como parmetro.
 */
class ConversionErrorHandler implements ErrorHandler {
  /**
   * Lista de erros.
   */
  private final List<String> errorList;

  /**
   * Cria um tratador de erros.
   * 
   * @param errorList - lista que armazenar as descries dos erros encontrados
   */
  ConversionErrorHandler(List<String> errorList) {
    this.errorList = errorList;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void error(SAXParseException exception) throws SAXException {
    errorList.add("[Error] " + exception.getMessage());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void fatalError(SAXParseException exception) throws SAXException {
    errorList.add("[Fatal Error] " + exception.getMessage());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void warning(SAXParseException exception) throws SAXException {
    errorList.add("[Warning] " + exception.getMessage());
  }
}
