package tecgraf.javautils.xml.conversion;

import java.io.File;
import java.io.OutputStream;
import java.io.Writer;

import tecgraf.javautils.xml.conversion.exception.XMLConversionException;

/**
 * Interface que define os servios bsicos de uma converso XML. Uma converso
 * deve ser capaz de processar um arquivo XML validado por um DTD especfico e
 * gerar contedo validado por outro DTD.
 * <p>
 * O DTD de origem ou de destino de uma converso pode ser <code>null</code>,
 * indicando neste caso que o documento de origem ou de destino no possui a
 * declarao DOCTYPE.
 * <p>
 * OBS.: embora seja vlido que o DTD de <b>destino</b> seja <code>null</code>,
 * esta  uma situao pouco comum (provavelmente invlida), e deve ser aplicada
 * com cuidado.
 * <p>
 * Converses XML so usadas tipicamente quando o DTD usado por uma aplicao
 * foi alterado e a aplicao deve ser capaz de ler arquivos gerados para
 * verses anteriores do DTD.
 * 
 * @see XMLConverter
 */
public interface XMLConversionInterface {

  /**
   * Escreve o documento em um <code>OutputStream</code>. O documento no
   * necessariamente foi convertido.
   * 
   * @param outStream - stream de sada
   * @throws XMLConversionException se a entrada no foi definida
   */
  public abstract void writeXML(OutputStream outStream)
    throws XMLConversionException;

  /**
   * Escreve o documento em um <code>Writer</code>. O documento no
   * necessariamente foi convertido.
   * 
   * @param writer - Writer de sada
   * @throws XMLConversionException se a entrada no foi definida
   */
  public abstract void writeXML(Writer writer) throws XMLConversionException;

  /**
   * Escreve o documento em um arquivo. O documento no necessariamente foi
   * convertido.
   * 
   * @param fileName - nome do arquivo de sada
   * @throws XMLConversionException se o arquivo  invlido, se a entrada no
   *         foi definida ou se houve algum problema na gravao
   */
  public abstract void writeXML(String fileName) throws XMLConversionException;

  /**
   * Escreve o documento em um arquivo. O documento no necessariamente foi
   * convertido.
   * 
   * @param file - arquivo de sada
   * @throws XMLConversionException se o arquivo  invlido, se a entrada no
   *         foi definida ou se houve algum problema na gravao
   */
  public abstract void writeXML(File file) throws XMLConversionException;

  /**
   * Indica se o conversor trata arquivos correspondentes a um DTD especfico.
   * DTDs nulos (correspondentes a documentos sem a declarao DOCTYPE) so
   * aceitos.
   * 
   * @param dtd - DTD a ser comparado com o tratado pelo conversor. Pode ser
   *        nulo, sugerindo que o documento de origem no possui a declarao
   *        DOCTYPE
   * @return flag indicando se o conversor trata arquivos associados ao DTD
   *         especificado
   */
  public abstract boolean canConvertFrom(String dtd);

  /**
   * Indica se a converso  capaz de processar arquivos gerados por outra
   * converso. Usado para validar o encadeamento de converses.
   * 
   * @param conversion - converso anterior
   * @return flag indiciando se o encadeamento de converses  possvel
   */
  public abstract boolean canConvertFrom(XMLConversionInterface conversion);

  /**
   * Indica se o resultado da converso corresponde a um DTD.
   * 
   * @param dtd - o DTD a ser comparado
   * @return flag indicando se o DTD em questo corresponde  sada da converso
   */
  public abstract boolean convertsTo(String dtd);

  /**
   * Indica que tipos de arquivo este conversor gera.
   * 
   * @return o DTD associado  sada gerada por este conversor
   */
  public abstract String getTargetDTD();

  /**
   * Converte um documento.
   * 
   * @param doc - documento a ser convertido
   * @return documento convertido
   * @throws XMLConversionException se:
   *         <ul>
   *         <li>tipo do documento (DTD)  incompatvel
   *         <li>houve algum erro de I/O
   *         <li>houve algum erro no processamento do documento
   *         </ul>
   */
  public abstract ConversionDocument convert(ConversionDocument doc)
    throws XMLConversionException;

  /**
   * Converte um documento representado por um arquivo.
   * 
   * @param file - arquivo com o contedo a ser convertido
   * @param validate - indica se o documento deve ser validado pelo seu DTD
   * @return documento convertido
   * @throws XMLConversionException se:
   *         <ul>
   *         <li>tipo do documento (DTD)  incompatvel
   *         <li>houve algum erro de I/O
   *         <li>houve algum erro no processamento do documento
   *         </ul>
   */
  public abstract ConversionDocument convert(File file, boolean validate)
    throws XMLConversionException;

  /**
   * Converte a sada de uma outra converso (para ser usado em
   * <i>pipelines</i>).
   * 
   * @param previousConversion - converso recm-aplicada
   * @return documento convertido
   * @throws XMLConversionException se:
   *         <ul>
   *         <li>tipo do documento (DTD)  incompatvel
   *         <li>houve algum erro de I/O
   *         <li>houve algum erro no processamento do documento
   *         </ul>
   */
  public abstract ConversionDocument convert(
    XMLConversionInterface previousConversion) throws XMLConversionException;

  /**
   * Executa a converso.
   * <p>
   * <b>IMPORTANTE</b>: a implementao deste mtodo deve se certificar de que o
   * estado do conversor est coerente (p.ex. a entrada de dados foi definida).
   * 
   * @return documento convertido
   * 
   * @throws XMLConversionException se:
   *         <ul>
   *         <li> tipo do documento (DTD)  incompatvel <li> houve algum erro
   *         de I/O <li> houve algum erro no processamento do documento <li> a
   *         entrada no foi definida
   *         </ul>
   */
  public abstract ConversionDocument convert() throws XMLConversionException;

  /**
   * @return documento a ser convertido
   */
  public abstract ConversionDocument getDocument();

  /**
   * Define o documento a ser convertido.
   * 
   * @param doc documento a ser convertido
   */
  public abstract void setDocument(ConversionDocument doc);
}