/* $Id: XMLParseException.java 75313 2008-04-24 18:35:27Z costa $ */

package tecgraf.javautils.xml.exception;

import org.xml.sax.SAXParseException;

/**
 * A classe <code>XMLParseException</code> modela excees causadas no
 * processamento de arquivos XML.
 * 
 * @author costa
 */
public class XMLParseException extends XMLException {

  /**
   * Coluna associada ao final do texto onde ocorreu um erro de procesamento (<code>SAXParseException</code>).
   */
  private int columnNumber = -1;

  /**
   * Linha associada ao final do texto onde ocorreu um erro de procesamento (<code>SAXParseException</code>).
   */
  private int lineNumber = -1;

  /** Atributo */
  final protected String attribute;

  /**
   * Retorna o atributo associado  exceo.
   * 
   * @return o atributo associado  exceo (pode ser null)
   */
  final public String getAttribute() {
    return attribute;
  }

  /**
   * Apresenta os dados da exceo sob a forma de uma string. Informa a tag em
   * que ocorreu o erro e o atributo (se for o caso). Se as informaes sobre a
   * posio do erro (linha, coluna) estiverem disponveis, acrescenta-as
   * tambm.
   * 
   * @return o texto relativo a exceo com o atributo.
   */
  @Override
  public String toString() {
    String msg = "";
    if (tag != null) {
      msg += String.format("Tag: %s ", tag);
    }
    if (attribute != null) {
      msg += String.format("atributo: %s ", attribute);
    }
    if (lineNumber != -1) {
      msg += String.format("linha: %s ", lineNumber);
    }
    if (columnNumber != -1) {
      msg += String.format("coluna: %s ", columnNumber);
    }
    return msg;
  }

  /**
   * Constri uma <code>XMLParseException</code> recebendo a tag em que
   * ocorreu o erro e o nome do atributo que causou o erro.
   * 
   * @param tag tag em que ocorreu o erro
   * @param attribute atributo que causou o erro
   */
  public XMLParseException(final String tag, final String attribute) {
    super(tag);
    this.attribute = attribute;
  }

  /**
   * Constri uma <code>XMLParseException</code> encapsulando outra exceo.
   * 
   * @param tag tag do elemento que lanou a exceo
   * @param e exceo original
   * @param message mensagem associada a esta exceo (a exceo encapsulada
   *        pode ter sua prpria mensagem)
   */
  public XMLParseException(final String tag, final Exception e,
    final String message) {
    super(tag, e, message);
    this.attribute = null;
  }

  /**
   * Constri uma <code>XMLParseException</code>.
   * 
   * @param tag tag do elemento que lanou a exceo
   */
  public XMLParseException(final String tag) {
    super(tag);
    this.attribute = null;
  }

  /**
   * Constri uma <code>XMLParseException</code> encapsulando outra exceo.
   * 
   * @param tag tag do elemento que lanou a exceo
   * @param e exceo original
   */
  public XMLParseException(final String tag, final Exception e) {
    super(tag, e);
    this.attribute = null;
  }

  /**
   * Constri uma <code>XMLParseException</code> encapsulando outra exceo.
   * Se a exceo  uma <code>SAXParseException</code>, armazena a linha e a
   * coluna onde o erro ocorreu, conforme reportado pelo parser (este  o
   * construtor usado pelas callbacks de erro no handler).
   * 
   * @param e exceo original
   */
  public XMLParseException(final Exception e) {
    super(null, e);
    this.attribute = null;
    if (e instanceof SAXParseException) {
      SAXParseException spe = (SAXParseException) e;
      lineNumber = spe.getLineNumber();
      columnNumber = spe.getColumnNumber();
    }
  }

  /**
   * Constri uma <code>XMLParseException</code> encapsulando outra exceo,
   * sem associar a uma tag.
   * 
   * @param e exceo original
   * @param message mensagem associada a esta exceo (a exceo encapsulada
   *        pode ter sua prpria mensagem)
   */
  public XMLParseException(final Exception e, final String message) {
    super(null, e, message);
    this.attribute = null;
  }

  /**
   * Retorna a coluna do final do texto onde ocorreu um erro de processamento. A
   * primeira coluna  a de nmero 1.
   * 
   * @return coluna do final do texto onde ocorreu um erro de processamento.
   *         Caso esta informao no esteja disponvel, retorna -1.
   */
  public int getColumnNumber() {
    return columnNumber;
  }

  /**
   * Retorna a linha do final do texto onde ocorreu um erro de processamento. A
   * primeira linha  a de nmero 1.
   * 
   * @return linha do final do texto onde ocorreu um erro de processamento. Caso
   *         esta informao no esteja disponvel, retorna -1.
   */
  public int getLineNumber() {
    return lineNumber;
  }

}
