package busexplorer;

import scs.core.IComponent;
import tecgraf.diagnostic.addons.openbus.v20.OpenBusMonitor;
import tecgraf.diagnostic.commom.StatusCode;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.admin.BusAdmin;
import tecgraf.openbus.admin.BusAdminImpl;
import tecgraf.openbus.assistant.Assistant;
import tecgraf.openbus.assistant.AssistantParams;
import tecgraf.openbus.assistant.OnFailureCallback;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.UnauthorizedOperation;
import tecgraf.openbus.core.v2_0.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;

/**
 * Trata, analisa e armazena dados de login no barramento.
 *
 * @author Tecgraf
 */
public class BusExplorerLogin {
  /** Entidade. */
  public final String entity;
  /** Host. */
  public final String host;
  /** Porta. */
  public final int port;
  /** Instncia de administrao do baramento. */
  private final BusAdmin admin;
  /** Indica se o login possui permisses administrativas. */
  private boolean adminRights = false;
  /** Assistente de conexo ao barramento. */
  private Assistant assistant;

  /**
   * Construtor
   *
   * @param admin Instncia de administrao do barramento.
   * @param entity Entidade.
   * @param host Host.
   * @param port Porta.
   */
  public BusExplorerLogin(BusAdmin admin, String entity, String host, int port)
  {
    this.admin = admin;
    this.entity = entity;
    this.host = host;
    this.port = port;
  }

  /**
   * Indica se o usurio autenticado no momento possui permisso de
   * administrao.
   * 
   * @return <code>true</code> se possui permisso de administrao e
   *         <code>false</code> caso contrrio. Se o login no foi realizado, o
   *         retorno ser <code>false</code>.
   */
  public boolean hasAdminRights() {
    return adminRights;
  }

  /**
   * Obtm o assistente utilizado para o login.
   *
   * @return o assistente utilizado para o login.
   */
  public Assistant getAssistant() {
    return assistant;
  }

  /**
   * Realiza o shutdown do ORB e do assistente, terminando o login.
   */
  public void logout() {
    if (assistant == null) {
      return;
    }
    org.omg.CORBA.ORB orb = assistant.orb();
    assistant.shutdown();
    orb.shutdown(true);
  }

  /**
   * Realiza uma verificao sobre a permisso de administrao deste login e
   * armazena o resultado em uma membro auxiliar.
   *
   * @throws ServiceFailure
   */
  private void checkAdminRights() throws ServiceFailure {
    try {
      admin.getLogins();
      adminRights = true;
    }
    catch (UnauthorizedOperation e) {
      adminRights = false;
    }
  }

  /**
   * Conecta-se a uma instncia de administrao do barramento.
   */
  private void connectToAdmin() {
    if (admin instanceof BusAdminImpl) {
      ((BusAdminImpl) admin).connect(host, port, assistant.orb());
    }
  }

  /**
   * Realiza o login.
   *
   * @param login Informaes de login.
   * @param password Senha.
   */
  public static void doLogin(BusExplorerLogin login, String password) throws
    Exception {
    /** Intervalo de tempo para verificar se o login j foi efetuado */
    final int LOGIN_CHECK_INTERVAL = 250;
    /** Nmero mximo de tentativas de login */
    final int MAX_LOGIN_FAILS = 3;

    OnFailureCallbackWithException callback =
     new OnFailureCallbackWithException() {
      volatile int failedAttempts = 0;

      volatile Exception exception = null;

      @Override
      public void onStartSharedAuthFailure(Assistant arg0, Exception arg1) {
        // no iremos utilizar este recurso
      }

      @Override
      public void onRegisterFailure(Assistant arg0, IComponent arg1,
        ServiceProperty[] arg2, Exception arg3) {
        // no iremos utilizar este recurso
      }

      @Override
      public void onLoginFailure(Assistant arg0, Exception arg1) {
        if (++failedAttempts == MAX_LOGIN_FAILS ||
          arg1 instanceof AccessDenied) {
          exception = (Exception) arg1;
        }
      }

      @Override
      public void onFindFailure(Assistant arg0, Exception arg1) {
        // TODO precisamos realizar algum tratamento aqui?
      }

      @Override
      public Exception getException() {
        return exception;
      }
    };

    AssistantParams params = new AssistantParams();
    params.callback = callback;

    login.assistant =
      Assistant.createWithPassword(login.host, login.port, login.entity,
        password.getBytes(), params);

    OpenBusMonitor monitor =
      new OpenBusMonitor("openbus", (OpenBusContext)
        login.assistant.orb().resolve_initial_references("OpenBusContext"));

    while (true) {
      if (monitor.checkResource().code == StatusCode.OK) {
        login.connectToAdmin();
        login.checkAdminRights();
        break;
      }
      if (callback.getException() != null) {
        login.logout();
        throw callback.getException();
      }

      try {
        Thread.sleep(LOGIN_CHECK_INTERVAL);
      }
      catch (InterruptedException e) {
      }
    }
  }
  
  /**
   * Extenso da callback de notificao de falhas do Assistente OpenBus.
   * Permite a recuperao de possveis excees.
   */
  private interface OnFailureCallbackWithException extends OnFailureCallback {
    /**
     * Recupera uma possvel exceo durante a execuo de uma das tarefas do
     * assistente.
     *
     * @return Uma exceo durante a execuo de uma tarefa. Nulo, se no houve
     * disparo de exceo.
     */
    public Exception getException();
  }

}
