package br.pucrio.tecgraf.soma.job.domain.model;

import org.hibernate.annotations.CollectionId;
import org.hibernate.annotations.Fetch;
import org.hibernate.annotations.FetchMode;
import org.hibernate.annotations.Type;

import javax.persistence.*;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * Job Entity
 */
@Entity
@Table(name="jobs")
public class Job {

	// Identifier
	@Id
	@GeneratedValue(strategy = GenerationType.SEQUENCE, generator = "jobs_generator")
	@SequenceGenerator(name="jobs_generator", sequenceName = "public.jobs_seq", allocationSize=1)
	@Column(name = "id", updatable = false, nullable = false)
	private Long id;

	// Execution system's job identifier
	@Column(name="job_id", unique = true)
	private String jobId;

	// Group identifier
	@Column(name="group_id", nullable=true)
	private String groupId = null;

	// Project associated with the job
	@Column(name="project_id", nullable=false)
	private String projectId = null;

	// User that submitted the job
	@Column(name="user_id", nullable=false)
	private String jobOwner = null;

	// Flag to indicate option for automatically machine selection
	@Column(name="auto_machine_selected", nullable=false)
	private Boolean automaticallyMachineSelection = null;

	// Machines selected by the user
	@ElementCollection
	@CollectionTable(
			name="job_selected_machines",
			joinColumns=@JoinColumn(name="job_id")
	)
	@Column(name="machine_name")
	@GeneratedValue(strategy = GenerationType.SEQUENCE, generator = "selected_machine_generator")
	@SequenceGenerator(name = "selected_machine_generator", sequenceName = "public.job_selected_machines_seq", allocationSize=1)
	@CollectionId(columns = {@Column(name="id")},
								generator="selected_machine_generator", type=@Type(type="long"))
	private List<String> selectedMachines = new ArrayList<>();

	// The total number of processes. In the case of multiple execution, this number can be greater than one.
	@Column(name="number_of_processes", nullable=false)
	private Integer numberOfProcesses = 1;

	// The number of processes by machine. In the case of multiple execution, this number can be greater than one."
	@Column(name="number_of_processes_by_machine", nullable=false)
	private Integer numberOfProcessesByMachine = 1;

	// The timestamp of job submission
	@Column(name="submission_time", nullable=false)
	private LocalDateTime submissionTime = null;

	// Description
	@Column(name="description", nullable=true, length=500)
	private String description = null;

	// Priority
	@Column(name="priority", nullable=false)
	private Integer priority = null;

	// If the job submission request a multiple execution
	@Column(name="multiple_execution", nullable=false)
	private Boolean multipleExecution = false;

	// If the job executes a simple algorithm or a flow
	@Column(name="job_type", nullable=false)
	@Enumerated(EnumType.STRING)
	private JobSpecificationType jobType = JobSpecificationType.ALGORITHM;

	// The number of times the job has already being re-schedule
	@Column(name="number_of_retries", nullable=false)
	private Integer numberOfRetries = 0;

	// The machine where the job executed
	@Column(name="execution_machine", nullable=true)
	private String executionMachine = null;

	// The timestamp when the job finished its execution
	@Column(name="end_time", nullable=true)
	private LocalDateTime endTime = null;

	// The job exit code
	@Column(name="exit_code", nullable=true)
	private Integer exitCode = null;

	// The id of the algorithm of the flow that caused the job to finish earlier than expected.
	@Column(name="guilty_node_id", nullable=true)
	private String guiltyNodeId = null;

	// The status of the job after execution
	@Enumerated(EnumType.STRING)
	@Column(name="exit_status", nullable=true)
	private ExitStatus exitStatus = null;

	// The amount of CPU time used for processing the job
	@Column(name="cpu_time", nullable=true)
	private Double cpuTime = null;

	// The actual amount of time taken to run the job
	@Column(name="wall_clock_time", nullable=true)
	private Integer wallclockTime = null;

	// The amount of memory used to run the job
	@Column(name="ram_mem", nullable=true)
	private Double ramMemory = null;

	// The list of events (with timestamp) representing the status change history of the job
	@OneToMany(mappedBy="job", fetch=FetchType.EAGER, cascade=CascadeType.ALL, orphanRemoval=true)
	@OrderBy("timestamp")
	@Fetch(value = FetchMode.SUBSELECT)
	private List<JobStatusHistory> statusHistory = new ArrayList<>();

	@Column(name="flow_id", nullable=true)
	// If the job specification type is a installed flow, the identification of this flow
	private String flowId = null;

	// If the job specification type is a installed flow, the version of this flow
	@Column(name="flow_version", nullable=true)
	private String flowVersion;

	// If the job specification type is a installed flow, the name of this flow
	@Column(name="flow_name", nullable=true)
	private String flowName;

	// If the job specification type is a installed flow, the configuration of this flow
	@Column(name="flow_raw", nullable=true)
	@Type(type = "org.hibernate.type.BinaryType")
	private byte[] flowRaw;

	// The timestamp when the job was last modified
	@Column(name="last_modified_time", nullable=true)
	private LocalDateTime lastModifiedTime = null;

	// The algorithms associated with this job execution
	@OneToMany(mappedBy="job", fetch=FetchType.EAGER, cascade=CascadeType.ALL, orphanRemoval=true)
	@OrderBy("id")
	@Fetch(value = FetchMode.SUBSELECT)
	private List<JobAlgorithm> algorithms = new ArrayList<>();

	// Soft delete flag
	@Column(name="is_deleted", nullable=false)
	private boolean isDeleted = false;

	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	public String getJobId() {
		return jobId;
	}

	public void setJobId(String jobId) {
		this.jobId = jobId;
	}

	public String getProjectId() {
		return projectId;
	}

	public void setProjectId(String projectId) {
		this.projectId = projectId;
	}

	public String getJobOwner() {
		return jobOwner;
	}

	public void setJobOwner(String jobOwner) {
		this.jobOwner = jobOwner;
	}

	public String getGroupId() {
		return groupId;
	}

	public void setGroupId(String groupId) {
		this.groupId = groupId;
	}

	public Boolean getAutomaticallyMachineSelection() {
		return automaticallyMachineSelection;
	}

	public void setAutomaticallyMachineSelection(Boolean automaticallyMachineSelection) {
		this.automaticallyMachineSelection = automaticallyMachineSelection;
	}

	public List<String> getSelectedMachines() {
		return selectedMachines;
	}

	public void setSelectedMachines(List<String> selectedMachines) {
		this.selectedMachines = selectedMachines;
	}

	public Integer getNumberOfProcesses() {
		return numberOfProcesses;
	}

	public void setNumberOfProcesses(Integer numberOfProcesses) {
		this.numberOfProcesses = numberOfProcesses;
	}

	public Integer getNumberOfProcessesByMachine() {
		return numberOfProcessesByMachine;
	}

	public void setNumberOfProcessesByMachine(Integer numberOfProcessesByMachine) {
		this.numberOfProcessesByMachine = numberOfProcessesByMachine;
	}

	public LocalDateTime getSubmissionTime() {
		return submissionTime;
	}

	public void setSubmissionTime(LocalDateTime submissionTime) {
		this.submissionTime = submissionTime;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public Integer getPriority() {
		return priority;
	}

	public void setPriority(Integer priority) {
		this.priority = priority;
	}

	public Boolean getMultipleExecution() {
		return multipleExecution;
	}

	public void setMultipleExecution(Boolean multipleExecution) {
		this.multipleExecution = multipleExecution;
	}

	public JobSpecificationType getJobType() {
		return jobType;
	}

	public void setJobType(JobSpecificationType jobType) {
		this.jobType = jobType;
	}

	public Integer getNumberOfRetries() {
		return numberOfRetries;
	}

	public void setNumberOfRetries(Integer numberOfRetries) {
		this.numberOfRetries = numberOfRetries;
	}

	public String getExecutionMachine() {
		return executionMachine;
	}

	public void setExecutionMachine(String executionMachine) {
		this.executionMachine = executionMachine;
	}

	public LocalDateTime getEndTime() {
		return endTime;
	}

	public void setEndTime(LocalDateTime endTime) {
		this.endTime = endTime;
	}

	public Integer getExitCode() {
		return exitCode;
	}

	public void setExitCode(Integer exitCode) {
		this.exitCode = exitCode;
	}

	public String getGuiltyNodeId() {
		return guiltyNodeId;
	}

	public void setGuiltyNodeId(String guiltyNodeId) {
		this.guiltyNodeId = guiltyNodeId;
	}

	public ExitStatus getExitStatus() {
		return exitStatus;
	}

	public void setExitStatus(ExitStatus exitStatus) {
		this.exitStatus = exitStatus;
	}

	public Double getCpuTime() {
		return cpuTime;
	}

	public void setCpuTime(Double cpuTime) {
		this.cpuTime = cpuTime;
	}

	public Integer getWallclockTime() {
		return wallclockTime;
	}

	public void setWallclockTime(Integer wallclockTime) {
		this.wallclockTime = wallclockTime;
	}

	public Double getRamMemory() {
		return ramMemory;
	}

	public void setRamMemory(Double ramMemory) {
		this.ramMemory = ramMemory;
	}

	public List<JobStatusHistory> getStatusHistory() {
		return statusHistory;
	}

	public void addStatusHistory(JobStatusHistory jobStatusHistory) {
		jobStatusHistory.setJob(this);
		this.statusHistory.add(jobStatusHistory);
	}

	public String getFlowId() {
		return flowId;
	}

	public void setFlowId(String flowId) {
		this.flowId = flowId;
	}

	public String getFlowVersion() {
		return flowVersion;
	}

	public void setFlowVersion(String flowVersion) {
		this.flowVersion = flowVersion;
	}

	public String getFlowName() {
		return flowName;
	}

	public void setFlowName(String flowName) {
		this.flowName = flowName;
	}

	public byte[] getFlowRaw() {
		return flowRaw;
	}

	public void setFlowRaw(byte[] flowRaw) {
		this.flowRaw = flowRaw;
	}

	public LocalDateTime getLastModifiedTime() {
		return lastModifiedTime;
	}

	public void setLastModifiedTime(LocalDateTime lastModifiedTime) {
		this.lastModifiedTime = lastModifiedTime;
	}

	public List<JobAlgorithm> getAlgorithms() {
		return algorithms;
	}

	public void addAlgorithm(JobAlgorithm algorithm) {
		algorithm.setJob(this);
		this.algorithms.add(algorithm);
	}

	public boolean isDeleted() {
		return isDeleted;
	}

	public void setDeleted(boolean deleted) {
		isDeleted = deleted;
	}

	public void setAlgorithmExitCode(Integer flowNodeId, Integer exitCode) {
		for(JobAlgorithm algorithm: algorithms) {
			if(algorithm.getFlowNodeId() == flowNodeId) {
				algorithm.setFlowNodeExitCode(exitCode);
				break;
			}
		}
	}
}
