package com.github.tennaito.rsql.jpa;

import com.github.tennaito.rsql.builder.BuilderTools;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import jakarta.persistence.criteria.From;
import jakarta.persistence.criteria.JoinType;
import jakarta.persistence.criteria.Path;
import jakarta.persistence.metamodel.Attribute;
import jakarta.persistence.metamodel.ManagedType;
import jakarta.persistence.metamodel.Metamodel;
import jakarta.persistence.metamodel.PluralAttribute;

public class JpaPathCache {
  private static final Logger LOG = Logger.getLogger(PredicateBuilder.class.getName());
  private final Map<String, Path> cache = new HashMap<String, Path>();
  private final Metamodel metaModel;
  private final BuilderTools misc;

  JpaPathCache(From root, Metamodel metaModel, BuilderTools misc) {
    cache.put("", root);
    this.metaModel = metaModel;
    this.misc = misc;
  }

  /**
   * Verifies if a class metamodel has the specified property.
   *
   * @param property      Property name.
   * @param classMetadata Class metamodel that may hold that property.
   * @return <tt>true</tt> if the class has that property, <tt>false</tt> otherwise.
   */
  private static <T> boolean hasPropertyName(String property, ManagedType<T> classMetadata) {
    Set<Attribute<? super T, ?>> names = classMetadata.getAttributes();
    for (Attribute<? super T, ?> name : names) {
      if (name.getName().equals(property)) return true;
    }
    return false;
  }

  /**
   * Get the property Type out of the metamodel.
   *
   * @param attribMeta Reference attribute metamodel.
   * @return Class java type for the property,
   * if the property is a pluralAttribute it will take the bindable java type of that collection.
   */
  private static <T> Class<?> findPropertyType(Attribute attribMeta) {
    if (attribMeta.isCollection()) {
      return ((PluralAttribute) attribMeta).getBindableJavaType();
    }
    return attribMeta.getJavaType();
  }

  private Path getPropertyPath(From entityPath, ManagedType entityManaged, String propertyName) {
    if (!hasPropertyName(propertyName, entityManaged)) {
      throw new IllegalArgumentException("Unknown property: " + propertyName + " from entity " + entityManaged.getJavaType().getName());
    }
    Attribute<?,?> attribMeta = entityManaged.getAttribute(propertyName);
    String entityTypeName = entityManaged.getJavaType().getName();

    if (attribMeta.isAssociation()) {
      LOG.log(Level.INFO, "Create a join between {0} and {1}.", new Object[]{entityTypeName, findPropertyType(attribMeta).getName()});
      return entityPath.join(propertyName, JoinType.LEFT);
    }
    if (attribMeta.isCollection()) {
      Class<?> collectionType = attribMeta.getJavaType();
      LOG.log(Level.INFO, "Create a collection join between {0} and {1}.", new Object[]{entityTypeName, collectionType.getName()});

      if (List.class.isAssignableFrom(collectionType)) {
        return entityPath.joinList(propertyName);
      }
      else if (Map.class.isAssignableFrom(collectionType)) {
        return entityPath.joinMap(propertyName);
      }
      else if (Set.class.isAssignableFrom(collectionType)) {
        return entityPath.joinSet(propertyName);
      }
      return entityPath.joinCollection(propertyName);
    }
    LOG.log(Level.INFO, "Create property path for type {0} property {1}.", new Object[]{entityTypeName, propertyName});
    return entityPath.get(propertyName);

//  TODO: create a test case for the scenario of an embedded attribute.
//  if (attribMeta.getPersistentAttributeType() == Attribute.PersistentAttributeType.EMBEDDED) {
//    Class<?> embeddedType = findPropertyType(attribMeta);
//    classMetadata = metaModel.managedType(embeddedType);
//  }
  }

  public Path getPath(String fullName) {
    Path found = cache.get(fullName);

    if (found != null) {
      return found;
    }

    From entityPath;
    String entityName;
    String propertyName;
    int splitIdx = fullName.lastIndexOf('.');
    if (splitIdx == -1) {
      entityName = "";
      propertyName = fullName;
    }
    else {
      entityName = fullName.substring(0, splitIdx);
      propertyName = fullName.substring(splitIdx + 1);
    }
    entityPath = (From) getPath(entityName);
    Path path;
    Class entityType = entityPath.getJavaType();
    String propertyAlias = misc.getPropertiesMapper().translate(propertyName, entityType);
    if (!propertyName.equals(propertyAlias)) {
      path = getPath(entityName + "." + propertyAlias);
    } else {
      ManagedType<?> entityManaged = metaModel.managedType(entityType);
      path = getPropertyPath(entityPath, entityManaged, propertyName);
    }
    cache.put(fullName, path);
    return path;
  }
}
