package br.pucrio.tecgraf.soma.job.application.configuration;

import javax.ws.rs.NotFoundException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.security.authentication.AuthenticationCredentialsNotFoundException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.io.IOException;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import br.pucrio.tecgraf.soma.job.application.appservice.JobAppService;
import br.pucrio.tecgraf.soma.job.domain.JobEventVO;

@Component
public class SocketHandler extends TextWebSocketHandler {

  @Autowired
  private JobAppService jobAppService;

  private List<WebSocketSession> sessions = new CopyOnWriteArrayList<>();

  private Map<String, List<String>> projectsBySession = new ConcurrentHashMap<>();

  public SocketHandler() {
    super();
  }

  @Override
  public void handleTextMessage(WebSocketSession session, TextMessage message)
  throws InterruptedException, IOException {
    if (!session.isOpen()) {
      this.removeSession(session);
      return;
    }
    session.sendMessage(message);
  }

  public void sendMessage(JobEventVO jobEvent) throws IOException, InterruptedException {
    for (WebSocketSession webSocketSession : sessions) {
      String sessionId = webSocketSession.getId();
      if (this.projectsBySession.containsKey(sessionId)) {
        List<String> projects = this.projectsBySession.get(sessionId);
        if (projects.contains(jobEvent.getProjectId())) {
          this.handleTextMessage(webSocketSession, new TextMessage(jobEvent.toJson()));
        }
      }
    }
  }

  @Override
  public void afterConnectionEstablished(WebSocketSession session) throws Exception {
    // Obter os dados da requisição via Uri
    URI sessionUri = session.getUri();
    String accessToken = sessionUri.getQuery();
    if (accessToken == null || accessToken.isEmpty()) {
      throw new AuthenticationCredentialsNotFoundException("Authorization token is required");
    }

    List<String> projects = jobAppService.getUserProjects(accessToken);
    if (projects.isEmpty()) {
      throw new NotFoundException("User has access to no projects");
    }

    // Guarda o token no mapa de tokens.
    this.projectsBySession.put(session.getId(), projects);

    // Salva a sessao
    this.sessions.add(session);
  }

  @Override
  public void afterConnectionClosed(WebSocketSession session, CloseStatus status) throws Exception {
    this.removeSession(session);
  }

  private void removeSession(WebSocketSession session) {
    this.projectsBySession.remove(session.getId());
    this.sessions.remove(session);
  }
}