package br.pucrio.tecgraf.soma.job.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import br.pucrio.tecgraf.soma.job.Algorithm;
import br.pucrio.tecgraf.soma.job.AlgorithmParameter;
import br.pucrio.tecgraf.soma.job.ExitStatus;
import br.pucrio.tecgraf.soma.job.Flow;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.flows.configurator.Node;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.serializer.FlowAlgorithmConfigurationSerializer;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerException;

/**
 * Classe para converter os dados do comando CSBase para o tipo definido no
 * schema.
 */
public class CSBaseCommandUtil {

	/**
	 * Converte um parâmetro do CSBase para o tipo definido no schema
	 *
	 * @param p
	 *            o parâmetro vendo do CSBase
	 * @return o parâmetro convertido para o schema
	 */
	public static AlgorithmParameter convertFromCSBaseParameter(SimpleParameter<?> p) {
		AlgorithmParameter param = new AlgorithmParameter();
		param.setParameterId(p.getName());
		param.setLabel(p.getLabel());
		param.setType(p.getType());
		param.setValue(p.getValueAsJSON());
		return param;
	}

	/**
	 * Cria os dados de fluxo a partir do configurador do fluxo.
	 * 
	 * @param flowConf
	 *            configurador do fluxo
	 * @return os dados do fluxo para a mensagem.
	 */
	public static Flow createFlowFromConfig(FlowAlgorithmConfigurator flowConf) {
		Flow flow = new Flow();
		// TODO: Pegar os dados de fluxo instalado do serviço de algoritmos
		if (flowConf.getAlgorithmId() != null) {
			flow.setFlowId(flowConf.getAlgorithmId());
		} else {
			flow.setFlowId("");
		}

		if (flowConf.getAlgorithmVersion() != null) {
			flow.setFlowVersion(flowConf.getAlgorithmVersion().toString());
		} else {
			flow.setFlowVersion("");
		}

		if (flowConf.getAlgorithmName() != null) {
			flow.setFlowName(flowConf.getAlgorithmName());
		} else {
			flow.setFlowName("");
		}

		List<Algorithm> algorithms = new ArrayList<>();
		for (Node node : flowConf.getNodes()) {
			algorithms.add(createAlgorithmFromConfig(node.getConfigurator()));
		}
		flow.setAlgorithms(algorithms);

		FlowAlgorithmConfigurationSerializer serializer = new FlowAlgorithmConfigurationSerializer();
		try (ByteArrayOutputStream serializerOutput = new ByteArrayOutputStream()) {
			serializer.write(flowConf, serializerOutput);
			byte[] configuratorData = serializerOutput.toByteArray();
			flow.setRaw(ByteBuffer.wrap(configuratorData));
		} catch (AlgorithmConfigurationSerializerException | IOException e) {
			e.printStackTrace();
		}
		return flow;
	}

	/**
	 * Cria os dados de algoritmo a partir do configurador do algoritmo.
	 * 
	 * @param algoConf
	 *            configurador do algoritmo
	 * @return os dados do algoritmo para a mensagem.
	 */
	public static Algorithm createAlgorithmFromConfig(AlgorithmConfigurator algoConf) {
		Algorithm algorithm = new Algorithm();
		algorithm.setAlgorithmId(algoConf.getAlgorithmId());
		algorithm.setAlgorithmVersion(algoConf.getAlgorithmVersion().toString());
		algorithm.setAlgorithmName(algoConf.getAlgorithmName());

		List<AlgorithmParameter> params = ((SimpleAlgorithmConfigurator) algoConf).getSimpleParameters().stream()
				.map(CSBaseCommandUtil::convertFromCSBaseParameter).collect(Collectors.toList());
		algorithm.setParameters(params);
		return algorithm;
	}

	/**
	 * Faz o mapeamento entre a enumeração de status de finalização de comando do
	 * schema e do csbase.
	 *
	 * @param finalizationInfo
	 *            informações de finalização do comando
	 * @return o status de finalização do comando no schema.
	 */
	public static ExitStatus getExitStatusFromInfo(CommandFinalizationInfo finalizationInfo) {
		switch (finalizationInfo.getFinalizationType()) {
		case SUCCESS:
			return ExitStatus.SUCCESS;
		case LOST:
			return ExitStatus.LOST;
		case EXECUTION_ERROR:
			return ExitStatus.EXECUTION_ERROR;
		case KILLED:
			return ExitStatus.KILLED;
		case FAILED:
			switch (finalizationInfo.getFailureCause()) {
			case UNKNOWN:
				return ExitStatus.UNKNOWN;
			case COMMAND_IDENTIFIER_NOT_FOUND:
				return ExitStatus.COMMAND_IDENTIFIER_NOT_FOUND;
			case SGA_EXECUTION_ERROR:
				return ExitStatus.UNEXPECTED_MACHINE_ERROR;
			case FAILED_SETUP_EXECUTION_ENVIRONMENT:
				return ExitStatus.FAILED_SETUP_EXECUTION_ENVIRONMENT;
			case PROJECT_NOT_FOUND:
				return ExitStatus.PROJECT_NOT_FOUND;
			case NO_SGA_AVAILABLE_TO_ROOT_COMMAND:
			case SGA_IS_NOT_AVAILABLE:
				return ExitStatus.NO_MACHINE_AVAILABLE;
			case USER_WITHOUT_PERMISSION_FOR_EXECUTION:
				return ExitStatus.NO_PERMISSION;
			default:
				return ExitStatus.UNDEFINED;
			}
		case NO_EXIT_CODE:
		case END:
		case UNKNOWN:
			return ExitStatus.UNKNOWN;
		default:
			return ExitStatus.UNDEFINED;
		}
	}

}
