package br.pucrio.tecgraf.soma.job.application.service;

import br.pucrio.tecgraf.soma.job.application.filter.ServiceRequestContext;
import br.pucrio.tecgraf.soma.serviceapi.configuration.ServiceConfiguration;
import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.ws.rs.ServiceUnavailableException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

@Service
public class ProjectService {
  private static final Logger LOG = Logger.getLogger(ProjectService.class);

  @Value("${job.history.project.permission.check.enable}")
  public boolean checkProjectPermissionEnabled;

  @Value("${job.history.project.permission.check.url}")
  public String projectPermissionURL;

  @Autowired private ServiceConfiguration config;

  @Autowired private RestTemplate restTemplate;

  @Autowired private ServiceRequestContext serviceRequestContext;

  public boolean hasPermission(String project) throws ServiceUnavailableException {
    if (!checkProjectPermissionEnabled) {
      LOG.info(String.format("Skipping user permission check to project %s", project));
      return true;
    } else {
      LOG.info(String.format("Checking user permission to project %s", project));
      List<String> projects = getUserProjects();

      return projects.contains(project);
    }
  }

  /**
   * Busca os projetos que o usuário tem acesso via API REST do serviço de projetos do CSGrid.
   *
   * @return a lista de ids dos projetos que o usuário tem acesso.
   */
  public List<String> getUserProjects() throws ServiceUnavailableException {
    UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(projectPermissionURL);

    String locale = serviceRequestContext.getLocale();
    if (locale != null) {
      builder.queryParam("locale", serviceRequestContext.getLocale());
    }

    String token = serviceRequestContext.getToken();
    if (token == null) {
      LOG.error("Null token");
      throw new ServiceUnavailableException("Failed retrieving user projects");
    }

    HttpHeaders headers = new HttpHeaders();
    headers.setAccept(Collections.singletonList(MediaType.APPLICATION_JSON));
    headers.set(HttpHeaders.AUTHORIZATION, token);

    try {
      ResponseEntity<String[]> response =
          restTemplate.exchange(
              builder.toUriString(), HttpMethod.GET, new HttpEntity<>(headers), String[].class);
      return Arrays.asList(Objects.requireNonNull(response.getBody()));
    } catch (RestClientException e) {
      String errorMsg = "Failed retrieving user projects";
      LOG.error(errorMsg, e);
      throw new ServiceUnavailableException(errorMsg);
    }
  }
}
