package br.pucrio.tecgraf.soma.job.application.controller;

import java.util.List;

import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.transaction.CannotCreateTransactionException;
import org.springframework.web.client.HttpClientErrorException;

import br.pucrio.tecgraf.soma.job.api.NotFoundException;
import br.pucrio.tecgraf.soma.job.api.ReplicaApi;
import br.pucrio.tecgraf.soma.job.api.model.ReplicaJob;
import br.pucrio.tecgraf.soma.job.api.model.ReplicaJobBatchResponse;
import br.pucrio.tecgraf.soma.job.api.model.ReplicaJobResponse;
import br.pucrio.tecgraf.soma.job.application.appservice.ReplicaAppService;
import jakarta.persistence.NoResultException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotNull;
import jakarta.ws.rs.ForbiddenException;
import jakarta.ws.rs.ServiceUnavailableException;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.SecurityContext;

@Component
public class ReplicaController extends ReplicaApi {
  private static final Logger LOG = Logger.getLogger(ReplicaController.class);

  @Autowired
  private HttpServletRequest request;
  @Autowired
  private ReplicaAppService replicaAppService;

  public ReplicaController() {
    super(null);
  }

  @Override
  public Response replicaPost(@NotNull @Valid ReplicaJob replicaJobData, String locale, SecurityContext securityContext)
    throws NotFoundException {

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    ReplicaJobResponse replicaJobResponse;
    try {
      replicaJobResponse = replicaAppService.createReplicaJob(replicaJobData);
    }
    catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Database connection error creating a Replica: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    }
    catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Permission error creating a Replica: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    }
    catch (NoResultException e) { // Multifluxo não encontrado
      String errorMsg = "Error creating a Replica. Multiflow not found: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.NOT_FOUND.value()).entity(errorMsg).build();
    }
    catch (HttpClientErrorException e) { // Job ID já está associado a alguma réplica
      String errorMsg = "HTTP error creating a Replica. " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(e.getStatusCode().value()).entity(e.getStatusText()).build();
    }
    catch (Exception e) { // Demais erros
      String errorMsg = "Unknown error creating a Replica: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.CREATED.value()).entity(replicaJobResponse).build();
  }

  @Override
  public Response replicaBatchPost(@NotNull @Valid List<ReplicaJob> replicaJobs, String locale,
    SecurityContext securityContext) throws NotFoundException {

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    if (replicaJobs == null || replicaJobs.size() <= 0) {
      String errorMsg = "Empty batch of jobs. ";
      LOG.info(errorMsg);
      return Response.status(HttpStatus.BAD_REQUEST.value()).entity(errorMsg).build();
    }

    List<ReplicaJobBatchResponse> replicaJobResponse;
    try {
      replicaJobResponse = replicaAppService.createReplicaJobs(replicaJobs);
    }
    catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o
      // BD
      String errorMsg = "Database connection error creating a Replica: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    }
    catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Permission error creating a Replica: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    }
    catch (NoResultException e) { // Multifluxo não encontrado
      String errorMsg = "Error creating a Replica. Multiflow not found: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.NOT_FOUND.value()).entity(errorMsg).build();
    }
    catch (Exception e) { // Demais erros
      String errorMsg = "Unknown error creating a Replica: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.CREATED.value()).entity(replicaJobResponse).build();
  }
}
