/**
 *  Several Javascript helper methods.
 *
 *  Leonardo Quijano
 *  Feb 2006
 *
 **/

dojo.require("dojo.regexp");

/** ********************* HELPER FUNCTIONS ********************* */

/**
 * Returns true if the specified string is a single digit ("3", for example).
 * @param str the string to test.
 * @return True if it's a single digit.
 */
function isDigit(str) {
    if (str.length > 1) {
        return false;
    }

    var string = "1234567890";
    return (string.indexOf(str) != -1);
}

/**
 * Returns true if the specified string is a lowercase alphanumeric char
 * (a-z or a digit, for example).
 * @param str the string to test.
 * @return True if it's a single lowercase alphanumeric char.
 */
function isLowerAlphaNumeric(str) {
    if (str.length > 1) {
        return false;
    }

    return str.match(/(?:[a-z]|[0-9])/);
}

/**
 * Returns true if the specified string is a uppercase alphanumeric char
 * (A-Z or a digit, for example).
 * @param str the string to test.
 * @return True if it's a single uppercase alphanumeric char.
 */
function isUpperAlphaNumeric(str) {
    if (str.length > 1) {
        return false;
    }

    return str.match(/(?:[A-Z]|[0-9])/);
}

/**
 * Returns true if the specified string is an alphanumeric char
 * (a-z, A-Z or a digit, for example).
 * @param str the string to test.
 * @return True if it's a single alphanumeric char.
 */
function isAlphaNumeric(str) {
    if (str.length > 1) {
        return false;
    }

    return str.match(/(?:[a-z]|[A-Z]|[0-9])/);
}

/**
 * Returns true if the specified string is a lowercase alphabetic char
 * (a-z, for example).
 * @param str the string to test.
 * @return True if it's a single lowercase alphabetic char.
 */
function isLowerAlpha(str) {
    if (str.length > 1) {
        return false;
    }

    return str.match(/[a-z]/);
}

/**
 * Returns true if the specified string is a uppercase alphabetic char
 * (A-Z, for example).
 * @param str the string to test.
 * @return True if it's a single uppercase alphabetic char.
 */
function isUpperAlpha(str) {
    if (str.length > 1) {
        return false;
    }

    return str.match(/[A-Z]/);
}

/**
 * Returns true if the specified string is an alphabetic char
 * (a-z or A-Z, for example).
 * @param str the string to test.
 * @return True if it's a single alphabetic char.
 */
function isAlpha(str) {
    if (str.length > 1) {
        return false;
    }

    return str.match(/(?:[a-z]|[A-Z])/);
}

/**
 * Returns true if the code is an special key (ENTER, PGUP, etc).
 * @param code the code to use.
 * @return True if the code is a special key.
 */
function isSpecialKey(code) {
    return (code === 0);
}

/**
 * Strips leading zeroes to the specified number string, ignoring sign chars.
 * Leaves the leading zero before a decimal separator.
 * @param str the string.
 * @param decimal decimal separator.
 */
function stripLeadingZeros(str, decimal) {
    var stripping = true;
    var result = "";
    for(var i = 0; i < str.length; i++) {
        var ch = str.substr(i, 1);

        if(isDigit(ch)) {
            if(ch != '0') {
                stripping = false;
            }

            if(!stripping || ch != '0' ||
               (i < (str.length - 1) && str.substr(i + 1, 1) == decimal)) {
                result = result + ch;
            }
        }
        else {
            result = result + ch;
        }
    }

    return result;
}

/**
 * Parses a floating numbers using the specified localized flags.
 * @param str the string to parse.
 * @param flags flags that specify separators and other options.
 * @return The parsed number.
 */
function parseLocalizedFloat(str, flags) {
    // Split the string.
    var len = str.length;
    var decimalPos = str.indexOf(flags.decimal);
    if(decimalPos == -1) {
        decimalPos = len;
    }

    var intPart = str.substr(0, decimalPos);
    var decimalPart = "0." +
        (decimalPos < len ?
         str.substr(decimalPos + 1, len - decimalPos) : "");

    var intNumber = parseLocalizedInteger(intPart, flags);
    if(isNaN(intNumber)) {
        return NaN;
    }

    var sign = (intNumber < 0 ? -1 : 1);
    var decimalNumber = parseFloat(decimalPart);
    if(isNaN(decimalNumber)) {
        return NaN;
    }

    return sign * (Math.abs(intNumber) + decimalNumber);
}

/**
 * Parses an integer using localized thousands separators.
 * @param str the string to parse.
 * @param flags flags that specify separators and other options.
 * @return The parsed number, or NaN if it is not a number.
 */
function parseLocalizedInteger(str, flags) {
    var separator = flags.separator;

    // Remove thousands separator.
    str = str.replace(
        new RegExp(escapeRegex(separator), "g"), "");
    return parseInt(str);
}

/**
 * Formats a floating number using localized thousand and decimal separator.
 * Other flags can be specified.
 * @param n the number to format.
 * @param flags the flags.
 * @return A string representing the number in localized form.
 */
function formatLocalizedFloat(n, flags) {
    var str =
        (isFinite(flags.roundedPlaces) ?
         n.toFixed(flags.roundedPlaces) :
         n.toString());
    return addThousandSeparators(str, flags);
}

/**
 * Formats an intenger using a localized thousands separator.
 * Other flags can be specified.
 * @param n the number to format.
 * @param flags the flags.
 * @return A string representing the integer in localized form.
 */
function formatLocalizedInteger(n, flags) {
    var str = n.toFixed(0);
    return addThousandSeparators(str, flags);
}

/**
 * Parses a localized date using the specified flags.
 * @param str the string to parse.
 * @param flags specifies formats, optional separators, etc.
 * @return The parsed date.
 */
function parseLocalizedDate(str, flags) {
    var regexpStr = dojo.regexp.posixDate(flags);
    var charsStr = dojo.regexp.posixDateChar();
    //dojo.debug("regexp: " + regexpStr);
    //dojo.debug("charsStr: " + charsStr);
    //dojo.debug("format: " + flags.format);

    var regex = new RegExp(regexpStr);
    var charRegex = new RegExp(charsStr, "g");

    var matches = str.match(regex);
    var charMatches = flags.format.match(charRegex);
    if(matches == null || charMatches == null ||
        matches.length != charMatches.length + 1) {
        dojo.debug("No match: " + str);
        return null;
    }

    var date = new Date(0);
    for(var i = 0; i < charMatches.length; i++) {
        var match = matches[i + 1];
        var charMatch = charMatches[i];

        // Strip leading zeroes.
        match = match.replace(/^0+/, "");
        //dojo.debug("[" + charMatch + "] -> " + match);

        if(charMatch == "%a") {
            dojo.unimplemented("parseLocalizedDate('%a')");
        }
        else if(charMatch == "%A") {
            dojo.unimplemented("parseLocalizedDate('%A')");
        }
        else if(charMatch == "%b" || charMatch == "%h") {
            dojo.unimplemented("parseLocalizedDate('%b')");
        }
        else if(charMatch == "%B") {
            dojo.unimplemented("parseLocalizedDate('%B')");
        }
        else if(charMatch == "%C") {
            var century = parseInt(match);
            date.setYear(century * 100);
        }
        else if(charMatch == "%d" || charMatch == "%e") {
            var day = parseInt(match);
            date.setDate(day);
        }
        else if(charMatch == "%j") {
            var day = parseInt(match);
            dojo.date.setDayOfYear(date, day);
        }
        else if(charMatch == "%m") {
            var month = parseInt(match) - 1; // From 00-11.
            date.setMonth(month);
        }
        else if(charMatch == "%u") {
            dojo.unimplemented(parseLocalizedDate("%u"));
        }
        else if(charMatch == "%U") {
            var week = parseInt(match);
            dojo.date.setWeekOfYear(date, week, 0); // First day = Sunday.
        }
        else if(charMatch == "%V") {
            var week = parseInt(match);
            dojo.date.setWeekOfYear(date, week, 1); // First day = Monday.
        }
        else if(charMatch == "%w") {
            dojo.unimplemented("parseLocalizedDate('%w')");
        }
        else if(charMatch == "%W") {
            dojo.unimplemented("parseLocalizedDate('%W')");
        }
        else if(charMatch == "%y") {
            date.setYear(match); // let setYear parse match!
        }
        else if(charMatch == "%Y") {
            var year = parseInt(match);
            date.setFullYear(year);
        }
    }
    return date;
}

/**
 * Adds thousands separators to a number, using the specified flags.
 * @param str the string.
 * @param flags the flags.
 */
function addThousandSeparators(str, flags) {
    if(flags.separator === "") {
        return str;
    }

    // If a "." is found, it's the decimal separator, unlocalized.
    if(flags.decimal != ".") {
        str = str.replace(".", flags.decimal);
    }

    var decimalStr = "";
    var decimalPos = str.indexOf(flags.decimal);
    if(decimalPos == -1) {
        decimalPos = str.length;
    }
    else {
        decimalStr = str.substr(decimalPos + 1, str.length - decimalPos);
    }

    var result = "";
    var k = 0;
    for(var i = decimalPos - 1; i >= 0; i--) {
        var ch = str.substr(i, 1);
        if(k == 3 && isDigit(ch)) {
            result = flags.separator + result;
            k = 0;
        }
        result = ch + result;
        k++;
    }

    return result +
      (decimalStr.length > 0 ? flags.decimal + decimalStr : "");
}

/**
 * Escapes a regular expression (really!).
 * @param str the regular expression to escape.
 * @return The escaped string.
 */
function escapeRegex(str) {
    return (str == null ? null : str.replace(/([-.*+?^${}()|[\]\/\\])/g, '\\$1'));
}

/**
 * Replaces all ocurrences of the specified string in str.
 * @param str the string to replace in.
 * @param from the string to search (not a regular expression!).
 * @param to the string to replace to (not a regular expression!).
 * @return The string with the replacements.
 */
function replaceAll(str, from, to) {
    return str.replace(new RegExp(escapeRegex(from), "g"), to);
}

/**
 * Gets the selected text start position for a text field
 * (it works in IE and Mozilla). Assumes the field is focused.
 * @param field the field to test.
 * @return The selection start for the field.
 */
function getSelectionStart(field) {
    var result = field.value.length;

    if (typeof field.selectionStart != "undefined") {
        result = field.selectionStart;
    }
    else if(document.selection) {
        result = Math.abs(
          document.selection.createRange().moveStart("character", -1000000));
    }

    return result;
}

/**
 * Gets the selected text end position for a text field
 * (it works in IE and Mozilla). Assumes the field is focused.
 * @param field the field to test.
 * @return The selection end for the field.
 */
function getSelectionEnd(field) {
    var result = field.value.length;

    if (typeof field.selectionEnd != "undefined") {
        result = field.selectionEnd;
    }
    else if(document.selection) {
        result = Math.abs(
          document.selection.createRange().moveEnd("character", -1000000));
    }

    return result;
}

/**
 * Sets the selection range for the specified field.
 * @param the field.
 * @param selectionStart the selection start pos.
 * @param selectionEnd the selection end pos.
 */
function setSelectionRange(field, selectionStart, selectionEnd) {
  if (field.setSelectionRange) {
    field.focus();
    field.setSelectionRange(selectionStart, selectionEnd);
  }
  else if (field.createTextRange) {
    var range = field.createTextRange();
    range.collapse(true);
    range.moveEnd('character', selectionEnd);
    range.moveStart('character', selectionStart);
    range.select();
  }
}

/**
 * Sets the caret position in the specified field to the last char.
 * @param field the text field.
 */
function setCaretToEnd(field) {
  setSelectionRange(field, field.value.length, field.value.length);
}

/**
 * Sets the caret position in the specified field to the beginning.
 * @param field the text field.
 */
function setCaretToBegin(field) {
  setSelectionRange(field, 0, 0);
}

/**
 * Sets the caret position in the specified field to the specified pos.
 * @param field the text field.
 * @param pos the position.
 */
function setCaretToPos(field, pos) {
  setSelectionRange(field, pos, pos);
}

/**
  Builds a regular expression to match any POSIX format for date.
  The RE can match one format or one of multiple formats.
  See <http://www.opengroup.org/onlinepubs/007908799/xsh/strftime.html>

  Format
  %a        locale's abbreviated weekday name.
  %A        locale's full weekday name.
  %b        locale's abbreviated month name.
  %B        locale's full month name.
  %C        century number (the year divided by 100 and truncated to
            an integer) as a decimal number [00-99].
  %d        day of the month as a decimal number [01,31].
  %e        day of the month as a decimal number [1,31]; a single digit is
            preceded by a space.
  %h        same as %b.
  %j        day of the year as a decimal number [001,366].
  %m        month as a decimal number [01,12].
  %t        tab character.
  %u        the weekday as a decimal number [1,7], with 1 representing Monday.
  %U        week number of the year (Sunday as the first day of the week) as a
            decimal number [00,53].
  %V        week number of the year (Monday as the first day of the week) as a
            decimal number [01,53]. If the week containing 1 January has four
            or more days in the new year, then it is considered week 1.
            Otherwise, it is the last week of the previous year, and the next
            week is week 1.
  %w        weekday as a decimal number [0,6], with 0 representing Sunday.
  %W        the week number of the year (Monday as the first day of the week)
            as a decimal number [00,53]. All days in a new year preceding the
            first Monday are considered to be in week 0.
  %y        year without century as a decimal number [00,99].
  %Y        year with century as a decimal number.
  All other characters must appear literally in the expression. Separators
  are specified by using the flags.separator value in the expression. Use
  the flags.ignoreSeparators flag to allow for optional separation.

  Example
    "%d/%m/%Y"  ->   30/01/2006

  @param flags  An object.
    flags.format  A string or an array of strings.  Default is "m/d/Y".
    flags.separator Date separator. Default is "/".
    flags.ignoreSeparators True if separators can be ignored.

  @return  A string for a regular expression for a date value.
*/
dojo.regexp.posixDate = function(flags) {
  // Assign default values to missing parameters.
  flags = (typeof flags == "object") ? flags : {};
  if (typeof flags.format == "undefined") { flags.format = "m/d/Y"; }
  if (typeof flags.separator != "string") { flags.separator = "/"; }

  // Converts a date format to a RE
  var dateRE = function(format) {
    // escape all special characters
    format = escapeRegex(format);
    var separator = escapeRegex(flags.separator);
    format = replaceAll(format, separator,
        "(?:" + separator + ")" + (flags.ignoreSeparators ? "?" : ""));

    // replace tokens with regular expressions
    format = replaceAll(format, "%a",
        dojo.regexp.buildGroupRE(dojo.date.shortDays, escapeRegex));
    format = replaceAll(format, "%A",
        dojo.regexp.buildGroupRE(dojo.date.days, escapeRegex));
    format = replaceAll(format, "%b",
        dojo.regexp.buildGroupRE(dojo.date.shortMonths, escapeRegex));
    format = replaceAll(format, "%B",
        dojo.regexp.buildGroupRE(dojo.date.months, escapeRegex));
    format = replaceAll(format, "%C",
        "([0-9]{2})");
    format = replaceAll(format, "%d",
        "((?:0[1-9])|(?:[1-2][0-9])|(?:3[0-1]))");
    format = replaceAll(format, "%e",
        "((?:(?:\\s[1-9])|(?:[1-2][0-9]))|(?:3[0-1]))");
    format = replaceAll(format, "%h",
        dojo.regexp.buildGroupRE(dojo.date.shortMonths, escapeRegex));
    format = replaceAll(format, "%j",
        "((?:00[1-9]))|(?:0[1-9][0-9])|(?:[1-2][0-9]{2})|(?:3(?:(?:[0-5][0-9])|6[0-6])))");
    format = replaceAll(format, "%m",
        "(0[1-9]|1[0-2])");
    format = replaceAll(format, "%t",
        "(\\\\t)");
    format = replaceAll(format, "%u",
        "([1-7])");
    format = replaceAll(format, "%U",
        "((?:[0-4][0-9])|(?:5[0-3]))");
    format = replaceAll(format, "%V",
        "((?:0[1-9])|(?:[1-4][0-9])|(?:5[0-3]))");
    format = replaceAll(format, "%w",
        "([0-6])");
    format = replaceAll(format, "%W",
        "((?:[0-4][0-9])|(?:5[0-3]))");
    format = replaceAll(format, "%y", "([0-9]{2})");
    format = replaceAll(format, "%Y", "([0-9]{4})");
    format = replaceAll(format, "%%", "%");

    return format;
  };

  // build RE for multiple date formats
  return dojo.regexp.buildGroupRE(flags.format, dateRE);
}

/**
  Builds a regular expression to match any POSIX format character.
  See <http://www.opengroup.org/onlinepubs/007908799/xsh/strftime.html>
  and the dojo.regexp.posixDate function.

  Example:
  - "%d/%m/%Y"  ->  "{'%d','%m','%Y'}".

  @return  A string for a regular expression for a date format char.
*/
dojo.regexp.posixDateChar = function() {

  var dateChars =
      ["%a","%A","%b","%C","%d","%e","%h","%j","%m",
       "%t","%u","%U","%V","%w","%W","%y","%Y"];

  // build RE for date formatting chars.
  return dojo.regexp.buildGroupRE(dateChars, escapeRegex);

}
