/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2002-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.graph.build;

import org.geotools.graph.structure.Graph;
import org.geotools.graph.structure.Graphable;

/**
 * Contructs a graph based on relationships between the entities (objects) 
 * modelled by the graph. <BR>
 * <BR>
 * The underlying graph is generated by continually adding objects to the 
 * generator. The Generator determines the relationships between the objects
 * and decides how to model the relationship and the objects themselves in the
 * graph.<BR>
 * <BR>
 * The GraphGenerator is the upper level of the graph construction process.
 * It is a wrapper around the GraphBuilder class that is intended to 
 * instruct the builder how to build the underyling graph structure.
 * 
 * @see GraphBuilder
 * 
 * @author Justin Deoliveira, Refractions Research Inc, jdeolive@refractions.net
 *
 * @source $URL$
 */
public interface GraphGenerator {
 
  /**
   * Adds an object to the graph.
   * 
   * @param obj The object to be modelled in the graph.
   * 
   * @return The graph component used to model the object.
   */
  public Graphable add(Object obj);
 
  /**
   * Retrieves a component of the graph.
   * 
   * @param obj The object modelled by the component.
   *
   * @return The graph component used to model the object.
   */ 
  public Graphable get(Object obj);
  
  /**
   * Removes an object from the graph.
   * 
   * @param obj The object modelled by the component.
   *
   * @return The graph component used to model the object.
   */
  public Graphable remove(Object obj);
  
  /**
   * Sets the underlying builder used to physically construct the graph.
   * 
   * @param builder The new underlying GraphBuilder.
   */
  public void setGraphBuilder(GraphBuilder builder);
  
  /**
   * Returns the underlying builder.
   * 
   * @return The underyling builder.
   */
  public GraphBuilder getGraphBuilder();
  
  /**
   * Returns the graph being generated. 
   * 
   * @return The generated graph.
   */
  public Graph getGraph();
  
//  /**
//   * Signals the generator that graph construction is about to begin.
//   *
//   */
//  public void init();
//  
//  /**
//   * Signals that generator that construction of the graph is complete.
//   */
//  public void finish();
}
