package org.geotools.renderer.style.shape;

import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

/**
 * This class is a decorator on top of a {@link Shape}
 * it extends the shape interface including the method setBounds making it possible to setup 
 * a bounding box not associated to the current shape path.
 * @author fmoura
 *
 */
public class ExplicitBoundsShape implements Shape {

	private Shape _shape;
	private Rectangle2D _bounds = null;
	
	/**
	 * The Constructor
	 * @param shape The actual shape on top of witch this decorator will stand.
	 */
	public ExplicitBoundsShape(Shape shape)
	{
		if(shape == null)
			throw new IllegalArgumentException("Shape can't be null.");
		_shape = shape;
	}
	
	/**
	 * Set the user defined bounds for this shape.
	 * @param bounds
	 */
	public void setBounds(Rectangle2D bounds)
	{
		_bounds = bounds;
	}

	
	public boolean contains(double x, double y, double w, double h) {
		return _shape.contains(x, y, w, h);
	}

	
	public boolean contains(double x, double y) {
		return _shape.contains(x, y);
	}

	
	public boolean contains(Point2D p) {
		return _shape.contains((Point2D)p);
	}

	
	public boolean contains(Rectangle2D r) {
		return _shape.contains((Rectangle2D)r);
	}

	/**
	 * Return the user defined bounds for this shape. If no user defined bounds where set
	 * it delegates the call to the actual shape.
	 * @return the Rectangle representing the Shape bounding box.
	 * @see Shape
	 */
	
	public Rectangle getBounds() {
		if(_bounds != null)
			return new Rectangle((int)_bounds.getMinX(),(int)_bounds.getMinY(),(int)_bounds.getWidth(),(int)_bounds.getHeight());
		return _shape.getBounds();
	}

	/**
	 * Return the user defined bounds for this shape. If no user defined bounds where set
	 * it delegates the call to the actual shape.
	 * @return the Rectangle representing the Shape bounding box.
	 * @see Shape
	 */
	public Rectangle2D getBounds2D() {
		if(_bounds != null)
			return _bounds;
		return _shape.getBounds2D();
	}

	
	public PathIterator getPathIterator(AffineTransform at, double flatness) {
		return _shape.getPathIterator(at, flatness);
	}

	
	public PathIterator getPathIterator(AffineTransform at) {
		return _shape.getPathIterator(at);
	}

	
	public boolean intersects(double x, double y, double w, double h) {
		return _shape.intersects(x, y, w, h);
	}

	
	public boolean intersects(Rectangle2D r) {
		return _shape.intersects(r);
	}
	
	@Override
	public boolean equals(Object obj)
	{
		if(obj instanceof ExplicitBoundsShape)
		{
			ExplicitBoundsShape other = (ExplicitBoundsShape) obj;
			boolean result = _shape.equals(other._shape);
			if(_bounds == null) return result & (other._bounds == null);
			return result & _bounds.equals(other._bounds);
		}
		else if(obj instanceof Shape)
		{
			if(_bounds == null) return _shape.equals(obj);
			return false;
		}
		return super.equals(obj);
		
	}
	
	@Override
	public int hashCode() {
		int hascode = _shape.hashCode();
		if(_bounds != null)
			hascode += hascode * 37 + _bounds.hashCode();
		return hascode;
	} 
	
	

}
