/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 * 
 * file: TerralibService.java
 * created: 22/12/2008
 */
package org.geotools.data.terralib;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.geotools.data.terralib.exception.IllegalStateException;
import org.geotools.data.terralib.exception.InvalidCrsWktException;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.exception.TerralibProviderRuntimeException;
import org.geotools.data.terralib.exception.TypeNotFoundException;
import org.geotools.data.terralib.persistence.exception.InvalidAttributeException;
import org.geotools.data.terralib.swig.DBConnection;
import org.geotools.data.terralib.swig.StringVector;
import org.geotools.data.terralib.swig.TeAttrDataType;
import org.geotools.data.terralib.swig.TeGeomRep;
import org.geotools.data.terralib.swig.TerralibAttributeDescriptor;
import org.geotools.data.terralib.swig.TerralibAttributeDescriptorVector;
import org.geotools.data.terralib.util.TerralibAttributeDescriptorTranslator;
import org.geotools.data.terralib.util.TypeAttributeMap;
import org.geotools.feature.AttributeTypeBuilder;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.CRS;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.GeometryDescriptor;
import org.opengis.geometry.MismatchedDimensionException;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

/**
 * This class is intended be the access point to Terralib native services.
 * It extends the automatically generated {@link TerralibServiceBase} class adapting
 * the returned data types to ease {@link TerralibDataStore} implementation.
 * @author fmoura, alexmc, fabiomano
 * @since TDK 3.0.0
 */
public class DefaultTerralibService extends ThreadBlockingTerralibService implements TerralibService
{
	
    //private static final Logger _logger = Logger.getLogger(DefaultTerralibService.class);

    /**
     * The constructor
     * @param connection Is the @{link DBconnection} that specifies the database on witch this service will execute.
     * @param enableLocking true if Terralib C++ services should be locked when in use to avoid multiple threads accessing the connection at the same time.
     * @since TDK 3.0.0
     */
    public DefaultTerralibService(DBConnection connection, boolean enableLocking)
    {
        super(connection,enableLocking);
    }
    
    /**
     * The constructor. The lock on Terralib C++ services is disabled. Multiple threads can access the database connection at the same time.
     * @param connection Is the @{link DBconnection} that specifies the database on witch this service will execute.
     * 
     * @since TDK 3.0.0
     */
    public DefaultTerralibService(DBConnection connection)
    {
        this(connection,false);
    }
    
    
    /* (non-Javadoc)
     * @see  org.tecgraf.tdk.data.terralib.ITerralibService#getTypeCRS(java.lang.String)
     */
    @Override    
    public CoordinateReferenceSystem getTypeCRS(String typeName) throws IllegalStateException,TypeNotFoundException,IOException
    {
        String wktString = null;
    	wktString = getTypeProjectionWKT(typeName);
       
        
        wktString = wktString.replaceAll("[eE]\\+", "E");
        CoordinateReferenceSystem crs = null;
        try
        {
            crs = CRS.parseWKT(wktString);
        }
        catch (FactoryException e)
        {
            throw new TerralibProviderRuntimeException("Could not parse the terralib generated wkt.",e);
        } 
        return crs;

    }
    
    /*
     * (non-Javadoc)
     * @see org.geotools.data.terralib.TerralibService#getViewCRS(java.lang.String)
     */
    @Override
    public CoordinateReferenceSystem getViewCRS(String viewID) throws TypeNotFoundException, IOException 
    {
        String wktString = null;
        wktString = getViewProjectionWKT(viewID);
        wktString = wktString.replaceAll("[eE]\\+", "E");
        CoordinateReferenceSystem crs = null;
        try
        {
            crs = CRS.parseWKT(wktString);
        }
        catch (FactoryException e)
        {
            throw new TerralibProviderRuntimeException("Could not parse the terralib generated wkt.",e);
        } 
        return crs;
    }
    
    /**
     * Gets a set of {@link TeGeomRep} that identifies the geometry types present in the given 
     * feature type
     * @param typeName A valid {@link TerralibDataStore} type name that identifies the type for witch the geometry types will be retrieved.
     * @return a set of {@link TeGeomRep} that identifies the geometry types retrieved for this feature type.
     * @since TDK 3.0.0
     * @throws IllegalStateException if the database is in a invalid state ( commonly not connected ).
     * @throws IllegalArgumentException if the type name does not conform to {@link TerralibDataStore} type name convention.
     * @throws TypeNotFoundException if the given type is not found in the working database.
     * @throws IOException 
     */
    protected Set<TeGeomRep> getTypeGeometryTypes(String typeName) throws IllegalStateException,TypeNotFoundException, IOException
    {
        HashSet<TeGeomRep> geomRepSet = new HashSet<TeGeomRep>();
        
        int rawGeomRep = getGeomRep(typeName);
        
        for(TeGeomRep geomRep : TeGeomRep.values())
        {
            if((rawGeomRep & geomRep.swigValue()) == geomRep.swigValue())
            {
                geomRepSet.add(geomRep);
            }
        }
        
        return geomRepSet;
    }
    
    
    /* (non-Javadoc)
     * @see  org.tecgraf.tdk.data.terralib.ITerralibService#getTypeGeometryDescriptors(java.lang.String)
     */
    @Override  
    public Set<GeometryDescriptor> getTypeGeometryDescriptors(String typeName) throws IllegalStateException, TypeNotFoundException , IOException
    {
    	
        Set<TeGeomRep> geomTypeSet = getTypeGeometryTypes(typeName);
        CoordinateReferenceSystem crs = getTypeCRS(typeName);
        
        return TerralibAttributeDescriptorTranslator.buildGeometryDescriptorSet(geomTypeSet, crs);
    }
    
    /* (non-Javadoc)
     * @see  org.tecgraf.tdk.data.terralib.ITerralibService#getTypeAttributeDescriptors(java.lang.String)
     */
    @Override  
    public Map<String,List<AttributeDescriptor>> getTypeAttributeDescriptors(String typeName) throws IllegalStateException, TypeNotFoundException, IOException
    {
        Map<String,List<AttributeDescriptor>> typeDescriptorMap = new HashMap<String, List<AttributeDescriptor>>();
        TerralibAttributeDescriptorVector typeAttributes = getTypeAttributes(typeName);
        
        long typeAttributesSize = typeAttributes.size();
        
        AttributeTypeBuilder attributeTypeBuilder = new AttributeTypeBuilder();
        
        for(long i = 0; i < typeAttributesSize ; i++)
        {
            TerralibAttributeDescriptor attribute = typeAttributes.get(i);
            
            //if (attribute.getDataType() != TeAttrDataType.TeDATETIME) {
                String attributeName = attribute.getName();
                String attributeTableName = attributeName.substring(0, attributeName.indexOf('.'));
                AttributeDescriptor attributeDescriptor =  TerralibAttributeDescriptorTranslator.terralibDescriptorToGeotoolsDescriptor(attributeTypeBuilder, attribute); 
                if(!typeDescriptorMap.containsKey(attributeTableName))
                {
                    typeDescriptorMap.put(attributeTableName, new ArrayList<AttributeDescriptor>());
                }
                typeDescriptorMap.get(attributeTableName).add(attributeDescriptor);
            //}
        
        }
        return typeDescriptorMap;
    }
    
    /* (non-Javadoc)
     * @see  org.tecgraf.tdk.data.terralib.ITerralibService#getTypeBoundingBox(java.lang.String)
     */
    @Override  
    public ReferencedEnvelope getTypeBoundingBox(String typeName) throws MismatchedDimensionException, IllegalStateException, TypeNotFoundException, IOException
    {
        return new ReferencedEnvelope(getBoundingBox(typeName),getTypeCRS(typeName));
    }

    
	@Override
	public String[] getViewIDs() throws IOException {
		StringVector out = new StringVector();
		getViewIDs(out);
		int typeNamesSize = (int) out.size();

		String[] viewIDs = new String[typeNamesSize];
        for(int i = 0 ; i < typeNamesSize ; i ++)
        	viewIDs[i] = out.get(i);
		
		return viewIDs;
	}
	
	@Override
	public String[] getViewTypesNames(String viewID) throws IOException {
		StringVector out = new StringVector();
		getViewTypesNames(viewID, out);
		
		int typeNamesSize = (int) out.size();

		String[] typeNames = new String[typeNamesSize];
        for(int i = 0 ; i < typeNamesSize ; i ++)
        	typeNames[i] = out.get(i);
		
		return typeNames;
	}

	@Override
	public String[] getTypeNames() throws IllegalStateException, IOException 
	{
		StringVector out = new StringVector();
		getTypesNames(out);
		int typeNamesSize = (int) out.size();

		String[] typeNames = new String[typeNamesSize];
        for(int i = 0 ; i < typeNamesSize ; i ++)
        	typeNames[i] = out.get(i);
		
		return typeNames;		
	}

	/*
	 * (non-Javadoc)
	 * @see org.geotools.data.terralib.TerralibService#createType(java.lang.String, org.opengis.feature.type.GeometryDescriptor, org.opengis.referencing.crs.CoordinateReferenceSystem)
	 */
	@Override
	public void createType(String typeName, GeometryDescriptor geometryDesc,
			CoordinateReferenceSystem projection) throws InvalidCrsWktException, IOException
	{
		if (geometryDesc == null)
			throw new NullArgumentException("geometryDesc");
		
		if (typeName == null)
			throw new NullArgumentException("typeName");

		if (projection == null)
			throw new NullArgumentException("projection");

		TypeAttributeMap type = TypeAttributeMap.fromBindingClass(geometryDesc.getType().getBinding());
		if (type == null)
			throw new IllegalArgumentException("Unsupported geometry at GeometryDescriptor: " + geometryDesc.getType().getBinding().getName());
		
		CoordinateReferenceSystem completeCrs = null;
		try {
			completeCrs = CRS.decode("EPSG:" + CRS.lookupEpsgCode(projection, true));
		} catch (Exception e) {
			throw new InvalidCrsWktException("Error trying to get more information about the given coordinate reference system.",e);
		}
		
		createType(typeName, type.getGeomRep(), completeCrs.toWKT());
	}
    
	/*
	 * \(non-Javadoc)
	 * @see org.geotools.data.terralib.swig.TerralibServiceNative#buildTerralibAttributeDescriptor(java.lang.String, boolean, boolean, int, org.geotools.data.terralib.swig.TeAttrDataType)
	 */
	@Override
	public TerralibAttributeDescriptor buildTerralibAttributeDescriptor(
			String name, boolean isNullable, boolean isPrimaryKey, int length,
			TeAttrDataType type) 
	{
		
		if (name == null)
			throw new NullArgumentException("name");

		if (type == null)
			throw new NullArgumentException("type");
		
		TypeAttributeMap typeMap = TypeAttributeMap.fromTeAttrDataType(type);
		if (typeMap == null)
			throw new InvalidAttributeException("Unrecognized attribute type " + type);
		
		if (isPrimaryKey && isNullable)
			throw new InvalidAttributeException("Primary keys can't be nullable");
		
		if (typeMap.isGeometry() && (length != DefaultTerralibService.getANY_LENGHT() ))
		{
			throw new InvalidAttributeException("You can not specify an attribute length for geometry types.");
		}
		
		
		return super.buildTerralibAttributeDescriptor(name, isNullable, isPrimaryKey,
				length, type);
	}

	/*
	 * (non-Javadoc)
	 * @see org.geotools.data.terralib.TerralibService#createAttributeTable(java.lang.String, java.util.List)
	 */
	@Override
	public void createAttributeTable(String tableName,
			List<AttributeDescriptor> attributes) throws IllegalStateException, IOException 
	{
		if (tableName == null)
			throw new NullArgumentException("tableName");
		if (attributes == null)
			throw new NullArgumentException("attributes");
		
		
//		if (nameIsInvalid(tableName))
//			throw new InvalidAttributeTableNameException("The given table name is invalid (" + tableName + ")");
		
		TerralibAttributeDescriptorVector teAttributes = new TerralibAttributeDescriptorVector();
		for (AttributeDescriptor attrDesc: attributes)
		{
//			if (nameIsInvalid(attrDesc.getLocalName()))
//				throw new InvalidAttributeException("The attribute name (" + attrDesc.getLocalName() + ") is invalid");
			
			TerralibAttributeDescriptor teDescriptor = TerralibAttributeDescriptorTranslator.geotoolsDescriptorToTerralibDescriptor(this, attrDesc);
			teAttributes.add(teDescriptor);
		}
		 
		createAttributeTable(tableName, teAttributes);
	}

	/*
	 * (non-Javadoc)
	 * @see org.geotools.data.terralib.swig.TerralibServiceNative#linkAttributeTable(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void linkAttributeTable(String typeName, String tableName, String linkAttributeColumn) throws IOException, TypeNotFoundException, IllegalStateException
	{
		if (tableName == null)
			throw new NullArgumentException("tableName");
		if (typeName == null)
			throw new NullArgumentException("typeName");
		if (linkAttributeColumn == null)
			throw new NullArgumentException("linkAttributeColumn");		
		
//		if (nameIsInvalid(typeName))
//			throw new IllegalArgumentException("The given typeName name is invalid (" + typeName + ")");
//
//		if (nameIsInvalid(tableName))
//			throw new InvalidAttributeTableNameException("The given table name is invalid (" + tableName + ")");
//
//		if (nameIsInvalid(linkAttributeColumn))
//			throw new IllegalArgumentException("The given link attribute column name is invalid (" + linkAttributeColumn + ")");

		super.linkAttributeTable(typeName,tableName,linkAttributeColumn);
	}

	
	public TerralibLayerDataType getLayerType(String typeName) throws IllegalStateException, TypeNotFoundException, IOException 
	{
		Set<TeGeomRep> gemetryTypeSet = getTypeGeometryTypes(typeName);
		if(/*gemetryTypeSet.contains(TeGeomRep.TeRASTER) ||*/ gemetryTypeSet.contains(TeGeomRep.TeRASTERFILE) /*|| gemetryTypeSet.contains(TeGeomRep.TeCOVERAGE)*/)
		{
			return TerralibLayerDataType.COVERAGE;
		}
		if(gemetryTypeSet.contains(TeGeomRep.TeLINES) || gemetryTypeSet.contains(TeGeomRep.TePOINTS) || gemetryTypeSet.contains(TeGeomRep.TePOLYGONS))
		{
			return TerralibLayerDataType.VECTOR;
		}		
		return TerralibLayerDataType.NONE;
		
	}

	public List<URL> getRasterFileURL(String typeName) throws TypeNotFoundException, IllegalStateException, IOException 
	{
		StringVector filePaths = getRasterFilePath(typeName);
		List<URL> filesURLs = new ArrayList<URL>(); 
		for(int i = 0 ; i < filePaths.size() ; i++)
		{
			String filePath = filePaths.get(i);
			filesURLs.add(new URL("file:///"+filePath));
		}		
		return filesURLs;
	}


//    /**
//     * @param attributeTypeBuilder
//     * @param attribute
//     * @return
//     * @since
//     */
//    private AttributeDescriptor buildAttributeDescriptor(
//            AttributeTypeBuilder attributeTypeBuilder, TypeAttribute attribute)
//    {
//        String attributeName = attribute.getName();
//        
//        
//        Class bindingClass;
//        switch (attribute.getDataType())
//        {
//        case TeREAL:
//            bindingClass = Double.class;
//            break;
//            
//        case TeINT:
//            bindingClass = Integer.class;
//            break;
//            
//        case TeSTRING:
//            bindingClass = String.class;
//            break;
//            
//        case TeDATETIME:
//            bindingClass = Date.class;
//            break;
//            
//
//        default:
//            bindingClass = Object.class;
//            _logger.warn("Unsupported data type ("+attribute.getDataType().name()+")  for attribute \""+attribute.getName()+"\". Using Object class for binding.");
//            break;
//        }
//        attributeTypeBuilder.setBinding(bindingClass);
//        AttributeType attributeType = attributeTypeBuilder.buildType();
//        AttributeDescriptor attributeDescriptor = attributeTypeBuilder.buildDescriptor(attributeName, attributeType);
//        return attributeDescriptor;
//    }
}
